import { Injectable } from '@nestjs/common';
import { DataSource, EntityManager } from 'typeorm';
import {
  CreateUserRegistrationMapDto,
  DeleteUserRegistrationMapDto,
  UpdateUserRegistrationMapDto,
} from './dto';
import { UserRegistrationMapRepository } from './user-registration-map.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { userRegistrationMapConstMessages } from 'src/common/constants/strings-constants';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import {
  User,
  ProgramRegistration,
  UserRegistrationMap,
  UserParticipationSummary,
} from 'src/common/entities';
import { OriginTypeEnum } from 'src/common/enum/origin-type.enum';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';

@Injectable()
export class UserRegistrationMapService {
  constructor(
    private readonly repository: UserRegistrationMapRepository,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
  ) {}

  async create(userId: number, dto: CreateUserRegistrationMapDto): Promise<UserRegistrationMap[]> {
    this.logger.log(userRegistrationMapConstMessages.CREATE_REQUEST_RECEIVED(userId), dto);

    try {
      return await this.dataSource.transaction(async (manager) => {

        const { user, createdByUser, updatedByUser } = await this.resolveUsers(
          userId,
          dto.createdBy,
          dto.updatedBy,
          manager,
        );

        const registrationMap = await this.ensureRegistrations(
          userId,
          dto.registrationIds,
          manager,
        );

        const mappings: UserRegistrationMap[] = [];
        for (const registrationId of this.unique(dto.registrationIds)) {
          const registration = registrationMap.get(registrationId) as ProgramRegistration;
          const summary = await this.ensureSummary(
            user,
            registration,
            createdByUser,
            updatedByUser,
            manager,
          );

          const mapping = await this.createOrUpdateMapping(
            user,
            registration,
            summary,
            createdByUser,
            updatedByUser,
            manager,
          );
          mappings.push(mapping);
        }

        return mappings;
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_SAVE_FAILED, error);
    }
  }

  /**
   * Updates user registration mappings by ignoring existing mappings 
   * and creating new ones only for registration IDs that don't already have mappings.
   * @param {number} userId - The ID of the user whose mappings are being updated
   * @param {UpdateUserRegistrationMapDto} dto - The update data containing new registration IDs
   * @returns {Promise<UserRegistrationMap[]>} A promise resolving to an array of updated user registration mappings
   */
  async update(userId: number, dto: UpdateUserRegistrationMapDto): Promise<UserRegistrationMap[]> {
    this.logger.log(userRegistrationMapConstMessages.UPDATE_REQUEST_RECEIVED(userId), dto);

    try {
      return await this.dataSource.transaction(async (manager) => {
        this.assertUserIdMatch(userId, dto.userId);

        const { user, updatedByUser } = await this.resolveUsers(
          userId,
          undefined,
          dto.updatedBy,
          manager,
        );

        const uniqueRegistrationIds = this.unique(dto.registrationIds);
        
        // Get existing mappings for the user
        const existingMappings = await this.repository.findMappingsForRegistrations(
          userId,
          uniqueRegistrationIds,
          false,
          manager,
        );

        // Get existing registration IDs
        const existingRegistrationIds = new Set(
          existingMappings.map((mapping) => mapping.registration.id)
        );

        // Filter out registration IDs that already have mappings
        const newRegistrationIds = uniqueRegistrationIds.filter(
          (id) => !existingRegistrationIds.has(id)
        );

        const allMappings: UserRegistrationMap[] = [...existingMappings];

        // Only create mappings for new registration IDs
        if (newRegistrationIds.length > 0) {
          const registrationMap = await this.ensureRegistrations(
            userId,
            newRegistrationIds,
            manager,
          );

          for (const registrationId of newRegistrationIds) {
            const registration = registrationMap.get(registrationId) as ProgramRegistration;
            const summary = await this.ensureSummary(
              user,
              registration,
              updatedByUser,
              updatedByUser,
              manager,
            );

            const mapping = await this.createOrUpdateMapping(
              user,
              registration,
              summary,
              null,
              updatedByUser,
              manager,
            );
            allMappings.push(mapping);
          }
        }

        return allMappings;
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_UPDATE_FAILED, error);
    }
  }

  /**
   * Deletes user registration mappings and their associated summaries by setting deletedAt timestamps.
   * @param {number} userId - The ID of the user whose mappings are being deleted
   * @param {DeleteUserRegistrationMapDto} dto - The delete data containing registration IDs to remove
   * @returns {Promise<void>} A promise that resolves when the deletion is complete
   */
  async delete(userId: number, dto: DeleteUserRegistrationMapDto): Promise<void> {
    this.logger.log(userRegistrationMapConstMessages.DELETE_REQUEST_RECEIVED(userId), dto);

    try {
      await this.dataSource.transaction(async (manager) => {
        this.assertUserIdMatch(userId, dto.userId);

        const { updatedByUser } = await this.resolveUsers(
          userId,
          undefined,
          dto.updatedBy,
          manager,
        );

        const uniqueRegistrationIds = this.unique(dto.registrationIds);
        const existingMappings = await this.repository.findMappingsForRegistrations(
          userId,
          uniqueRegistrationIds,
          false,
          manager,
        );

        const foundIds = new Set(existingMappings.map((mapping) => Number(mapping.registration.id)));
        const missing = uniqueRegistrationIds.filter((id) => !foundIds.has(id));
        if (missing.length > 0) {
          throw new InifniNotFoundException(
            ERROR_CODES.USER_REGISTRATION_MAP_NOT_FOUND,
            null,
            null,
            missing.join(', '),
          );
        }

        // Update mappings and summaries with deletedAt timestamps
        for (const mapping of existingMappings) {
          mapping.deletedAt = new Date();
          
          // Update updatedBy field if updatedByUser is provided
          if (updatedByUser) {
            mapping.updatedBy = updatedByUser;
          }
          
          // Also soft delete the associated summary
          if (mapping.summary) {
            mapping.summary.deletedAt = new Date();
            
            // Update summary's updatedBy field if updatedByUser is provided
            if (updatedByUser) {
              mapping.summary.updatedBy = updatedByUser.id;
            }
            
            await this.repository.saveSummary(mapping.summary, manager);
          }
        }
        
        await this.repository.saveMappings(existingMappings, manager);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_DELETE_FAILED, error);
    }
  }

  /**
   * Retrieves all active user registration mappings for a specific user.
   * @param {number} userId - The ID of the user whose mappings are being retrieved
   * @returns {Promise<UserRegistrationMap[]>} A promise resolving to an array of user registration mappings
   */
  async findByUserId(userId: number): Promise<UserRegistrationMap[]> {
    this.logger.log(userRegistrationMapConstMessages.GET_REQUEST_RECEIVED(userId));

    try {
      // Validate that the user exists
      await this.ensureUser(userId, undefined, userRegistrationMapConstMessages.VALIDATING_USER(userId));

      // Get all active mappings for the user
      const mappings = await this.repository.findActiveMappingsForUser(userId);
      
      this.logger.log(`Retrieved ${mappings.length} active mappings for user ${userId}`);
      return mappings;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_GET_FAILED, error);
    }
  }

  private unique(ids: number[]): number[] {
    return [...new Set(ids)];
  }

  private assertUserIdMatch(pathUserId: number, bodyUserId: number): void {
    if (pathUserId !== bodyUserId) {
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniBadRequestException(
        ERROR_CODES.USER_REGISTRATION_MAP_USER_MISMATCH,
        null,
        null,
        bodyUserId.toString(),
        pathUserId.toString(),
      );
    }
  }

  private async resolveUsers(
    userId: number,
    createdById?: number,
    updatedById?: number,
    manager?: EntityManager,
  ): Promise<{
    user: User;
    createdByUser: User | null;
    updatedByUser: User | null;
  }> {
    const user = await this.ensureUser(userId, manager);
    let createdByUser: User | null = null;
    let updatedByUser: User | null = null;

    if (createdById) {
      createdByUser = await this.ensureUser(
        createdById,
        manager,
        userRegistrationMapConstMessages.VALIDATING_CREATOR(createdById),
      );
    }

    if (updatedById) {
      updatedByUser = await this.ensureUser(
        updatedById,
        manager,
        userRegistrationMapConstMessages.VALIDATING_UPDATER(updatedById),
      );
    } else {
      updatedByUser = createdByUser;
    }

    return { user, createdByUser, updatedByUser };
  }

  private async ensureUser(
    userId: number,
    manager?: EntityManager,
    logMessage?: string,
  ): Promise<User> {
    if (logMessage) {
      this.logger.log(logMessage);
    }
    const user = await this.repository.findUserById(userId, manager);
    if (!user) {
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
    }
    return user;
  }

  private async ensureRegistrations(
    userId: number,
    registrationIds: number[],
    manager?: EntityManager,
  ): Promise<Map<number, ProgramRegistration>> {
    const uniqueIds = this.unique(registrationIds);
    const registrations = await this.repository.findRegistrationsByIds(uniqueIds, manager);
    const map = new Map<number, ProgramRegistration>();
    registrations?.forEach((registration) => {
      if (registration) {
        const registrationId = Number(registration.id);
        map.set(registrationId, registration);
      }
    });

    const missing = uniqueIds.filter((id) =>{
      return !map.has(Number(id));
    });
    if (missing.length > 0) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        missing.join(', '),
      );
    }

    for (const registration of map.values()) {
      if (registration.registrationStatus !== RegistrationStatusEnum.COMPLETED) {
        this.logger.error(
          userRegistrationMapConstMessages.REGISTRATION_STATUS_INVALID(registration.id),
        );
        throw new InifniBadRequestException(
          ERROR_CODES.USER_REGISTRATION_MAP_REGISTRATION_NOT_COMPLETED,
          null,
          null,
          registration.id.toString(),
        );
      }
    }

    return map;
  }

  private async ensureSummary(
    user: User,
    registration: ProgramRegistration,
    createdByUser: User | null,
    updatedByUser: User | null,
    manager?: EntityManager,
  ): Promise<UserParticipationSummary> {
    const baseProgram = registration.program ?? registration.programSession?.program ?? null;
    if (!baseProgram) {
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniBadRequestException(
        ERROR_CODES.USER_REGISTRATION_MAP_SUMMARY_FAILED,
        null,
        null,
        registration.id.toString(),
      );
    }

    const allocatedProgram = registration.allocatedProgram ?? null;
    const session = registration.allocatedSession ?? registration.programSession ?? null;
    const summary = await this.repository.findExistingSummary(
      user.id,
      baseProgram.id,
      allocatedProgram?.id ?? null,
      session?.id ?? null,
      manager,
    );

    const programStartsAt =
      session?.startsAt ?? baseProgram.startsAt ?? baseProgram.registrationStartsAt ?? null;
    const programEndsAt =
      session?.endsAt ?? baseProgram.endsAt ?? baseProgram.registrationEndsAt ?? null;

    if (summary) {
      summary.userId = user.id;
      summary.user = user;
      summary.programName = baseProgram.name ?? summary.programName;
      summary.subProgramId = allocatedProgram?.id ?? 0; 
      summary.subProgramName = allocatedProgram?.name ?? '';
      summary.sessionId = session?.id ?? null;
      summary.sessionName = session?.name ?? null;
      summary.subProgramType =
      allocatedProgram?.subProgramType ?? baseProgram.subProgramType ?? undefined;
      summary.programStartsAt = programStartsAt ?? summary.programStartsAt;
      summary.programEndsAt = programEndsAt ?? summary.programEndsAt;
      summary.updatedBy = updatedByUser?.id ?? summary.updatedBy ?? createdByUser?.id ?? null;
      summary.originType = OriginTypeEnum.CODE_INSERTED;
      return this.repository.saveSummary(summary, manager);
    }

    const newSummary = this.repository.createSummary(
      {
        userId: user.id,
        user,
        programId: baseProgram.id,
        programName: baseProgram.name,
        subProgramId: allocatedProgram?.id ?? undefined,
        subProgramName: allocatedProgram?.name ?? undefined,
        sessionId: session?.id ?? null,
        sessionName: session?.name ?? null,
        subProgramType: allocatedProgram?.subProgramType ?? baseProgram.subProgramType ?? undefined,
        programStartsAt,
        programEndsAt,
        createdBy: createdByUser?.id ?? updatedByUser?.id ?? undefined,
        updatedBy: updatedByUser?.id ?? createdByUser?.id ?? undefined,
        originType: OriginTypeEnum.CODE_INSERTED,
      },
      manager,
    );

    const savedSummary = await this.repository.saveSummary(newSummary, manager);
    this.logger.log(userRegistrationMapConstMessages.SUMMARY_CREATED(savedSummary.id));
    return savedSummary;
  }

  private async createOrUpdateMapping(
    user: User,
    registration: ProgramRegistration,
    summary: UserParticipationSummary,
    createdByUser: User | null,
    updatedByUser: User | null,
    manager?: EntityManager,
  ): Promise<UserRegistrationMap> {
    let mapping = await this.repository.findMappingByUserAndRegistration(
      user.id,
      registration.id,
      true,
      manager,
    );

    if (mapping) {
      mapping.user = user;
      mapping.registration = registration;
      mapping.summary = summary;
      if (!mapping.createdBy && createdByUser) {
        mapping.createdBy = createdByUser;
      }
      mapping.updatedBy = updatedByUser ?? mapping.updatedBy ?? createdByUser ?? null;
      mapping.deletedAt = null;
      mapping.originType = OriginTypeEnum.CODE_INSERTED;
      mapping = await this.repository.saveMapping(mapping, manager);
      this.logger.log(userRegistrationMapConstMessages.MAPPING_UPDATED(mapping.id));
      return mapping;
    }

    mapping = this.repository.createMapping(
      {
        user,
        registration,
        summary,
        createdBy: createdByUser ?? updatedByUser ?? null,
        updatedBy: updatedByUser ?? createdByUser ?? null,
        originType: OriginTypeEnum.CODE_INSERTED,
      },
      manager,
    );

    const saved = await this.repository.saveMapping(mapping, manager);
    this.logger.log(userRegistrationMapConstMessages.MAPPING_CREATED(saved.id));
    return saved;
  }
}
