import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  ParseIntPipe,
  Post,
  Put,
  Res,
  UseGuards,
  ValidationPipe,
} from '@nestjs/common';
import {
  ApiBearerAuth,
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiResponse,
  ApiSecurity,
  ApiTags,
} from '@nestjs/swagger';
import { Response } from 'express';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import {
  SWAGGER_API_RESPONSE,
  userRegistrationMapConstMessages,
} from 'src/common/constants/strings-constants';
import {
  CreateUserRegistrationMapDto,
  UpdateUserRegistrationMapDto,
  DeleteUserRegistrationMapDto,
} from './dto';
import { UserRegistrationMapService } from './user-registration-map.service';
import { UserRegistrationMap } from 'src/common/entities';

@ApiTags('user-registration-map')
@Controller('user-registration-map')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('rm', 'relational_manager', 'shoba', 'rm_support')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class UserRegistrationMapController {
  constructor(
    private readonly service: UserRegistrationMapService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: userRegistrationMapConstMessages.CREATE_OPERATION })
  @ApiBody({ type: CreateUserRegistrationMapDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  async create(
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: CreateUserRegistrationMapDto,
    @Res() res: Response,
  ): Promise<void> {
    const userId = dto.userId;
    this.logger.log(userRegistrationMapConstMessages.CREATE_REQUEST_RECEIVED(userId), dto);
    try {
      const mappings = await this.service.create(userId, dto);
      await this.responseService.success(
        res,
        userRegistrationMapConstMessages.MAPPINGS_RECORDED,
        { data: mappings },
        HttpStatus.CREATED,
      );
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Put(':userId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: userRegistrationMapConstMessages.UPDATE_OPERATION })
  @ApiParam({ name: 'userId', type: Number, description: 'Seeker user identifier' })
  @ApiBody({ type: UpdateUserRegistrationMapDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  async update(
    @Param('userId', ParseIntPipe) userId: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: UpdateUserRegistrationMapDto,
    @Res() res: Response,
  ): Promise<void> {
    this.logger.log(userRegistrationMapConstMessages.UPDATE_REQUEST_RECEIVED(userId), dto);
    try {
      const mappings = await this.service.update(userId, dto);
      await this.responseService.success(res, userRegistrationMapConstMessages.MAPPINGS_UPDATED, {
        data: mappings,
      });
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Delete(':userId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: userRegistrationMapConstMessages.DELETE_OPERATION })
  @ApiParam({ name: 'userId', type: Number, description: 'Seeker user identifier' })
  @ApiBody({ type: DeleteUserRegistrationMapDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  async delete(
    @Param('userId', ParseIntPipe) userId: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: DeleteUserRegistrationMapDto,
    @Res() res: Response,
  ): Promise<void> {
    this.logger.log(userRegistrationMapConstMessages.DELETE_REQUEST_RECEIVED(userId), dto);
    try {
      await this.service.delete(userId, dto);
      await this.responseService.success(res, userRegistrationMapConstMessages.MAPPINGS_DELETED);
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Get(':userId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: userRegistrationMapConstMessages.GET_OPERATION })
  @ApiParam({ name: 'userId', type: Number, description: 'Seeker user identifier' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.SUCCESS })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  /**
   * Retrieves all active user registration mappings for a specific user.
   * @param {number} userId - The ID of the user whose mappings are being retrieved
   * @param {Response} res - The Express response object
   * @returns {Promise<void>} A promise that resolves when the response is sent
   */
  async findByUserId(
    @Param('userId', ParseIntPipe) userId: number,
    @Res() res: Response,
  ): Promise<void> {
    this.logger.log(userRegistrationMapConstMessages.GET_REQUEST_RECEIVED(userId));
    try {
      const mappings = await this.service.findByUserId(userId);
      await this.responseService.success(res, userRegistrationMapConstMessages.MAPPINGS_RETRIEVED, {
        data: mappings,
      });
    } catch (error) {
      await handleControllerError(res, error);
    }
  }
}
