import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { EntityManager, In, Repository } from 'typeorm';
import { LookupData, ProgramRegistration, User, UserProgramExperience } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { CommonStatus } from 'src/common/enum/common-status.enum';
import { userProgramExperienceConstMessages } from 'src/common/constants/strings-constants';

@Injectable()
export class UserProgramExperienceRepository {
  constructor(
    @InjectRepository(UserProgramExperience)
    private readonly experienceRepo: Repository<UserProgramExperience>,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepo: Repository<ProgramRegistration>,
    @InjectRepository(LookupData)
    private readonly lookupDataRepo: Repository<LookupData>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  private getExperienceRepository(manager?: EntityManager): Repository<UserProgramExperience> {
    return manager ? manager.getRepository(UserProgramExperience) : this.experienceRepo;
  }

  private getUserRepository(manager?: EntityManager): Repository<User> {
    return manager ? manager.getRepository(User) : this.userRepo;
  }

  private getRegistrationRepository(manager?: EntityManager): Repository<ProgramRegistration> {
    return manager ? manager.getRepository(ProgramRegistration) : this.registrationRepo;
  }

  private getLookupRepository(manager?: EntityManager): Repository<LookupData> {
    return manager ? manager.getRepository(LookupData) : this.lookupDataRepo;
  }

  async findUserById(userId: number, manager?: EntityManager): Promise<User | null> {
    this.logger.log(userProgramExperienceConstMessages.USER_VALIDATION(userId));
    return this.commonDataService.findOneById(this.getUserRepository(manager), userId, false);
  }

  async findRegistrationById(
    registrationId: number,
    manager?: EntityManager,
  ): Promise<ProgramRegistration | null> {
    this.logger.log(userProgramExperienceConstMessages.REGISTRATION_VALIDATION(registrationId));
    return this.commonDataService.findOneById(
      this.getRegistrationRepository(manager),
      registrationId,
    );
  }

  async findLookupDataByIds(
    lookupDataIds: number[],
    manager?: EntityManager,
  ): Promise<LookupData[]> {
    this.logger.log(userProgramExperienceConstMessages.VALIDATING_LOOKUP_DATA(lookupDataIds));
    try {
      return await this.getLookupRepository(manager).find({
        where: {
          id: In(lookupDataIds),
          lookupCategory: userProgramExperienceConstMessages.CATEGORY,
          lookupStatus: CommonStatus.ACTIVE,
        },
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PROGRAM_EXPERIENCE_LOOKUP_FAILED, error);
    }
  }

  createExperience(partial: Partial<UserProgramExperience>, manager?: EntityManager) {
    return this.getExperienceRepository(manager).create(partial);
  }

  async saveExperiences(
    experiences: UserProgramExperience[],
    manager?: EntityManager,
  ): Promise<UserProgramExperience[]> {
    try {
      return await this.getExperienceRepository(manager).save(experiences);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PROGRAM_EXPERIENCE_SAVE_FAILED, error);
    }
  }

  async findActiveExperiences(
    userId: number,
    registrationId: number | null,
    manager?: EntityManager,
  ): Promise<UserProgramExperience[]> {
    try {
      this.logger.log(userProgramExperienceConstMessages.FETCHING_EXISTING(userId, registrationId));
      const repo = this.getExperienceRepository(manager);
      const qb = repo
        .createQueryBuilder('experience')
        .innerJoinAndSelect('experience.user', 'user')
        .innerJoinAndSelect('experience.lookupData', 'lookupData')
        .leftJoinAndSelect('experience.registration', 'registration')
        .where('user.id = :userId', { userId })
        .andWhere('experience.deletedAt IS NULL');

      if (registrationId !== null && registrationId !== undefined) {
        qb.andWhere('registration.id = :registrationId', { registrationId });
      } else {
        qb.andWhere('registration.id IS NULL');
      }

      return await qb.orderBy('experience.id', 'ASC').getMany();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PROGRAM_EXPERIENCE_GET_FAILED, error);
    }
  }

  async findActiveExperiencesForUser(
    userId: number,
    manager?: EntityManager,
  ): Promise<UserProgramExperience[]> {
    try {
      this.logger.log(userProgramExperienceConstMessages.FETCHING_EXISTING(userId));
      const repo = this.getExperienceRepository(manager);

      return await repo
        .createQueryBuilder('experience')
        .innerJoin('experience.user', 'user')
        .innerJoinAndSelect('experience.lookupData', 'lookupData')
        .leftJoinAndSelect('experience.registration', 'registration')
        .where('user.id = :userId', { userId })
        .andWhere('experience.deletedAt IS NULL')
        .orderBy('experience.id', 'ASC')
        .getMany();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PROGRAM_EXPERIENCE_GET_FAILED, error);
    }
  }

  async softDeleteByUserAndRegistration(
    userId: number,
    registrationId: number | null,
    manager?: EntityManager,
  ): Promise<void> {
    this.logger.log(userProgramExperienceConstMessages.REMOVING_EXISTING(userId, registrationId));
    try {
      const queryBuilder = (manager ?? this.experienceRepo.manager)
        .createQueryBuilder()
        .softDelete()
        .from(UserProgramExperience)
        .where('user_id = :userId', { userId });

      // if (registrationId !== null && registrationId !== undefined) {
      //   queryBuilder.andWhere('registration_id = :registrationId', { registrationId });
      // }
      // } else {
      //   queryBuilder.andWhere('registration_id IS NULL');
      // }

      await queryBuilder.execute();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PROGRAM_EXPERIENCE_DELETE_FAILED, error);
    }
  }
}
