import { Injectable } from '@nestjs/common';
import { UserParticipationRepository } from './user-participation.repository';
import {
  UserParticipationSummaryDto,
  ProgramParticipationDto,
  YearWiseParticipationDto,
  GetUserParticipationQueryDto,
  AssociationQuestionDto,
  UserProgramExperienceDto,
} from './dto';
import { UserParticipationSummary } from '../common/entities/user-participation-summary.entity';
import { AppLoggerService } from '../common/services/logger.service';
import { handleKnownErrors } from '../common/utils/handle-error.util';
import { ERROR_CODES } from '../common/constants/error-string-constants';
import { InifniNotFoundException } from '../common/exceptions/infini-notfound-exception';
import { userParticipationConstMessages } from '../common/constants/strings-constants';
import { UserProgramExperienceRepository } from '../user-program-experience/user-program-experience.repository';
import { UserProgramExperience } from '../common/entities/user-program-experience.entity';

@Injectable()
export class UserParticipationService {
  constructor(
    private readonly userParticipationRepository: UserParticipationRepository,
    private readonly userProgramExperienceRepository: UserProgramExperienceRepository,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Get comprehensive user participation data
   */
  async getUserParticipation(
    userId: number, 
    queryDto: GetUserParticipationQueryDto = {}
  ): Promise<UserParticipationSummaryDto> {
    try {
      this.logger.log(userParticipationConstMessages.GET_USER_PARTICIPATION);
      
      // First check if user exists in users table
      const userExistsInUsersTable = await this.userParticipationRepository.checkUserExistsInUsersTable(userId);
      
      if (!userExistsInUsersTable) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
      }

      // Get user basic information from users table
      const userInfo = await this.userParticipationRepository.getUserInfo(userId);
      
      // Get user profile extension data (includes experience information)
      const userProfileExtension = await this.userParticipationRepository.getUserProfileExtension(userId);

      // Fetch recorded user program experiences
      const userProgramExperienceEntities =
        (await this.userProgramExperienceRepository.findActiveExperiencesForUser(userId)) ?? [];
      const userProgramExperiences = this.transformUserProgramExperiences(userProgramExperienceEntities);
      
      // Calculate association years (check extension if not in summary)
      const associationYears = await this.userParticipationRepository.calculateAssociationYearsFromExtension(userId);
      
      // Build association questions with current answers
      const associationQuestions = await this.buildAssociationQuestions(userId, associationYears);
      
      // Check if user has any participation records
      const hasParticipationRecords = await this.userParticipationRepository.userExists(userId);
      
      // If user exists but has no participation records, return empty data with association data
      if (!hasParticipationRecords) {
        this.logger.log(userParticipationConstMessages.USER_PARTICIPATION_RETRIEVED);
        const emptyResult = {
          userId,
          userFullName: userInfo?.fullName || '',
          userEmail: userInfo?.email || '',
          whichOfTheFollowingHaveYouExperienced: userProfileExtension?.whichOfTheFollowingHaveYouExperienced || '',
          contactPerson: userProfileExtension?.infinitheismContact?.orgUsrName || '',
          whenWasYourLastHdbMsd: userProfileExtension?.whenWasYourLastHdbMsd || '',
          howManyHdbMsdHaveYouDone: userProfileExtension?.howManyHdbMsdHaveYouDone || 0,
          totalProgramsAttended: 0,
          firstParticipationDate: null,
          latestParticipationDate: null,
          totalYearsParticipated: 0,
          yearWiseParticipation: [],
          subProgramTypeWiseParticipation: [],
          allPrograms: [],
          uniqueProgramNames: [],
          associationYears,
          associationQuestions,
          userProgramExperiences,
        };
        return emptyResult;
      }

      // Get filtered participation records
      const participationRecords = await this.userParticipationRepository.findByUserIdWithFilters(
        userId,
        queryDto.year,
        queryDto.programName
      );

      // Get overall statistics
      const stats = await this.userParticipationRepository.getUserParticipationStats(userId);
      
      // Get year-wise counts
      const yearWiseCounts = await this.userParticipationRepository.getYearWiseParticipationCount(userId);
      
      // Get sub program type wise counts
      const subProgramTypeWiseCounts = await this.userParticipationRepository.getSubProgramTypeWiseParticipationCount(userId);
      
      // Get unique program names
      const uniqueProgramNames = await this.userParticipationRepository.getUniqueProgramNames(userId);

      // Transform data
      const allPrograms = this.transformToParticipationDto(participationRecords);
      const yearWiseParticipation = this.buildYearWiseParticipation(allPrograms, yearWiseCounts);
      const subProgramTypeWiseParticipation = subProgramTypeWiseCounts.map(item => ({
        subProgramType: item.subProgramType?.replace(/^PST_/, '') ?? '',
        count: item.count,
      }));

      // Use user info from users table (prioritize over participation records)
      const participationUserInfo = participationRecords[0]?.user;

      this.logger.log(userParticipationConstMessages.USER_PARTICIPATION_RETRIEVED);

      const finalResult = {
        userId,
        userFullName: userInfo?.fullName || participationUserInfo?.fullName || '',
        userEmail: userInfo?.email || participationUserInfo?.email || '',
        whichOfTheFollowingHaveYouExperienced: userProfileExtension?.whichOfTheFollowingHaveYouExperienced || '',
        contactPerson: userProfileExtension?.infinitheismContact?.orgUsrName || '',
        whenWasYourLastHdbMsd: userProfileExtension?.whenWasYourLastHdbMsd || '',
        howManyHdbMsdHaveYouDone: userProfileExtension?.howManyHdbMsdHaveYouDone || 0,
        totalProgramsAttended: stats.totalPrograms,
        firstParticipationDate: stats.firstParticipation,
        latestParticipationDate: stats.latestParticipation,
        totalYearsParticipated: stats.totalYears,
        yearWiseParticipation,
        subProgramTypeWiseParticipation,
        allPrograms,
        uniqueProgramNames,
        associationYears,
        associationQuestions,
        userProgramExperiences,
      };
      
      return finalResult;
    } catch (error) {
      if (error instanceof InifniNotFoundException) {
        throw error; // Re-throw custom exceptions
      }
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  private transformUserProgramExperiences(
    experiences: UserProgramExperience[],
  ): UserProgramExperienceDto[] {
    return experiences.map(experience => ({
      id: experience.id,
      registrationId: experience.registration?.id ?? null,
      lookupDataId: experience.lookupData?.id ?? null,
      lookupKey: experience.lookupData?.lookupKey ?? '',
      lookupLabel: experience.lookupData?.lookupLabel ?? '',
      lookupCategory: experience.lookupData?.lookupCategory ?? '',
    }));
  }

  /**
   * Get user participation for a specific year
   */
  async getUserParticipationByYear(userId: number, year: number): Promise<ProgramParticipationDto[]> {
    try {
      this.logger.log(userParticipationConstMessages.GET_USER_PARTICIPATION_BY_YEAR);
      
      const participationRecords = await this.userParticipationRepository.findByUserIdAndYear(userId, year);
      
      if (!participationRecords.length) {
        throw new InifniNotFoundException(ERROR_CODES.USER_PARTICIPATION_BY_YEAR_NOT_FOUND, null, null, userId.toString(), year.toString());
      }

      this.logger.log(userParticipationConstMessages.USER_PARTICIPATION_BY_YEAR_RETRIEVED);
      return this.transformToParticipationDto(participationRecords);
    } catch (error) {
      if (error instanceof InifniNotFoundException) {
        throw error; // Re-throw custom exceptions
      }
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get user participation statistics only
   */
  async getUserParticipationStats(userId: number): Promise<{
    userId: number;
    totalProgramsAttended: number;
    totalYearsParticipated: number;
    firstParticipationDate: Date | null;
    latestParticipationDate: Date | null;
    yearWiseCounts: Array<{ year: number; count: number }>;
    uniqueProgramNames: string[];
  }> {
    try {
      this.logger.log(userParticipationConstMessages.GET_USER_PARTICIPATION_STATS);
      
      const userExists = await this.userParticipationRepository.userExists(userId);
      if (!userExists) {
        throw new InifniNotFoundException(ERROR_CODES.USER_PARTICIPATION_NOT_FOUND, null, null, userId.toString());
      }

      const stats = await this.userParticipationRepository.getUserParticipationStats(userId);
      const yearWiseCounts = await this.userParticipationRepository.getYearWiseParticipationCount(userId);
      const uniqueProgramNames = await this.userParticipationRepository.getUniqueProgramNames(userId);

      this.logger.log(userParticipationConstMessages.USER_PARTICIPATION_STATS_RETRIEVED);
      
      return {
        userId,
        totalProgramsAttended: stats.totalPrograms,
        totalYearsParticipated: stats.totalYears,
        firstParticipationDate: stats.firstParticipation,
        latestParticipationDate: stats.latestParticipation,
        yearWiseCounts,
        uniqueProgramNames,
      };
    } catch (error) {
      if (error instanceof InifniNotFoundException) {
        throw error; // Re-throw custom exceptions
      }
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_STATS_FAILED, error);
    }
  }

  /**
   * Transform participation records to DTOs
   */
  private transformToParticipationDto(records: UserParticipationSummary[]): ProgramParticipationDto[] {
    try {
      this.logger.log(userParticipationConstMessages.TRANSFORMING_DATA);
      
      return records
        .filter(record => record.programStartsAt) // Filter out records without start date
        .map(record => ({
          programId: record.programId,
          programName: record.programName,
          subProgramId: record.subProgramId,
          subProgramName: record.subProgramName,
          subProgramType: record.subProgramType,
          sessionId: record.sessionId,
          sessionName: record.sessionName,
          programStartsAt: record.programStartsAt,
          programEndsAt: record.programEndsAt,
          year: new Date(record.programStartsAt).getFullYear(),
        }));
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Build year-wise participation breakdown
   */
  private buildYearWiseParticipation(
    allPrograms: ProgramParticipationDto[],
    yearWiseCounts: Array<{ year: number; count: number }>
  ): YearWiseParticipationDto[] {
    try {
      this.logger.log(userParticipationConstMessages.BUILDING_YEAR_WISE_BREAKDOWN);
      
      const yearWiseMap = new Map<number, ProgramParticipationDto[]>();

      // Group programs by year
      allPrograms.forEach(program => {
        if (program.year) {
          if (!yearWiseMap.has(program.year)) {
            yearWiseMap.set(program.year, []);
          }
          yearWiseMap.get(program.year)!.push(program);
        }
      });

      // Build the response
      return yearWiseCounts.map(({ year, count }) => ({
        year,
        programsCount: count,
        programs: yearWiseMap.get(year) || [],
      }));
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

    /**
   * Build association questions based on user's association years
   */
  private async buildAssociationQuestions(userId: number, associationYears: number): Promise<AssociationQuestionDto[]> {
    try {
      const questions: AssociationQuestionDto[] = [];

      // Get user info for app usage
      const userInfo = await this.userParticipationRepository.getUserInfo(userId);

      if (associationYears < 1) {
        // Case 1: Association < 1 year - TAT programs (Order 1)
        const latestTat = await this.userParticipationRepository.getLatestTatProgram();
        
        if (latestTat) {
          const tatAttended = await this.userParticipationRepository.hasUserAttendedLatestTat(userId);
          const tatYear = latestTat.program_year || new Date().getFullYear();
          
          const tatQuestion = {
            label: `TAT ${tatYear} attended?`,
            key: 'tat_attended',
            type: 'radio',
            options: ['Yes', 'No'],
            answer: tatAttended ? 'Yes' : 'No',
            order: 1,
            metadata: { 
              programId: latestTat.id, 
              programName: latestTat.name,
              programYear: tatYear,
              attended: tatAttended 
            }
          };
          questions.push(tatQuestion);
        }
      } else {
        // Case 2: Association >= 1 year - HDB programs (Order 1)
        const hdbCount = await this.userParticipationRepository.getHdbParticipationFromProfile(userId, associationYears);
        
        const hdbQuestion = {
          label: 'Number of HDBs attended',
          key: 'hdb_programs_since_association',
          type: 'compare',
          answer: String(hdbCount.attended),
          totalCount: hdbCount.total,
          order: 1,
          metadata: {
            count: hdbCount.attended,
            total: hdbCount.total,
            associationYears: associationYears
          },
        };
        questions.push(hdbQuestion);
      }

      // Infiniminute usage (Order 2)
      const infiniminuteQuestion = {
        label: 'infiniminute user',
        key: 'infiniminute_user',
        type: 'radio',
        options: ['Yes', 'No'],
        answer: userInfo?.isInfiminute ? 'Yes' : 'No',
        order: 3,
        metadata: { usage: userInfo?.isInfiminute }
      };
      questions.push(infiniminuteQuestion);

      // Ravelation usage (Order 3)
      const ravelationQuestion = {
        label: 'Ravelation user',
        key: 'ravelation_user',
        type: 'radio',
        options: ['Yes', 'No'],
        answer: userInfo?.isInfimessages ? 'Yes' : 'No',
        order: 4,
        metadata: { usage: userInfo?.isInfimessages }
      };
      questions.push(ravelationQuestion);

      // Entrainment programs (Order 4)
      const entrainmentCount = await this.userParticipationRepository.getEntrainmentsAttendedFrom2024(userId);
      
      const entrainmentQuestion = {
        label: 'Entrainment 2025',
        key: 'entrainment_programs_2025',
        type: 'radio',
        options: ['Yes', 'No'],
        answer: (entrainmentCount.attended) == 1 ? 'Yes': 'No',
        totalCount: entrainmentCount.total,
        order: 2,
        metadata: {
          count: entrainmentCount.attended,
          total: entrainmentCount.total,
          sinceYear: 2025
        }
      };
      questions.push(entrainmentQuestion);

      // Sort questions by order
      const sortedQuestions = questions.sort((a, b) => a.order - b.order);
      
      return sortedQuestions;
    } catch (error) {
      this.logger.error('Error building association questions:', error);
      return [];
    }
  }
}

