import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, IsNull } from 'typeorm';
import { UserParticipationSummary } from '../common/entities/user-participation-summary.entity';
import { User } from '../common/entities/user.entity';
import { UserProfileExtension } from '../common/entities/user-profile-extension.entity';
import { Program } from '../common/entities/program.entity';
import { ProgramType } from '../common/entities/program-type.entity';
import { SubProgramTypeEnum } from '../common/enum/sub-program-type.enum';
import { CommonDataService } from '../common/services/commonData.service';
import { AppLoggerService } from '../common/services/logger.service';
import { handleKnownErrors } from '../common/utils/handle-error.util';
import { ERROR_CODES } from '../common/constants/error-string-constants';
import { InifniNotFoundException } from '../common/exceptions/infini-notfound-exception';
import { userParticipationConstMessages } from '../common/constants/strings-constants';

@Injectable()
export class UserParticipationRepository {
  constructor(
    @InjectRepository(UserParticipationSummary)
    private readonly userParticipationSummaryRepository: Repository<UserParticipationSummary>,
    @InjectRepository(User)
    private readonly userRepository: Repository<User>,
    @InjectRepository(UserProfileExtension)
    private readonly userProfileExtensionRepository: Repository<UserProfileExtension>,
    @InjectRepository(Program)
    private readonly programRepository: Repository<Program>,
    @InjectRepository(ProgramType)
    private readonly programTypeRepository: Repository<ProgramType>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Get all participation records for a user
   */
  async findByUserId(userId: number): Promise<UserParticipationSummary[]> {
    try {
      this.logger.log(userParticipationConstMessages.FETCHING_PARTICIPATION_RECORDS);
      
      return await this.commonDataService.get(
        this.userParticipationSummaryRepository,
        undefined, // select
        { userId, deletedAt: IsNull() }, // where
        'all', // limit
        0, // offset
        { programStartsAt: 'DESC' }, // sort
        undefined, // search
        ['user'] // relations
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get user participation filtered by year
   */
  async findByUserIdAndYear(userId: number, year: number): Promise<UserParticipationSummary[]> {
    try {
      this.logger.log(userParticipationConstMessages.FILTERING_BY_YEAR);
      
      return await this.userParticipationSummaryRepository
        .createQueryBuilder('ups')
        .leftJoinAndSelect('ups.user', 'user')
        .where('ups.userId = :userId', { userId })
        .andWhere('ups.deletedAt IS NULL')
        .andWhere('EXTRACT(YEAR FROM ups.programStartsAt) = :year', { year })
        .orderBy('ups.programStartsAt', 'DESC')
        .getMany();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get user participation filtered by program name
   */
  async findByUserIdAndProgramName(userId: number, programName: string): Promise<UserParticipationSummary[]> {
    try {
      this.logger.log(userParticipationConstMessages.FILTERING_BY_PROGRAM_NAME);
      
      return await this.userParticipationSummaryRepository
        .createQueryBuilder('ups')
        .leftJoinAndSelect('ups.user', 'user')
        .where('ups.userId = :userId', { userId })
        .andWhere('ups.deletedAt IS NULL')
        .andWhere('LOWER(ups.programName) LIKE LOWER(:programName)', { programName: `%${programName}%` })
        .orderBy('ups.programStartsAt', 'DESC')
        .getMany();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get user participation with combined filters
   */
  async findByUserIdWithFilters(
    userId: number,
    year?: number,
    programName?: string,
  ): Promise<UserParticipationSummary[]> {
    try {
      this.logger.log(userParticipationConstMessages.FETCHING_PARTICIPATION_RECORDS);
      
      const queryBuilder = this.userParticipationSummaryRepository
        .createQueryBuilder('ups')
        .leftJoinAndSelect('ups.user', 'user')
        .where('ups.userId = :userId', { userId })
        .andWhere('ups.deletedAt IS NULL');

      if (year) {
        queryBuilder.andWhere('EXTRACT(YEAR FROM ups.programStartsAt) = :year', { year });
      }

      if (programName) {
        queryBuilder.andWhere('LOWER(ups.programName) LIKE LOWER(:programName)', { 
          programName: `%${programName}%` 
        });
      }

      return await queryBuilder
        .orderBy('ups.programStartsAt', 'DESC')
        .getMany();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get participation statistics for a user
   */
  async getUserParticipationStats(userId: number): Promise<{
    totalPrograms: number;
    totalYears: number;
    firstParticipation: Date | null;
    latestParticipation: Date | null;
  }> {
    try {
      this.logger.log(userParticipationConstMessages.FETCHING_PARTICIPATION_STATS);
      
      const result = await this.userParticipationSummaryRepository
        .createQueryBuilder('ups')
        .select([
          'COUNT(ups.id) as "totalPrograms"',
          'COUNT(DISTINCT EXTRACT(YEAR FROM ups.programStartsAt)) as "totalYears"',
          'MIN(ups.programStartsAt) as "firstParticipation"',
          'MAX(ups.programStartsAt) as "latestParticipation"',
        ])
        .where('ups.userId = :userId', { userId })
        .andWhere('ups.deletedAt IS NULL')
        .getRawOne();

      return {
        totalPrograms: parseInt(result.totalPrograms) || 0,
        totalYears: parseInt(result.totalYears) || 0,
        firstParticipation: result.firstParticipation || null,
        latestParticipation: result.latestParticipation || null,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_STATS_FAILED, error);
    }
  }

  /**
   * Get year-wise participation count for a user
   */
  async getYearWiseParticipationCount(userId: number): Promise<Array<{ year: number; count: number }>> {
    try {
      this.logger.log(userParticipationConstMessages.FETCHING_PARTICIPATION_STATS);
      
      const result = await this.userParticipationSummaryRepository
        .createQueryBuilder('ups')
        .select([
          'EXTRACT(YEAR FROM ups.programStartsAt) as year',
          'COUNT(ups.id) as count',
        ])
        .where('ups.userId = :userId', { userId })
        .andWhere('ups.deletedAt IS NULL')
        .andWhere('ups.programStartsAt IS NOT NULL')
        .groupBy('EXTRACT(YEAR FROM ups.programStartsAt)')
        .orderBy('year', 'DESC')
        .getRawMany();

      return result.map(row => ({
        year: parseInt(row.year),
        count: parseInt(row.count),
      }));
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_YEAR_COUNT_FAILED, error);
    }
  }

  /**
   * Get unique program names for a user
   */
  async getUniqueProgramNames(userId: number): Promise<string[]> {
    try {
      this.logger.log(userParticipationConstMessages.FETCHING_PARTICIPATION_RECORDS);
      
      const result = await this.userParticipationSummaryRepository
        .createQueryBuilder('ups')
        .select('DISTINCT ups.programName', 'programName')
        .where('ups.userId = :userId', { userId })
        .andWhere('ups.deletedAt IS NULL')
        .andWhere('ups.programName IS NOT NULL')
        .orderBy('ups.programName', 'ASC')
        .getRawMany();

      return result.map(row => row.programName);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_UNIQUE_PROGRAMS_FAILED, error);
    }
  }

  /**
   * Get sub program type wise participation count for a user
   */
  async getSubProgramTypeWiseParticipationCount(userId: number): Promise<Array<{ subProgramType: string; count: number }>> {
    try {
      this.logger.log(userParticipationConstMessages.FETCHING_PARTICIPATION_STATS);
      
      const result = await this.userParticipationSummaryRepository
        .createQueryBuilder('ups')
        .select([
          'ups.subProgramType as "subProgramType"',
          'COUNT(ups.id) as count',
        ])
        .where('ups.userId = :userId', { userId })
        .andWhere('ups.deletedAt IS NULL')
        .andWhere('ups.subProgramType IS NOT NULL')
        .groupBy('ups.subProgramType')
        .orderBy('ups.subProgramType', 'ASC')
        .getRawMany();

      return result.map(row => ({
        subProgramType: row.subProgramType,
        count: parseInt(row.count),
      }));
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_YEAR_COUNT_FAILED, error);
    }
  }

  /**
   * Check if user exists in users table
   */
  async checkUserExistsInUsersTable(userId: number): Promise<boolean> {
    try {
      this.logger.log(userParticipationConstMessages.CHECKING_USER_EXISTS);
      
      const user = await this.userRepository.findOne({
        where: { id: userId },
        select: ['id']
      });
      return !!user;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get user basic information from users table
   */
  async getUserInfo(userId: number): Promise<{ 
    fullName: string; 
    email: string; 
    phoneNumber: string; 
    dob: Date; 
    gender: string;
    isInfiminute: boolean;
    isInfimessages: boolean;
    isInfiminuteUserActive: boolean;
    isInfimessageUserActive: boolean;
  } | null> {
    try {
      this.logger.log(userParticipationConstMessages.CHECKING_USER_EXISTS);
      
      const user = await this.userRepository.findOne({
        where: { id: userId },
        select: [
          'fullName', 
          'email', 
          'phoneNumber', 
          'dob', 
          'gender',
          'isInfiminute',
          'isInfimessages', 
          'isInfiminuteUserActive',
          'isInfimessageUserActive'
        ]
      });
      
      const result = user ? { 
        fullName: user.fullName || '', 
        email: user.email || '', 
        phoneNumber: user.phoneNumber || '',
        dob: user.dob || null,
        gender: user.gender || '',
        isInfiminute: user.isInfiminute || false,
        isInfimessages: user.isInfimessages || false,
        isInfiminuteUserActive: user.isInfiminuteUserActive || false,
        isInfimessageUserActive: user.isInfimessageUserActive || false
      } : null;
      
      return result;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Check if user exists in participation records
   */
  async userExists(userId: number): Promise<boolean> {
    try {
      this.logger.log(userParticipationConstMessages.CHECKING_USER_EXISTS);
      
      const count = await this.userParticipationSummaryRepository.count({
        where: { userId, deletedAt: IsNull() },
      });
      return count > 0;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get user profile extension data
   */
  async getUserProfileExtension(userId: number): Promise<UserProfileExtension | null> {
    try {
      this.logger.log('Fetching user profile extension data');
      
      return await this.userProfileExtensionRepository.findOne({
        where: { user: { id: userId } },
        relations: ['infinitheismContact'],
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get count of Entrainments attended from 2024
   */
  async getEntrainmentsAttendedFrom2024(userId: number): Promise<{ attended: number; total: number }> {
    try {
      this.logger.log('Getting Entrainments attended from 2024');
      
      // Get count of user's entrainment participation from 2024
      const attendedCount = await this.userParticipationSummaryRepository.count({
        where: { 
          userId, 
          deletedAt: IsNull(),
          subProgramType: SubProgramTypeEnum.PST_ENTRAINMENT
        },
      });

      // Get total count of entrainment programs since 2024
      const totalQuery = await this.programRepository
        .createQueryBuilder('p')
        .leftJoin('p.type', 'pt')
        .select('COUNT(DISTINCT p.id)', 'total')
        .where('p.deletedAt IS NULL')
        .andWhere('pt.key = :programType', { programType: 'PT_ENTRAINMENT' })
        .andWhere('EXTRACT(YEAR FROM p.startsAt) >= :year', { year: 2024 })
        .getRawOne();

      const result = {
        attended: attendedCount,
        total: parseInt(totalQuery.total) || 0
      };
      
      return result;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Calculate years since association with infinitheism
   */
  async calculateAssociationYears(userId: number): Promise<number> {
    try {
      this.logger.log('Calculating years since association');
      
      // Get user info and profile extension
      const user = await this.userRepository.findOne({
        where: { id: userId },
        select: ['createdAt']
      });

      const userProfile = await this.userProfileExtensionRepository
        .createQueryBuilder('upe')
        .leftJoin('upe.user', 'u')
        .select('upe.sinceWhenHaveYouBeenAssociatedWithThisPath', 'sinceWhenHaveYouBeenAssociatedWithThisPath')
        .where('u.id = :userId', { userId })
        .getRawOne();

      if (!userProfile && !user) {
        throw new InifniNotFoundException(`User ${userId} not found`);
      }

      let associationDate: Date;

      if (userProfile?.sinceWhenHaveYouBeenAssociatedWithThisPath) {
        // Try to parse the association date from profile extension
        const associationYear = parseInt(userProfile.sinceWhenHaveYouBeenAssociatedWithThisPath);
        
        if (!isNaN(associationYear) && associationYear >= 1995 && associationYear <= new Date().getFullYear()) {
          associationDate = new Date(associationYear, 0, 1); // Start of the year
        } else {
          associationDate = user?.createdAt || new Date();
        }
      } else {
        // Fallback to user creation date
        associationDate = user?.createdAt || new Date();
      }

      const currentDate = new Date();
      const yearsDiff = currentDate.getFullYear() - associationDate.getFullYear();
      
      // Adjust for month and day
      const currentMonth = currentDate.getMonth();
      const associationMonth = associationDate.getMonth();
      
      let finalYears;
      if (currentMonth < associationMonth || 
          (currentMonth === associationMonth && currentDate.getDate() < associationDate.getDate())) {
        finalYears = Math.max(0, yearsDiff - 1);
      } else {
        finalYears = Math.max(0, yearsDiff);
      }
      
      return finalYears;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Get the latest TAT program using TypeORM QueryBuilder with injected entities
   */
  async getLatestTatProgram(): Promise<any> {
    try {
      const result = await this.programRepository
        .createQueryBuilder('p')
        .leftJoin('p.type', 'pt')
        .select('p.id', 'id')
        .addSelect('p.name', 'name')
        .addSelect('p.description', 'description')
        .addSelect('pt.key', 'key')
        .addSelect('p.startsAt', 'startsAt')
        .addSelect('p.endsAt', 'endsAt')
        .where('pt.key = :programType', { programType: 'PT_TAT' })
        .andWhere('p.deletedAt IS NULL')
        .orderBy('p.startsAt', 'DESC')
        .getRawOne();
      
      if (!result) {
        return null;
      }

      const formattedResult = {
        id: result.id,
        name: result.name,
        description: result.description,
        program_type_key: result.key,
        program_year: result.startsAt ? new Date(result.startsAt).getFullYear() : new Date().getFullYear(),
        starts_at: result.startsAt,
        ends_at: result.endsAt
      };
      
      return formattedResult;
    } catch (error) {
      this.logger.error('Error getting latest TAT program:', error);
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
    }
  }

  /**
   * Check if user attended the latest TAT program using TypeORM QueryBuilder
   */
  async hasUserAttendedLatestTat(userId: number): Promise<boolean> {
    try {
      const latestTat = await this.getLatestTatProgram();
      
      if (!latestTat) {
        return false;
      }
      
      const count = await this.userParticipationSummaryRepository
        .createQueryBuilder('ups')
        .where('ups.userId = :userId', { userId })
        .andWhere('ups.programId = :programId', { programId: latestTat.id })
        .andWhere('ups.subProgramType = :subProgramType', { subProgramType: 'PST_TAT' })
        .andWhere('ups.deletedAt IS NULL')
        .getCount();
      
      const attended = count > 0;
      
      return attended;
    } catch (error) {
      this.logger.error('Error checking TAT attendance:', error);
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
      return false;
    }
  }

  /**
   * Get available Entrainment program years using TypeORM QueryBuilder with injected entities
   */
  async getEntrainmentProgramYears(): Promise<string[]> {
    try {
      const results = await this.programRepository
        .createQueryBuilder('p')
        .leftJoin('p.type', 'pt')
        .select('DISTINCT EXTRACT(YEAR FROM p.startsAt)', 'program_year')
        .where('pt.key = :programType', { programType: 'PT_ENTRAINMENT' })
        .andWhere('p.deletedAt IS NULL')
        .orderBy('program_year', 'DESC')
        .getRawMany();
      
      return results.map(row => row.program_year.toString());
    } catch (error) {
      this.logger.error('Error getting Entrainment program years:', error);
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
      return [];
    }
  }

  /**
   * Get user's association years from user_profile_extension if not in user_participation_summary
   */
  async calculateAssociationYearsFromExtension(userId: number): Promise<number> {
    try {
      // First try to get from user_participation_summary
      const existingAssociation = await this.calculateAssociationYears(userId);
      if (existingAssociation > 0) {
        return existingAssociation;
      }

      // If not found, get from user_profile_extension using TypeORM QueryBuilder
      const result = await this.userProfileExtensionRepository
        .createQueryBuilder('upe')
        .leftJoin('upe.user', 'u')
        .select('upe.sinceWhenHaveYouBeenAssociatedWithThisPath', 'sinceWhenHaveYouBeenAssociatedWithThisPath')
        .where('u.id = :userId', { userId })
        .getRawOne();
      
      const sinceYear = result?.sinceWhenHaveYouBeenAssociatedWithThisPath;
      
      if (!sinceYear) {
        return 0; // No association data found
      }

      const currentYear = new Date().getFullYear();
      return Math.max(0, currentYear - parseInt(sinceYear));
    } catch (error) {
      this.logger.error('Error calculating association years from extension:', error);
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
      return 0;
    }
  }

  /**
   * Get HDB programs that have parent_id null (main programs) using TypeORM QueryBuilder with injected entities
   */
  async getMainHdbPrograms(): Promise<any[]> {
    try {
      const results = await this.programRepository
        .createQueryBuilder('p')
        .leftJoin('p.type', 'pt')
        .select('p.id', 'id')
        .addSelect('p.name', 'name')
        .addSelect('p.description', 'description')
        .addSelect('pt.key', 'key')
        .addSelect('p.startsAt', 'startsAt')
        .addSelect('p.endsAt', 'endsAt')
        .where('pt.key = :programType', { programType: 'PT_HDBMSD' })
        .andWhere('p.deletedAt IS NULL')
        .andWhere('(p.primaryProgramId IS NULL OR p.primaryProgramId = p.id)') // Main programs
        .orderBy('p.startsAt', 'DESC')
        .getRawMany();
      
      return results.map(result => ({
        id: result.id,
        name: result.name,
        description: result.description,
        program_type_key: result.key,
        program_year: result.startsAt ? new Date(result.startsAt).getFullYear() : new Date().getFullYear(),
        starts_at: result.startsAt,
        ends_at: result.endsAt
      }));
    } catch (error) {
      this.logger.error('Error getting main HDB programs:', error);
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
      return [];
    }
  }

  /**
   * Get HDB count from user profile extension and total attended since association
   */
  async getHdbParticipationFromProfile(userId: number, associationYears: number): Promise<{ attended: number; total: number; }> {
    try {
      // Get HDB count from user profile extension
      const profileResult = await this.userProfileExtensionRepository
        .createQueryBuilder('upe')
        .leftJoin('upe.user', 'u')
        .select('upe.howManyHdbMsdHaveYouDone', 'howManyHdbMsdHaveYouDone')
        .where('u.id = :userId', { userId })
        .getRawOne();
      
      const attendedFromProfile = parseInt(profileResult?.howManyHdbMsdHaveYouDone || '0');

      // Calculate since when to count (based on association years)
      // const sinceYear = new Date().getFullYear() - Math.floor(associationYears);
      
      // Get total HDB programs available since association using PST_HDB
      // Get total HDB programs available since association from program_v1 where is_primary_program = true
      // const totalCount = await this.programRepository
      //   .createQueryBuilder('p')
      //   .leftJoin('p.type', 'pt')
      //   .where('pt.key = :programType', { programType: 'PT_HDBMSD' })
      //   .andWhere('p.isPrimaryProgram = true')
      //   .andWhere('EXTRACT(YEAR FROM p.startsAt) >= :sinceYear', { sinceYear })
      //   .andWhere('p.deletedAt IS NULL')
      //   .getCount();

      const result = { 
        attended: attendedFromProfile, 
        total: associationYears
      };
      
      return result;
    } catch (error) {
      this.logger.error('Error getting HDB participation from profile:', error);
      handleKnownErrors(ERROR_CODES.USER_PARTICIPATION_GET_FAILED, error);
      return { attended: 0, total: 0 };
    }
  }
}
