import { Injectable } from '@nestjs/common';
import { DataSource, EntityManager } from 'typeorm';
import { SeekerProgramExperienceRepository } from './seeker-program-experience.repository';
import { CreateSeekerProgramExperienceDto } from './dto/create-seeker-program-experience.dto';
import { UpdateSeekerProgramExperienceDto } from './dto/update-seeker-program-experience.dto';
import { DeleteSeekerProgramExperienceDto } from './dto/delete-seeker-program-experience.dto';
import { FilterSeekerProgramExperienceDto } from './dto/filter-seeker-program-experience.dto';
import { AppLoggerService } from 'src/common/services/logger.service';
import { seekerExperienceRecordMessages } from 'src/common/constants/strings-constants';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { ProgramRegistration, SeekerProgramExperience, User } from 'src/common/entities';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';

@Injectable()
export class SeekerProgramExperienceService {
  constructor(
    private readonly repository: SeekerProgramExperienceRepository,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
  ) {}

  async create(dto: CreateSeekerProgramExperienceDto, seeker: User): Promise<any> {
    this.logger.log(seekerExperienceRecordMessages.CREATE_REQUEST(dto.registrationId));
    try {
      return await this.dataSource.transaction(async (manager) => {
        const existing = await this.repository.findByRegId(dto.registrationId, false, manager);
        if (existing) {
          this.logger.log(seekerExperienceRecordMessages.EXISTING_RECORD_FOUND(existing.id));
          throw new InifniBadRequestException(
            ERROR_CODES.SEEKER_PROGRAM_EXPERIENCE_ALREADY_EXISTS,
            null,
            null,
            dto.registrationId.toString(),
          );
        }
        const { user, registration } = await this.loadSeekers(
          dto.registrationId,
          manager,
        );
        const record = this.repository.createRecord(
          {
            userId: user ? user.id : null,
            registrationId: registration ? registration.id : null,
            type: dto.type,
            message: dto.message,
            mediaUrl: dto.mediaUrl,
            isPrivate: dto.isPrivate ?? false,
            isViewed: dto.isViewed ?? false,
            isApproved: dto.isApproved ?? false,
            createdById: seeker.id,
            updatedById: seeker.id,
          },
          manager,
        );
        const result = await this.repository.saveRecord(record, manager);
        return {
          id: result.id,
          userId: result.userId,
          registrationId: result.registrationId,
          type: result.type,
          message: result.message,
          mediaUrl: result.mediaUrl,
          isPrivate: result.isPrivate,
          isViewed: result.isViewed,
          isApproved: result.isApproved
        };
      });
    } catch (error) {
      this.logger.error('CREATE SeekerProgramExperience failed', error);
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_SAVE_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateSeekerProgramExperienceDto, seeker: User): Promise<any> {
    this.logger.log(seekerExperienceRecordMessages.UPDATE_REQUEST(id));
    try {
      return await this.dataSource.transaction(async (manager) => {
        const existing = await this.ensureRecord(id, false, manager);
        if (!existing) {
          throw new InifniNotFoundException(ERROR_CODES.SEEKER_EXPERIENCE_NOT_FOUND, null, null, id.toString());
        }
        if (!existing.registrationId) {
          throw new InifniBadRequestException(
            ERROR_CODES.SEEKER_EXPERIENCE_REGISTRATION_MISSING,
            null,
            null,
            id.toString(),
          );
        }
        const { user, registration } = await this.loadSeekers(
          existing?.registrationId,
          manager,
        );
        if (dto.type !== undefined) existing.type = dto.type;
        if (dto.message !== undefined) existing.message = dto.message;
        if (dto.mediaUrl !== undefined) existing.mediaUrl = dto.mediaUrl;
        if (dto.isPrivate !== undefined) existing.isPrivate = dto.isPrivate;
        if (dto.isViewed !== undefined) existing.isViewed = dto.isViewed;
        if (dto.isApproved !== undefined) existing.isApproved = dto.isApproved;
        if (registration !== undefined) existing.registration = registration ?? null;
        if (seeker) {
          existing.updatedById = seeker.id;
        }
        const result = await this.repository.saveRecord(existing, manager);
        return {
          id: result.id,
          userId: result.userId,
          registrationId: result.registrationId,
          type: result.type,
          message: result.message,
          mediaUrl: result.mediaUrl,
          isPrivate: result.isPrivate,
          isViewed: result.isViewed,
          isApproved: result.isApproved
        };
      });
    } catch (error) {
      this.logger.error('UPDATE SeekerProgramExperience failed', error);
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_UPDATE_FAILED, error);
    }
  }

  async findById(id: number, includeDeleted = false): Promise<SeekerProgramExperience> {
    this.logger.log(seekerExperienceRecordMessages.FETCH_REQUEST(id));
    try {
      const record = await this.repository.findById(id, includeDeleted);
      if (!record) {
        throw new InifniNotFoundException(ERROR_CODES.SEEKER_EXPERIENCE_NOT_FOUND, null, null, id.toString());
      }
      return record;
    } catch (error) {
      this.logger.error('FIND_BY_ID SeekerProgramExperience failed', error);
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_FETCH_FAILED, error);
    }
  }

  async findAll(filter: FilterSeekerProgramExperienceDto) {
    this.logger.log(seekerExperienceRecordMessages.LISTING(filter.limit, filter.offset, filter.search));
    try {
      return await this.repository.fetchAll(filter);
    } catch (error) {
      this.logger.error('FIND_ALL SeekerProgramExperience failed', error);
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_FETCH_FAILED, error);
    }
  }

  async softDelete(id: number, dto: DeleteSeekerProgramExperienceDto, seeker: any): Promise<any> {
    this.logger.log(seekerExperienceRecordMessages.DELETE_REQUEST(id));
    try {
      return await this.dataSource.transaction(async (manager) => {
        const record = await this.ensureRecord(id, false, manager);
        if (!record) {
          throw new InifniNotFoundException(ERROR_CODES.SEEKER_EXPERIENCE_NOT_FOUND, null, null, id.toString());
        }
        if (record.deletedAt) {
          this.logger.log(seekerExperienceRecordMessages.ALREADY_DELETED(id));
          return record;
        }
        const deletedBy = await this.loadUser(seeker.id, manager);
        record.deletedById = deletedBy.id;
        record.deletedReason = dto.deletedReason ?? null;
        record.deletedAt = new Date();
        const result = await this.repository.saveRecord(record, manager);
        return {
          id: result.id,
          userId: result.userId,
          registrationId: result.registrationId,
          type: result.type,
          message: result.message,
          mediaUrl: result.mediaUrl,
          isPrivate: result.isPrivate,
          isViewed: result.isViewed,
          isApproved: result.isApproved,
          deletedAt: result.deletedAt
        };
      });
    } catch (error) {
      this.logger.error('SOFT_DELETE SeekerProgramExperience failed', error);
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_DELETE_FAILED, error);
    }
  }

  private async loadSeekers(
    registrationId?: number,
    manager?: EntityManager,
  ): Promise<{
    user?: User;
    registration?: ProgramRegistration | null;
  }> {
    let registration: ProgramRegistration | null | undefined = undefined;
    if (registrationId !== undefined) {
      registration = await this.loadRegistration(registrationId, manager);
    }
    let user: User | undefined = undefined;
    user = registration?.userId ? await this.loadUser(registration.userId, manager) : undefined;

    return { user, registration };
  }

  private async loadUser(userId?: number, manager?: EntityManager): Promise<User> {
    if (!userId) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, '');
    }
    const user = await this.repository.findUser(userId, manager);
    if (!user) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
    }
    return user;
  }

  private async loadRegistration(registrationId: number, manager?: EntityManager) {
    const registration = await this.repository.findRegistration(registrationId, manager);
    if (!registration) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        registrationId.toString(),
      );
    }
    return registration;
  }

  private async ensureRecord(id: number, includeDeleted = false, manager?: EntityManager) {
    const record = await this.repository.findById(id, includeDeleted, manager);
    if (!record) {
      throw new InifniNotFoundException(ERROR_CODES.SEEKER_EXPERIENCE_NOT_FOUND, null, null, id.toString());
    }
    return record;
  }
    async findByRegistrationId(registrationId: number, includeDeleted = false): Promise<SeekerProgramExperience | null> {
    this.logger.log(`Fetch seeker program experience by registrationId: ${registrationId}`);
    try {
      const record = await this.repository.findByRegId(registrationId, includeDeleted);
      return record;
    } catch (error) {
      this.logger.error('FIND_BY_REGISTRATION_ID SeekerProgramExperience failed', error);
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_FETCH_FAILED, error);
    }
  }

  async deleteByRegistrationId(
    registrationId: number,
    seeker: any,
    dto: DeleteSeekerProgramExperienceDto,
  ): Promise<void> {
    this.logger.log(`Delete seeker program experience by registrationId: ${registrationId}`);
    try {
      await this.dataSource.transaction(async (manager) => {
        const record = await this.repository.findByRegId(registrationId, false, manager);
        if (!record) {
          throw new InifniNotFoundException(ERROR_CODES.SEEKER_EXPERIENCE_NOT_FOUND, null, null, registrationId.toString());
        }
        if (record.deletedAt) {
          this.logger.log(seekerExperienceRecordMessages.ALREADY_DELETED(registrationId));
          return;
        }
        const deletedBy = await this.loadUser(seeker.id, manager);
        record.deletedBy = deletedBy;
        record.deletedReason = dto.deletedReason ?? null;
        record.deletedAt = new Date();
        await this.repository.saveRecord(record, manager);
      });
    } catch (error) {
      this.logger.error('DELETE_BY_REGISTRATION_ID SeekerProgramExperience failed', error);
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_DELETE_FAILED, error);
    }
  }
}