import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, Between, In } from 'typeorm';
import { SchedulerLog } from '../common/entities/scheduler-log.entity';
import { SchedulerLogFilters, SchedulerLogStatistics } from './interface/scheduler.interface';

@Injectable()
export class SchedulerLogRepository {
  constructor(
    @InjectRepository(SchedulerLog)
    private readonly schedulerLogRepo: Repository<SchedulerLog>,
  ) {}

  /**
   * Create a new scheduler log entry
   */
  async createSchedulerLog(data: Partial<SchedulerLog>): Promise<SchedulerLog> {
    const log = this.schedulerLogRepo.create(data);
    return await this.schedulerLogRepo.save(log);
  }

  /**
   * Update an existing scheduler log entry
   */
  async updateSchedulerLog(id: number, data: Partial<SchedulerLog>): Promise<SchedulerLog | null> {
    await this.schedulerLogRepo.update(id, data);
    return await this.schedulerLogRepo.findOne({ where: { id } });
  }

  /**
   * Find scheduler logs with optional filters
   */
  async findSchedulerLogs(
    filters: SchedulerLogFilters = {},
    limit: number = 100,
    offset: number = 0,
  ): Promise<{ logs: SchedulerLog[]; total: number }> {
    const queryBuilder = this.schedulerLogRepo.createQueryBuilder('log');

    if (filters.jobName) {
      queryBuilder.andWhere('log.job_name = :jobName', { jobName: filters.jobName });
    }

    if (filters.jobType) {
      queryBuilder.andWhere('log.job_type = :jobType', { jobType: filters.jobType });
    }

    if (filters.executionStatus) {
      queryBuilder.andWhere('log.execution_status = :executionStatus', {
        executionStatus: filters.executionStatus,
      });
    }

    if (filters.triggeredType) {
      queryBuilder.andWhere('log.triggered_type = :triggeredType', {
        triggeredType: filters.triggeredType,
      });
    }

    if (filters.programId) {
      queryBuilder.andWhere('log.program_id = :programId', { programId: filters.programId });
    }

    if (filters.registrationId) {
      queryBuilder.andWhere('log.registration_id = :registrationId', {
        registrationId: filters.registrationId,
      });
    }

    if (filters.triggeredByUserId) {
      queryBuilder.andWhere('log.triggered_by_user_id = :triggeredByUserId', {
        triggeredByUserId: filters.triggeredByUserId,
      });
    }

    if (filters.startDate && filters.endDate) {
      queryBuilder.andWhere('log.created_at BETWEEN :startDate AND :endDate', {
        startDate: filters.startDate,
        endDate: filters.endDate,
      });
    }

    queryBuilder.orderBy('log.created_at', 'DESC');
    queryBuilder.skip(offset);
    queryBuilder.take(limit);

    const [logs, total] = await queryBuilder.getManyAndCount();

    return { logs, total };
  }

  /**
   * Get scheduler log by ID
   */
  async findSchedulerLogById(id: number): Promise<SchedulerLog | null> {
    return await this.schedulerLogRepo.findOne({
      where: { id },
      relations: ['triggeredByUser'],
    });
  }

  /**
   * Get statistics for scheduler logs
   */
  async getSchedulerStatistics(filters: SchedulerLogFilters = {}): Promise<SchedulerLogStatistics> {
    const queryBuilder = this.schedulerLogRepo.createQueryBuilder('log');

    // Apply filters
    if (filters.jobName) {
      queryBuilder.andWhere('log.job_name = :jobName', { jobName: filters.jobName });
    }

    if (filters.jobType) {
      queryBuilder.andWhere('log.job_type = :jobType', { jobType: filters.jobType });
    }

    if (filters.startDate && filters.endDate) {
      queryBuilder.andWhere('log.created_at BETWEEN :startDate AND :endDate', {
        startDate: filters.startDate,
        endDate: filters.endDate,
      });
    }

    const logs = await queryBuilder.getMany();

    const statistics: SchedulerLogStatistics = {
      totalJobs: logs.length,
      successfulJobs: logs.filter((log) => log.executionStatus === 'success').length,
      failedJobs: logs.filter((log) => log.executionStatus === 'failed').length,
      pendingJobs: logs.filter((log) => log.executionStatus === 'pending').length,
      averageExecutionTime: 0,
      byJobType: {},
      byStatus: {},
    };

    // Calculate average execution time
    const completedLogs = logs.filter((log) => log.executionTimeMs != null);
    if (completedLogs.length > 0) {
      const totalTime = completedLogs.reduce((sum, log) => sum + log.executionTimeMs, 0);
      statistics.averageExecutionTime = Math.round(totalTime / completedLogs.length);
    }

    // Group by job type
    logs.forEach((log) => {
      statistics.byJobType[log.jobType] = (statistics.byJobType[log.jobType] || 0) + 1;
      statistics.byStatus[log.executionStatus] = (statistics.byStatus[log.executionStatus] || 0) + 1;
    });

    return statistics;
  }

  /**
   * Get recent failed jobs for retry
   */
  async getFailedJobsForRetry(jobType: string, maxAttempts: number = 3): Promise<SchedulerLog[]> {
    return await this.schedulerLogRepo.find({
      where: {
        jobType,
        executionStatus: 'failed',
      },
      order: {
        createdAt: 'DESC',
      },
      take: 50,
    });
  }
}
