import { Injectable } from '@nestjs/common';
import { RoomInventoryRepository } from './room-inventory.repository';
import { ProgramRoomInventoryMapRepository } from './program-room-inventory-map.repository';
import { RoomStatus } from '../common/enums/room-status.enum';
import { RoomStatusKpiDto } from './dto/room-inventory-response.dto';
import { ProgramRoomInventoryMap } from '../common/entities/program-room-inventory-map.entity';
import { AppLoggerService } from '../common/services/logger.service';
import { handleKnownErrors } from '../common/utils/handle-error.util';
import { ERROR_CODES } from '../common/constants/error-string-constants';
import { UserRepository } from '../user/user.repository';
import { ROLE_KEYS } from '../common/constants/strings-constants';
import { LookupDataRepository } from 'src/lookup-data/lookup-data.repository';
import { FilterConfig } from '../common/interfaces/interface';
import {
  ROOM_INVENTORY_AGE_FILTER_OPTIONS,
  ROOM_INVENTORY_GENDER_FILTER_OPTIONS,
  ROOM_INVENTORY_HDB_FILTER_OPTIONS,
  ROOM_INVENTORY_ROOMMATE_PREFERENCE_OPTIONS,
  ROOM_INVENTORY_PAIRED_STATUS_OPTIONS,
  ROOM_INVENTORY_DEPARTURE_DATE_FILTER_OPTIONS,
  ROOM_INVENTORY_RETURN_TERMINAL_OPTIONS,
  ROOM_INVENTORY_FILTER_CONFIG_BASE,
  ROOM_STATUS_CONFIG,
  ROOM_INVENTORY_CONTROLLER_FILTERS,
} from '../common/constants/constants';
import { ProgramRegistrationRepository } from 'src/program-registration/program-registration.repository';
import InifniBadRequestException from '../common/exceptions/infini-badrequest-exception';
import { UpdateRoomInventoryDto } from './dto/update-room-inventory.dto';
import { InifniNotFoundException } from '../common/exceptions/infini-notfound-exception';

/**
 * Service for managing room inventory operations
 * Provides business logic for room inventory listing, KPI calculations, and data aggregation
 */
@Injectable()
export class RoomInventoryService {
  constructor(
    private readonly roomInventoryRepository: RoomInventoryRepository,
    private readonly programRoomInventoryMapRepository: ProgramRoomInventoryMapRepository,
    private readonly userRepository: UserRepository,
    private readonly logger: AppLoggerService,
    private readonly lookupDataRepository: LookupDataRepository,
    private readonly programRegistrationRepository: ProgramRegistrationRepository
  ) {}

  /**
   * Retrieves all room inventory items with complete relations and calculates KPIs
   * @param limit - Number of records per page
   * @param offset - Offset for pagination
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param searchText - Optional search text for room number or label
   * @param filter - Optional advanced filter object with additional criteria
   * @returns Promise with raw room inventory data, KPIs, and filter options
   */
  async getAllRoomInventoryWithKpis(
    limit: number = 10,
    offset: number = 0,
    programId?: number,
    subProgramId?: number,
    searchText?: string,
    filter?: any
  ): Promise<{
    data: ProgramRoomInventoryMap[];
    kpis: RoomStatusKpiDto[];
    filters: Array<{
      key: string;
      label: string;
      type: string;
      sortable: boolean;
      filterable: boolean;
      order: number;
      options: Array<{ label: string; value: string }>;
    }>;
    total: number;
    limit: number;
    offset: number;
  }> {
    this.logger.log('Room inventory with KPIs and filters request received', {
      limit,
      offset,
      programId,
      subProgramId,
      searchText,
      filter
    });

    try {
      // Get room inventory data with pagination from repository
      const { data, totalRecords } = await this.roomInventoryRepository.findRoomInventoryWithPagination(
        limit,
        offset,
        programId,
        subProgramId,
        searchText,
        filter
      );

      // Calculate KPIs for all data (not just paginated results)
      const kpis = await this.calculateRoomInventoryKpis(programId, subProgramId, searchText);

      // Get filter dropdown options for property and block
      const [propertyOptions, blockOptions,floorOptions ] = await Promise.all([
        this.roomInventoryRepository.getDistinctVenues(programId, subProgramId),
        this.roomInventoryRepository.getDistinctBlocks(programId, subProgramId),
        this.roomInventoryRepository.getDistinctFloors(programId, subProgramId),
      ]);

      // Add default "All" option at the top
      propertyOptions.unshift({ label: 'All', value: [...new Set(propertyOptions.map(option => option.value))].join(',') });
      blockOptions.unshift({ label: 'All', value: [...new Set(blockOptions.map(option => option.value))].join(',') });
      // Build filters in the detailed format
      const filters: FilterConfig[] = ROOM_INVENTORY_CONTROLLER_FILTERS.map(baseFilter => {
        const filterConfig: FilterConfig = { 
          ...baseFilter,
          options: []
        };
        
        // Assign appropriate options based on filter key
        switch (filterConfig.key) {
          case 'venues':
            filterConfig.options = propertyOptions;
            break;
          case 'block':
            filterConfig.options = blockOptions;
            break;
          case 'floor':
            filterConfig.options = floorOptions;
            break;
          default:
            filterConfig.options = [];
            this.logger.warn('Unknown filter key found in controller filters', { key: filterConfig.key });
        }
        
        return filterConfig;
      });

      this.logger.log('Room inventory data processed successfully', {
        totalRecords,
        currentPageRecords: data.length,
        kpisSummary: {
          statusBreakdownCount: kpis.length
        },
        filters: {
          propertyCount: propertyOptions.length,
          blockCount: blockOptions.length
        }
      });

      return {
        data,
        kpis,
        filters,
        total: totalRecords,
        limit,
        offset
      };

    } catch (error) {
      this.logger.error('Error in room inventory retrieval with KPIs service', error.stack, {
        limit,
        offset,
        programId,
        subProgramId,
        searchText,
        filter
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_GET_FAILED, error);
    }
  }

  /**
   * Retrieves simplified room inventory KPIs - returns array of status and count
   * @param programId - Optional program ID filter for KPI calculation
   * @param subProgramId - Optional sub program ID filter for KPI calculation
   * @param searchText - Optional search text filter for KPI calculation
   * @returns Promise<RoomStatusKpiDto[]> Array of room status and count
   */
  async getRoomInventoryKpis(
    programId?: number,
    subProgramId?: number,
    searchText?: string
  ): Promise<RoomStatusKpiDto[]> {
    this.logger.log('Room inventory KPIs request received', {
      programId,
      subProgramId,
      searchText
    });

    try {
      const kpis = await this.calculateRoomInventoryKpis(programId, subProgramId, searchText);
      
      this.logger.log('Room inventory KPIs retrieved successfully', {
        kpisCount: kpis.length,
        programId,
        subProgramId
      });
      
      return kpis;
    } catch (error) {
      this.logger.error('Error in room inventory KPIs retrieval service', error.stack, {
        programId,
        subProgramId,
        searchText
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_KPI_CALCULATION_FAILED, error);
    }
  }

  /**
   * Calculates KPIs for room inventory - returns array of status and count directly from repository
   * Maps all RoomStatus enum values with their counts from database
   * @param programId - Optional program ID filter for KPI calculation
   * @param subProgramId - Optional sub program ID filter for KPI calculation
   * @param searchText - Optional search text filter for KPI calculation
   * @returns Promise<RoomStatusKpiDto[]> Array of all room statuses with their counts
   */
  private async calculateRoomInventoryKpis(
    programId?: number,
    subProgramId?: number,
    searchText?: string
  ): Promise<RoomStatusKpiDto[]> {
    this.logger.debug('Calculating room inventory KPIs', {
      programId,
      subProgramId,
      searchText
    });

    try {
      // Get status breakdown from database (only returns statuses that exist)
      const statusBreakdown = await this.roomInventoryRepository.calculateStatusBreakdown(
        programId,
        subProgramId,
      );

      // Create a map of key to count from database results
      const statusCountMap = new Map<string, number>();
      let totalCount = 0;

      statusBreakdown.forEach((item) => {
        statusCountMap.set(item.status, Number(item.count));
        totalCount += Number(item.count);
      });

      // Define all possible statuses from RoomStatus enum with their labels
      const allStatuses: { status: string[]; label: string; key: string }[] = ROOM_STATUS_CONFIG.map(
        (statusInfo) => ({
          status: [...statusInfo.status],
          label: statusInfo.label,
          key: statusInfo.key,
        }),
      );

      // Get allocation status breakdown from program registration repository
      const allocationStatusBreakdown =
        await this.programRegistrationRepository.getRegistrationCountByAllocationStatus(
          programId,
          subProgramId,
        );
      statusCountMap.set('ALLOCATED_SEEKERS', allocationStatusBreakdown.allocated || 0);  
      
      // Map all statuses with their counts (0 if not in database) and include canClick property
      const kpis: RoomStatusKpiDto[] = allStatuses.map((statusInfo) => ({
        key: statusInfo.key,
        status: [...statusInfo.status],
        label: statusInfo.label,
        count:
          statusInfo.key === 'ALL'
            ? totalCount
            : statusCountMap.get(statusInfo.key) || 0,
      }));

      this.logger.debug('Room inventory KPIs calculated successfully', {
        kpisCount: kpis.length,
        totalCount: totalCount,
        programId,
        subProgramId,
      });

      return kpis;
    } catch (error) {
      this.logger.error('Error calculating room inventory KPIs', error.stack, {
        programId,
        subProgramId,
        searchText
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_KPI_CALCULATION_FAILED, error);
    }
  }

  /**
   * Get comprehensive filter configuration for room inventory
   * Provides all available filter options with sortability and filterability settings
   * Following the same pattern as registration service filters
   * 
   * @param filter - Optional filter object containing programId, subProgramId, venues, block, floor arrays
   * @returns Promise<FilterConfig[]> Complete filter configuration with all available options
   */
  async getFilterConfiguration(
    filter?: any
  ): Promise<FilterConfig[]> {
    this.logger.log('Room inventory filter configuration request received', {
      filter
    });

    try {
      // Extract filter properties - handle data processing in service
      const programId = filter?.programId;
      const subProgramId = filter?.subProgramId;
      const venueIds = Array.isArray(filter?.venues) ? filter.venues : [];
      const blockIds = Array.isArray(filter?.block) ? filter.block : [];
      const floorIds = Array.isArray(filter?.floor) ? filter.floor : [];

      // Clean and validate IDs - convert strings to numbers and remove duplicates
      const cleanedVenueIds: number[] = venueIds.length > 0 
        ? ([...new Set(venueIds.map((id: any) => typeof id === 'string' ? parseInt(id) : id).filter((id: any) => !isNaN(id)))] as number[])
        : [];
      
      const cleanedBlockIds: number[] = blockIds.length > 0 
        ? ([...new Set(blockIds.map((id: any) => typeof id === 'string' ? parseInt(id) : id).filter((id: any) => !isNaN(id)))] as number[])
        : [];
      
      const cleanedFloorIds: number[] = floorIds.length > 0 
        ? ([...new Set(floorIds.map((id: any) => typeof id === 'string' ? parseInt(id) : id).filter((id: any) => !isNaN(id)))] as number[])
        : [];

      this.logger.debug('Filter IDs cleaned and validated', {
        originalVenueIds: venueIds,
        cleanedVenueIds,
        originalBlockIds: blockIds,
        cleanedBlockIds,
        originalFloorIds: floorIds,
        cleanedFloorIds
      });

      // Get dynamic data for filter options from repository
      const [
        availableVenues,
        availableBlocks,
        availableFloors,
        availableRoomTypes,
        availableBedTypes,
        occupancyRanges
      ] = await Promise.all([
        this.roomInventoryRepository.getDistinctVenues(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds),
        this.roomInventoryRepository.getDistinctBlocks(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds),
        this.roomInventoryRepository.getDistinctFloors(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds),
        this.roomInventoryRepository.getDistinctRoomTypes(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds),
        this.roomInventoryRepository.getDistinctBedTypes(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds),
        this.roomInventoryRepository.getOccupancyRanges(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds)
      ]);

      // Get cities from lookup data using repository
      const citiesResponse = await this.lookupDataRepository.findActiveByCategory('CITY_NAME');
      const cities = citiesResponse.data;

      const rmUsers = await this.userRepository.getUsersByRoleKey(ROLE_KEYS.RELATIONAL_MANAGER);

      // Transform lookup data into filter options
      let availableLocations = cities
        .map((city) => ({
          value: city.label?.trim() || '',
          label: city.label?.trim() || '',
        }))
        .filter((location) => location.value !== '' && location.label !== '')
        .sort((a, b) => a.label.localeCompare(b.label));

      // Check if 'Other' city exists, if not, add it at the end
      const otherCity = availableLocations.find(
        (location) => location.label.toLowerCase() === 'other',
      );
      if (otherCity) {
        availableLocations = availableLocations.filter(
          (location) => location.label.toLowerCase() !== 'other',
        );
        availableLocations.push(otherCity);
      } else {
        availableLocations.push({ label: 'Other', value: 'Other' });
      }
  
      // Transform RM users to filter options format in service layer
      const availableRMs = rmUsers
        .map(user => ({
          value: user.id,
          label: user.orgUsrName || user.fullName || `${user.firstName || ''} ${user.lastName || ''}`.trim() || 'Unnamed RM'
        }))
        .filter(rm => rm.label !== 'Unnamed RM')
        .sort((a, b) => a.label.localeCompare(b.label));

      // Create filter configuration using constants with FilterConfig interface
      const filters: FilterConfig[] = ROOM_INVENTORY_FILTER_CONFIG_BASE.map(baseConfig => {
        const config: FilterConfig = { 
          ...baseConfig,
          options: []
        };
        
        // Assign appropriate options based on filter key
        switch (config.key) {
          case 'age':
            config.options = [...ROOM_INVENTORY_AGE_FILTER_OPTIONS];
            break;
          case 'gender':
            config.options = [...ROOM_INVENTORY_GENDER_FILTER_OPTIONS];
            break;
          case 'departureDatetime':
            config.options = [...ROOM_INVENTORY_DEPARTURE_DATE_FILTER_OPTIONS];
            break;
          case 'noOfHdbs':
            config.options = [...ROOM_INVENTORY_HDB_FILTER_OPTIONS];
            break;
          case 'preferredRoommate':
            config.options = [...ROOM_INVENTORY_ROOMMATE_PREFERENCE_OPTIONS];
            break;
          case 'rmContact':
            config.options = availableRMs;
            break;
          case 'city':
            config.options = availableLocations;
            break;
          case 'pairedStatus':
            config.options = [...ROOM_INVENTORY_PAIRED_STATUS_OPTIONS];
            break;
          case 'occupancy':
            config.options = [
              { label: 'Occupancy', min: 0, max: 10, eq: 15 }
            ];
            config.placeholder = 'Enter room occupancy';
            break;
          case 'floor':
            config.options = availableFloors;
            break;
          case 'returnTravelTerminal':
            config.options = [...ROOM_INVENTORY_RETURN_TERMINAL_OPTIONS];
            break;
          default:
            config.options = [];
            this.logger.warn('Unknown filter key found in configuration', { key: config.key });
        }
        
        return config;
      });

      // Sort filters by order
      const sortedFilters = filters.sort((a, b) => a.order - b.order);

      this.logger.log('Room inventory filter configuration generated successfully', {
        totalFilters: sortedFilters.length,
        appliedFilters: {
          programId,
          subProgramId,
          venueIds: cleanedVenueIds,
          blockIds: cleanedBlockIds,
          floorIds: cleanedFloorIds
        },
        dynamicOptionsLoaded: {
          venues: availableVenues.length,
          blocks: availableBlocks.length,
          floors: availableFloors.length,
          roomTypes: availableRoomTypes.length,
          bedTypes: availableBedTypes.length,
          occupancyRange: occupancyRanges.length > 0 ? `${occupancyRanges[0].minOccupancy}-${occupancyRanges[0].maxOccupancy}` : 'N/A'
        }
      });

      return sortedFilters;

    } catch (error) {
      this.logger.error('Error generating room inventory filter configuration', error.stack, {
        filter
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_FILTER_CONFIG_FAILED, error);
    }
  }

  /**
   * Update room inventory details with conditional status management
   * @param id - Room inventory identifier
   * @param updateRoomInventoryDto - Payload containing fields to update
   * @param updatedById - User performing the update
   * @returns Promise<ProgramRoomInventoryMap>
   */
  async updateRoomInventory(
    id: number,
    updateRoomInventoryDto: UpdateRoomInventoryDto,
    updatedById?: number,
  ): Promise<ProgramRoomInventoryMap> {
    this.logger.log('Room inventory update request received', {
      id,
      updateRoomInventoryDto,
      updatedById
    });

    try {
      // Fetch room inventory with allocations to check for existing allocations
      const roomInventory = await this.programRoomInventoryMapRepository.findByIdWithRelations(id);

      if (!roomInventory) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_INVENTORY_NOT_FOUND,
          null,
          null,
          id.toString(),
        );
      }

      // Check if trying to set room to reserved status when allocations exist
      const isSettingToReserved = updateRoomInventoryDto.isReserved === true;
      const hasActiveAllocations = roomInventory.roomAllocations && roomInventory.roomAllocations.length > 0;
      
      if (isSettingToReserved && hasActiveAllocations) {
        this.logger.error(
          'Cannot set room to reserved status when allocations exist',
          undefined,
          {
            roomInventoryId: id,
            activeAllocations: roomInventory.roomAllocations.length,
            allocationIds: roomInventory.roomAllocations.map(allocation => allocation.id)
          }
        );
        
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_INVENTORY_RESERVED_WITH_ALLOCATIONS,
          null,
          null,
          `Room has ${roomInventory.roomAllocations.length} active allocation(s). Remove allocations before setting to reserved.`,
        );
      }

      const totalCapacity = roomInventory.room?.occupancy || 0;
      const newRemainingOccupancy = updateRoomInventoryDto.remainingOccupancy ?? roomInventory.remainingOccupancy;

      if (newRemainingOccupancy < 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_INVENTORY_VALIDATION_FAILED,
          null,
          null,
          'Remaining occupancy cannot be negative',
        );
      }

      if (totalCapacity && newRemainingOccupancy > totalCapacity) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_INVENTORY_VALIDATION_FAILED,
          null,
          null,
          'Remaining occupancy cannot exceed total capacity',
          totalCapacity.toString(),
        );
      }

      const updatedIsReserved = updateRoomInventoryDto.isReserved ?? roomInventory.isReserved;
      const resolvedRoomStatus = this.resolveRoomStatus(
        updatedIsReserved,
        newRemainingOccupancy,
        totalCapacity,
        updateRoomInventoryDto.roomStatus,
        roomInventory.roomStatus
      );

      const updatedInventory = await this.roomInventoryRepository.save(
        new ProgramRoomInventoryMap({
          ...roomInventory,
          ...updateRoomInventoryDto,
          remainingOccupancy: newRemainingOccupancy,
          isReserved: updatedIsReserved,
          reservedFor: updatedIsReserved
            ? updateRoomInventoryDto.reservedFor ?? roomInventory.reservedFor
            : undefined,
          roomStatus: resolvedRoomStatus,
          occupiedStartsAt: updateRoomInventoryDto.occupiedStartsAt ?? roomInventory.occupiedStartsAt,
          occupiedEndsAt: updateRoomInventoryDto.occupiedEndsAt ?? roomInventory.occupiedEndsAt,
          updatedById: updatedById ?? roomInventory.updatedById ?? roomInventory.createdById,
        }),
      );

      this.logger.log('Room inventory updated successfully', {
        id,
        newRemainingOccupancy,
        resolvedRoomStatus,
        updatedIsReserved,
      });

      return updatedInventory;
    } catch (error) {
      this.logger.error('Error updating room inventory', error.stack, {
        id,
        updateRoomInventoryDto
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_UPDATE_FAILED, error);
    }
  }

  /**
   * Determine final room status based on reservation flag and occupancy values
   * Validates occupancy constraints for each status and maintains existing status if invalid
   * @param isReserved - Indicates if the room is reserved
   * @param remainingOccupancy - Remaining occupancy after update
   * @param totalCapacity - Total room capacity
   * @param requestedStatus - Optional status provided by client
   * @param currentStatus - Current status of the room (for fallback)
   * @returns RoomStatus
   */
  private resolveRoomStatus(
    isReserved: boolean,
    remainingOccupancy: number,
    totalCapacity: number,
    requestedStatus?: RoomStatus,
    currentStatus?: RoomStatus,
  ): RoomStatus {
    // Reserved rooms always have RESERVED status
    if (isReserved) {
      return RoomStatus.RESERVED;
    }

    // If a specific status is requested (and not RESERVED)
    if (requestedStatus && requestedStatus !== RoomStatus.RESERVED) {
      // Validate occupancy constraints for the requested status
      switch (requestedStatus) {
        case RoomStatus.AVAILABLE:
          // Available: remaining occupancy should equal total capacity
          if (totalCapacity > 0 && remainingOccupancy === totalCapacity) {
            return RoomStatus.AVAILABLE;
          }
          break;
          
        case RoomStatus.PARTIALLY_ALLOTTED:
          // Partially allotted: remaining occupancy must be less than total capacity and greater than 0
          if (totalCapacity > 0 && remainingOccupancy > 0 && remainingOccupancy < totalCapacity) {
            return RoomStatus.PARTIALLY_ALLOTTED;
          }
          break;
          
        case RoomStatus.ALLOTTED:
          // Allotted: remaining occupancy should be zero
          if (remainingOccupancy === 0) {
            return RoomStatus.ALLOTTED;
          }
          break;
      }
      
      // If requested status doesn't match occupancy constraints, return current status or auto-determine
      if (currentStatus) {
        return currentStatus;
      }
    }

    // Auto-determine status based on occupancy when no valid requested status
    if (totalCapacity > 0 && remainingOccupancy === totalCapacity) {
      return RoomStatus.AVAILABLE;
    }

    if (remainingOccupancy === 0) {
      return RoomStatus.ALLOTTED;
    }

    if (totalCapacity > 0 && remainingOccupancy > 0 && remainingOccupancy < totalCapacity) {
      return RoomStatus.PARTIALLY_ALLOTTED;
    }

    // Fallback to current status if available, otherwise AVAILABLE
    return currentStatus || RoomStatus.AVAILABLE;
  }
}