import {
  Controller,
  Body,
  Get,
  Param,
  Query,
  HttpStatus,
  HttpCode,
  Res,
  UseGuards,
  ValidationPipe,
  UsePipes,
  Logger,
  ParseIntPipe,
  Put,
  Req
} from '@nestjs/common';
import { Response } from 'express';
import { ApiTags, ApiOperation, ApiResponse, ApiQuery, ApiBearerAuth, ApiSecurity, ApiParam, ApiBody } from '@nestjs/swagger';
import { RoomInventoryService } from './room-inventory.service';
import { RoomInventoryListResponseDto, RoomStatusKpiDto } from './dto/room-inventory-response.dto';
import { UpdateRoomInventoryDto } from './dto/update-room-inventory.dto';
import { RoomStatus } from '../common/enums/room-status.enum';
import { ResponseService } from '../common/response-handling/response-handler';
import ErrorHandler from '../common/response-handling/error-handling';
import { RolesGuard } from '../common/guards/roles.guard';
import { Roles } from '../common/decorators/roles.decorator';
import { CombinedAuthGuard } from '../auth/combined-auth.guard';
import { handleControllerError } from '../common/utils/controller-response-handling';
import { SWAGGER_API_RESPONSE, paginationConstants } from '../common/constants/strings-constants';
import InifniBadRequestException from '../common/exceptions/infini-badrequest-exception';
import { ERROR_CODES } from '../common/constants/error-string-constants';

/**
 * Controller for managing room inventory operations
 * Provides endpoint for room inventory listing with comprehensive KPIs and filtering capabilities
 */
@ApiTags('room-inventory')
@Controller('room-inventory')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'mahatria', 'viewer', 'rm', 'operational_manager', 'rm_support', 'shoba')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class RoomInventoryController {
  private readonly logger = new Logger(RoomInventoryController.name);

  constructor(
    private readonly roomInventoryService: RoomInventoryService,
    private readonly responseService: ResponseService,
    private readonly errorHandler: ErrorHandler,
  ) {}

  /**
   * Updates a room inventory record and recalculates status conditionally
   * @param id - Room inventory identifier
   * @param updateRoomInventoryDto - Payload containing fields to update
   * @param req - Express request containing authenticated user
   * @param res - Express response
   */
  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: 'Update room inventory details' })
  @ApiParam({ name: 'id', type: Number, description: 'Room inventory identifier' })
  @ApiBody({ type: UpdateRoomInventoryDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.UNAUTHORIZED, description: SWAGGER_API_RESPONSE.UNAUTHORIZED })
  @ApiResponse({ status: HttpStatus.FORBIDDEN, description: SWAGGER_API_RESPONSE.FORBIDDEN })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async updateRoomInventory(
    @Param('id', ParseIntPipe) id: number,
    @Body() updateRoomInventoryDto: UpdateRoomInventoryDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log('Update room inventory request received', {
      id,
      updateRoomInventoryDto,
    });

    try {
      const updatedInventory = await this.roomInventoryService.updateRoomInventory(
        id,
        updateRoomInventoryDto,
        req.user?.id,
      );

      await this.responseService.success(
        res,
        'Room inventory updated successfully',
        updatedInventory,
        HttpStatus.OK,
      );
    } catch (error) {
      this.logger.error('Failed to update room inventory', {
        error: error.message,
        stack: error.stack,
        requestParams: { id, updateRoomInventoryDto },
      });

      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves all room inventory items with complete relations and comprehensive KPIs
   * Returns paginated room inventory data with real-time KPI calculations
   * 
   * Features:
   * - Pagination support with configurable limit and offset
   * - Program and sub-program filtering
   * - Search capability by room number or room label
   * - Advanced filtering via JSON filter object
   * - Real-time KPI calculation including ALL status with total count
   * - Returns raw entity data with all nested relations
   * 
   * @param limit - Number of records per page (default: 10, max: 100)
   * @param offset - Offset for pagination (default: 0)
   * @param programId - Optional program ID filter to show rooms for specific program
   * @param subProgramId - Optional sub program ID filter to show rooms for specific sub program
   * @param search - Optional search text to filter by room number or room label
   * @param filters - Optional advanced filter options as URL-encoded JSON string
   * @param res - Express Response object
   * @returns Promise<void> Response with room inventory data, KPIs (including ALL status), and pagination metadata
   */
  @Get()
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ 
    summary: 'Get paginated room inventory with KPIs including ALL status',
    description: `Retrieves comprehensive room inventory data with the following features:
    
    **Data Returned:**
    - Paginated room inventory entities with all relations (room, floor, allocations, created/updated by users)
    - Real-time KPIs with room status breakdown including "ALL" status for total count
    - Total records count for pagination
    
    **Filtering Capabilities:**
    - Program-based filtering (programId)
    - Sub-program filtering (subProgramId)
    - Text search by room number or label (search)
    - Advanced JSON filters (roomStatus, isReserved, occupancyRange, venues, blocks, floors, roomType, bedType, gender, age)
    
    **KPI Calculation:**
    - ALL: Total count of all rooms
    - AVAILABLE: Rooms ready for allocation
    - PARTIALLY_ALLOTTED: Rooms with remaining capacity
    - ALLOTTED: Fully occupied rooms
    - RESERVED: Rooms reserved for specific purposes
    - Statuses with zero count are still included in response
    
    **Response Format:**
    Returns raw entity data directly from database with all nested relations intact`
  })
  @ApiQuery({ 
    name: 'limit', 
    type: Number, 
    required: false, 
    description: `${paginationConstants.LIMIT} (max: 100)`,
    example: 10
  })
  @ApiQuery({ 
    name: 'offset', 
    type: Number, 
    required: false, 
    description: paginationConstants.OFFSET,
    example: 0
  })
  @ApiQuery({ 
    name: 'programId', 
    type: Number, 
    required: false, 
    description: 'Filter rooms by specific program ID. KPIs will be calculated based on this program.',
    example: 1
  })
  @ApiQuery({ 
    name: 'subProgramId', 
    type: Number, 
    required: false, 
    description: 'Filter rooms by specific sub program ID. Requires programId to be effective.',
    example: 2
  })
  @ApiQuery({ 
    name: 'search', 
    type: String, 
    required: false, 
    description: 'Search by room number or room label (case-insensitive partial match). Applied to both paginated results.',
    example: '101'
  })
  @ApiQuery({ 
    name: 'filters', 
    type: String, 
    required: false, 
    description: `Advanced filter options as URL-encoded JSON string. 
    
    **Filter Parameter Format:**
    The \`filters\` parameter must be a URL-encoded JSON string. The JSON object can contain various filter criteria.
    
    **Example Filter Usage:**
    \`\`\`
    // JavaScript example - encoding filters
    const filters = {
      roomStatus: ["AVAILABLE", "PARTIALLY_ALLOTTED"],
      venues: [1],           // Array of venue IDs
      block: [3, 1],         // Array of block IDs (Conference Wing, Lakeside Block)
      floor: [2, 5, 8],      // Array of floor IDs (First Floor)
      roomType: ["standard", "suite"],
      bedType: ["queen", "king"],
      gender: ["male"],
      age: ["26-35", "36-45"],
      occupancy: 2
    };
    const encodedFilters = encodeURIComponent(JSON.stringify(filters));
    
    // API call
    GET /room-inventory?programId=123&limit=20&offset=0&filters=\${encodedFilters}
    \`\`\`
    
    **Supported Filter Fields:**
    - \`roomStatus\`: string[] - Filter by room status array (AVAILABLE, ALLOTTED, RESERVED, PARTIALLY_ALLOTTED)
    - \`isReserved\`: boolean - Filter by reservation status (true for reserved rooms, false for non-reserved)
    - \`venues\`: number[] | string[] - Filter by venue IDs (e.g., [1] or ["1"])
    - \`block\`: number[] | string[] - Filter by block IDs (e.g., [3, 1] or ["3", "1"])
    - \`floor\`: number[] | string[] - Filter by floor IDs (e.g., [2, 5, 8] or ["2", "5", "8"])
    - \`occupancy\`: number - Filter by exact room capacity
    - \`occupancyRange\`: object - Filter by room capacity range (legacy support)
      - \`min\`: number - Minimum occupancy capacity
      - \`max\`: number - Maximum occupancy capacity
    - \`roomType\`: string[] - Filter by room types (e.g., ["standard", "suite", "family"])
    - \`bedType\`: string[] - Filter by bed types (e.g., ["queen", "king", "twin"])
    - \`gender\`: string[] - Filter by gender preference for allocated rooms (e.g., ["male", "female"])
    - \`age\`: string[] - Filter by age range categories for allocated rooms (e.g., ["18-25", "26-35", "36-45", "46-55", "56+"])
    
    **Note:** The filter accepts simple arrays of IDs for venues, blocks, and floors. String IDs are automatically converted to numbers.
    
    **Query Parameter Format:**
    GET /room-inventory?programId=123&search=101&limit=20&offset=0&filters=<encoded_filters>`,
    example: '{"roomStatus":["AVAILABLE"],"venues":[1],"block":[1,2],"roomType":["standard"]}'
  })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Room inventory data retrieved successfully with KPIs including ALL status and filter dropdowns - returns raw entity data with all relations',
    type: RoomInventoryListResponseDto,
    schema: {
      example: {
        success: true,
        message: 'Room inventory data retrieved successfully',
        data: {
          data: [
            {
              id: 1,
              remainingOccupancy: 1,
              isReserved: false,
              roomStatus: 'AVAILABLE',
              occupiedStartsAt: null,
              occupiedEndsAt: null,
              remarks: null,
              reservedFor: null,
              createdAt: '2024-01-01T10:00:00Z',
              updatedAt: '2024-01-01T10:00:00Z',
              room: {
                id: 1,
                label: 'Deluxe Room A',
                roomNumber: '101',
                occupancy: 2,
                roomStatus: 'ACTIVE',
                roomType: 'DELUXE',
                bedType: 'DOUBLE',
                description: 'Comfortable deluxe room with AC',
                floor: {
                  id: 1,
                  label: 'Ground Floor',
                  floorNumber: 'G'
                }
              },
              createdBy: {
                id: 1,
                firstName: 'Admin',
                lastName: 'User'
              },
              updatedBy: null,
              roomAllocations: [
                {
                  id: 1,
                  checkInDate: '2024-01-15T10:00:00Z',
                  checkOutDate: '2024-01-20T12:00:00Z',
                  user: {
                    id: 1,
                    firstName: 'John',
                    lastName: 'Doe'
                  }
                }
              ]
            }
          ],
          kpis: [
            {
              status: 'ALL',
              label: 'All',
              count: 100
            },
            {
              status: 'AVAILABLE',
              label: 'Available',
              count: 45
            },
            {
              status: 'PARTIALLY_ALLOTTED',
              label: 'Partially allotted',
              count: 10
            },
            {
              status: 'ALLOTTED',
              label: 'Allotted',
              count: 30
            },
            {
              status: 'RESERVED',
              label: 'Reserved',
              count: 15
            }
          ],
          filters: {
            property: [
              { label: 'Main Campus', value: 'main_campus' },
              { label: 'Guest House', value: 'guest_house' },
              { label: 'Retreat Center', value: 'retreat_center' }
            ],
            block: [
              { label: 'Block A', value: 'block_a' },
              { label: 'Block B', value: 'block_b' },
              { label: 'Block C', value: 'block_c' }
            ]
          },
          totalRecords: 100,
          limit: 10,
          offset: 0
        }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.BAD_REQUEST, 
    description: SWAGGER_API_RESPONSE.BAD_REQUEST,
    schema: {
      example: {
        success: false,
        message: 'Invalid request parameters',
        errorCode: 'INVALID_PARAMETERS',
        details: 'Limit must be between 1 and 100'
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.UNAUTHORIZED, 
    description: SWAGGER_API_RESPONSE.UNAUTHORIZED,
    schema: {
      example: {
        success: false,
        message: 'Unauthorized access',
        errorCode: 'UNAUTHORIZED'
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.FORBIDDEN, 
    description: SWAGGER_API_RESPONSE.FORBIDDEN,
    schema: {
      example: {
        success: false,
        message: 'Insufficient permissions to access room inventory',
        errorCode: 'FORBIDDEN'
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.INTERNAL_SERVER_ERROR, 
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
    schema: {
      example: {
        success: false,
        message: 'Internal server error while retrieving room inventory',
        errorCode: 'INTERNAL_SERVER_ERROR'
      }
    }
  })
  async getAllRoomInventory(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Res() res: Response,
    @Query('programId') programId?: number,
    @Query('subProgramId') subProgramId?: number,
    @Query('search') search?: string,
    @Query('filters') filters?: string,
  ) {
    this.logger.log('Room inventory list request received', {
      limit,
      offset,
      programId,
      subProgramId,
      search,
      filters,
      timestamp: new Date().toISOString()
    });

    try {
      // Validate pagination parameters
      if (limit && (limit < 1 || limit > 100)) {
        this.logger.warn('Invalid limit parameter provided', { limit });
        return this.errorHandler.badRequest(
          res, 
          'Limit must be between 1 and 100'
        );
      }

      if (offset && offset < 0) {
        this.logger.warn('Invalid offset parameter provided', { offset });
        return this.errorHandler.badRequest(
          res, 
          'Offset must be greater than or equal to 0'
        );
      }

      // Decode and parse filters if provided
      let parsedFilter = {};
      if (filters) {
        try {
          const decodedFilter = decodeURIComponent(filters);
          parsedFilter = JSON.parse(decodedFilter);
          this.logger.log('Filter decoded and parsed successfully', { 
            originalFilter: filters,
            decodedFilter,
            parsedFilter
          });
        } catch (error) {
          this.logger.error('Failed to decode or parse filter', error.stack, {
            filters
          });
          throw new InifniBadRequestException(
            ERROR_CODES.PQ_VALIDATION_FAILED,
            null,
            null,
            'Invalid filter format - must be valid URL-encoded JSON'
          );
        }
      }

      // Fetch room inventory data with KPIs
      const result = await this.roomInventoryService.getAllRoomInventoryWithKpis(
        limit,
        offset,
        programId,
        subProgramId,
        search?.trim(),
        parsedFilter
      );

      this.logger.log('Room inventory data retrieved successfully', {
        totalRecords: result.total,
        currentPageRecords: result.data.length,
        filters: { programId, subProgramId, search, filters: parsedFilter },
        statusBreakdownCount: result.kpis.length
      });

      await this.responseService.success(
        res,
        'Room inventory data retrieved successfully',
        result,
        HttpStatus.OK
      );

    } catch (error) {
      this.logger.error('Failed to retrieve room inventory data', {
        error: error.message,
        stack: error.stack,
        requestParams: { limit, offset, programId, subProgramId, search, filters }
      });
      
      handleControllerError(res, error);
    }
  }

  /**
   * Get comprehensive filter configuration for room inventory
   * Provides all available filter options with sortability and filterability settings
   * 
   * @param programId - Optional program ID to get program-specific filter options
   * @param subProgramId - Optional sub program ID for more specific filtering
   * @param filters - Optional URL-encoded JSON string containing venue IDs, block IDs, floor IDs arrays
   * @param res - Express Response object
   * @returns Promise<any[]> Complete filter configuration with all available options
   */
  @Get('filter-config')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ 
    summary: 'Get room inventory filter configuration',
    description: `Retrieves comprehensive filter configuration for room inventory including:
    - Venue/location filters with all available venues
    - Block filters with building/block information
    - Floor filters with floor-wise breakdown
    - Occupancy filters with capacity ranges
    - Gender-based allocation filters
    - Age range filters for occupants
    - Room status filters (Available, Reserved, Maintenance, etc.)
    - Each filter includes sortability and filterability metadata
    
    **Query Parameters:**
    - \`programId\`: number (optional) - Program ID to get program-specific filter options
    - \`subProgramId\`: number (optional) - Sub program ID for more specific filtering
    - \`filters\`: string (optional) - URL-encoded JSON string for additional filters
    
    **Filter Parameter Format:**
    The \`filters\` parameter is a URL-encoded JSON string that can contain:
    - \`venues\`: number[] - Array of venue IDs to filter configuration
    - \`block\`: number[] - Array of block IDs to filter configuration
    - \`floor\`: number[] - Array of floor IDs to filter configuration
    
    **Example:**
    \`\`\`
    const filters = {
      venues: [1, 2],
      block: [3, 4],
      floor: [5, 6]
    };
    const encodedFilters = encodeURIComponent(JSON.stringify(filters));
    GET /room-inventory/filter-config?programId=1&subProgramId=2&filters=\${encodedFilters}
    \`\`\``
  })
  @ApiQuery({ 
    name: 'programId', 
    type: Number, 
    required: false, 
    description: 'Filter configuration by specific program ID',
    example: 1
  })
  @ApiQuery({ 
    name: 'subProgramId', 
    type: Number, 
    required: false, 
    description: 'Filter configuration by specific sub program ID',
    example: 2
  })
  @ApiQuery({ 
    name: 'filters', 
    type: String, 
    required: false, 
    description: 'URL-encoded JSON string containing venues, block, and floor arrays for filtering',
    example: '{"venues":[1],"block":[3],"floor":[5]}'
  })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Filter configuration retrieved successfully',
    schema: {
      example: {
        success: true,
        message: 'Room inventory filter configuration retrieved successfully',
        data: [
          {
            key: 'venues',
            label: 'Select Venue',
            type: 'checkbox',
            sortable: true,
            filterable: true,
            order: 1,
            options: [
              { label: 'Main Campus', value: [1, 2] },
              { label: 'Guest House', value: [3] }
            ]
          },
          {
            key: 'block',
            label: 'Select Block',
            type: 'checkbox',
            sortable: true,
            filterable: true,
            order: 2,
            options: [
              { label: 'Block A', value: [1, 2] },
              { label: 'Block B', value: [3] }
            ]
          },
          {
            key: 'floor',
            label: 'Floor',
            type: 'checkbox',
            sortable: true,
            filterable: true,
            order: 3,
            options: [
              { label: 'Ground Floor', value: [1] },
              { label: '1st Floor', value: [2] }
            ]
          }
        ]
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.BAD_REQUEST, 
    description: SWAGGER_API_RESPONSE.BAD_REQUEST
  })
  @ApiResponse({ 
    status: HttpStatus.UNAUTHORIZED, 
    description: SWAGGER_API_RESPONSE.UNAUTHORIZED
  })
  @ApiResponse({ 
    status: HttpStatus.FORBIDDEN, 
    description: SWAGGER_API_RESPONSE.FORBIDDEN
  })
  @ApiResponse({ 
    status: HttpStatus.INTERNAL_SERVER_ERROR, 
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR
  })
  async getRoomInventoryFilterConfiguration(
    @Res() res: Response,
    @Query('programId') programId?: number,
    @Query('subProgramId') subProgramId?: number,
    @Query('filters') filters?: string,
  ) {
    this.logger.log('Room inventory filter configuration request received', {
      programId,
      subProgramId,
      filters,
      timestamp: new Date().toISOString()
    });

    try {
      // Decode and parse filters if provided
      let parsedFilter = {};
      if (filters) {
        try {
          const decodedFilter = decodeURIComponent(filters);
          parsedFilter = JSON.parse(decodedFilter);
          this.logger.log('Filter decoded and parsed successfully', { 
            originalFilter: filters,
            decodedFilter,
            parsedFilter
          });
        } catch (error) {
          this.logger.error('Failed to decode or parse filter', error.stack, {
            filters
          });
          throw new InifniBadRequestException(
            ERROR_CODES.PQ_VALIDATION_FAILED,
            null,
            null,
            'Invalid filter format - must be valid URL-encoded JSON'
          );
        }
      }

      // Merge programId and subProgramId with parsed filters
      const mergedFilter = {
        ...parsedFilter,
        programId,
        subProgramId
      };

      // Get filter configuration from service
      const filterConfig = await this.roomInventoryService.getFilterConfiguration(
        mergedFilter
      );

      this.logger.log('Room inventory filter configuration retrieved successfully', {
        filtersCount: filterConfig.length,
        programId,
        subProgramId,
        appliedFilters: mergedFilter
      });

      await this.responseService.success(
        res,
        'Room inventory filter configuration retrieved successfully',
        filterConfig,
        HttpStatus.OK
      );

    } catch (error) {
      this.logger.error('Failed to retrieve room inventory filter configuration', {
        error: error.message,
        stack: error.stack,
        requestParams: { programId, subProgramId, filters }
      });
      
      handleControllerError(res, error);
    }
  }
}