import { ApiPropertyOptional } from '@nestjs/swagger';
import { Type } from 'class-transformer';
import {
  IsBoolean,
  IsDate,
  IsEnum,
  IsInt,
  IsOptional,
  IsString,
  MaxLength,
  Min,
  ValidateIf
} from 'class-validator';
import { RoomStatus } from '../../common/enums/room-status.enum';

export class UpdateRoomInventoryDto {
  @ApiPropertyOptional({ description: 'Updated remaining occupancy for the room', minimum: 0 })
  @IsOptional()
  @Type(() => Number)
  @IsInt()
  @Min(0)
  remainingOccupancy?: number;

  @ApiPropertyOptional({ description: 'Flag to mark the room as reserved' })
  @IsOptional()
  @IsBoolean()
  isReserved?: boolean;

  @ApiPropertyOptional({ description: 'Details on why the room is reserved' })
  @IsOptional()
  @ValidateIf((dto) => dto.isReserved === true)
  @IsString()
  @MaxLength(255)
  reservedFor?: string;

  @ApiPropertyOptional({ enum: RoomStatus, description: 'Updated room status' })
  @IsOptional()
  @IsEnum(RoomStatus)
  roomStatus?: RoomStatus;

  @ApiPropertyOptional({ description: 'When the room occupancy starts' })
  @IsOptional()
  @Type(() => Date)
  @IsDate()
  occupiedStartsAt?: Date;

  @ApiPropertyOptional({ description: 'When the room occupancy ends' })
  @IsOptional()
  @Type(() => Date)
  @IsDate()
  occupiedEndsAt?: Date;

  @ApiPropertyOptional({ description: 'Additional remarks for the room' })
  @IsOptional()
  @IsString()
  remarks?: string;
}
