import { Injectable } from '@nestjs/common';
import { RoomAllocationRepository } from './room-allocation.repository';
import { CreateRoomAllocationDto } from './dto/create-room-allocation.dto';
import { UpdateRoomAllocationDto } from './dto/update-room-allocation.dto';
import { BulkRoomAllocationDto, BulkRoomAllocationDeleteDto, GetRegistrationsByProgramDto } from './dto/bulk-room-allocation.dto';
import { RoomAllocation } from 'src/common/entities/room-allocation.entity';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { User, Program, ProgramRoomInventoryMap } from 'src/common/entities';
import { ProgramRoomInventoryMapService } from '../room-inventory/program-room-inventory-map.service';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, In, IsNull } from 'typeorm';
import { ProgramRegistration } from 'src/common/entities/program-registration.entity';
import { TransferRoomAllocationDto } from './dto/transfer-room-allocation.dto';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { ProgramRegistrationRepository } from '../program-registration/program-registration.repository';
import { UserRepository } from 'src/user/user.repository';
import { ROLE_KEYS } from 'src/common/constants/strings-constants';
import { LookupDataRepository } from 'src/lookup-data/lookup-data.repository';
import { FilterConfig, RoomAllocationFilterConfig } from 'src/common/interfaces/interface';
import { 
  ROOM_ALLOCATION_AGE_FILTER_OPTIONS,
  ROOM_ALLOCATION_GENDER_FILTER_OPTIONS,
  ROOM_ALLOCATION_HDB_FILTER_OPTIONS,
  ROOM_ALLOCATION_ROOMMATE_PREFERENCE_OPTIONS,
  ROOM_ALLOCATION_PAIRED_STATUS_OPTIONS,
  ROOM_ALLOCATION_DEPARTURE_DATE_FILTER_OPTIONS,
  ROOM_ALLOCATION_RETURN_TERMINAL_OPTIONS,
  ROOM_ALLOCATION_FILTER_CONFIG_BASE
} from 'src/common/constants/constants';
import { GenderEnum } from 'src/common/enum/gender.enum';

/**
 * Service class for managing room allocation business logic
 * Handles validation, business rules, and orchestrates repository operations
 */
@Injectable()
export class RoomAllocationService {
  constructor(
    private readonly repository: RoomAllocationRepository,
    private readonly logger: AppLoggerService,
    private readonly roomInventoryService: ProgramRoomInventoryMapService,
    private readonly programRegistrationRepository: ProgramRegistrationRepository,
    private readonly userRepository: UserRepository,
    private readonly lookupDataRepository: LookupDataRepository,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepository: Repository<ProgramRegistration>,
    @InjectRepository(Program)
    private readonly programRepository: Repository<Program>,
    @InjectRepository(ProgramRoomInventoryMap)
    private readonly programRoomInventoryMapRepository: Repository<ProgramRoomInventoryMap>,
  ) {}

  /**
   * Create a new room allocation with business validation
   * @param dto - Room allocation creation data
   * @param user - User creating the allocation
   * @returns Created room allocation with relations
   */
  async create(dto: CreateRoomAllocationDto, user: User): Promise<RoomAllocation> {
    this.logger.log('Room allocation creation request received', { dto, userId: user.id });
    
    try {
      // Validate user permissions and data
      await this.validateCreateRequest(dto, user);
      
      // Create the room allocation
      const roomAllocation = await this.repository.create(dto, user.id);
      
      this.logger.log('Room allocation created successfully', { 
        id: roomAllocation.id, 
        registrationId: dto.registrationId,
        userId: user.id 
      });
      
      return roomAllocation;
    } catch (error) {
      this.logger.error('Error in room allocation creation service', error.stack, { dto, userId: user.id });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_CREATE_FAILED, error);
    }
  }

  /**
   * Retrieve all room allocations with pagination and filtering
   * @param limit - Number of records per page (default: 10)
   * @param offset - Offset for pagination (default: 0)
   * @param searchText - Search text for filtering
   * @param filters - Additional filters
   * @returns Paginated list of room allocations
   */
  async findAll(
    limit: number = 10,
    offset: number = 0,
    searchText: string = '',
    filters: Record<string, any> = {}
  ) {
    this.logger.log('Getting room allocations list', { limit, offset, searchText, filters });
    
    try {
      const result = await this.repository.findAll(limit, offset, searchText, filters);
      
      this.logger.log('Room allocations retrieved successfully', { 
        count: result.data.length, 
        total: result.pagination.totalRecords 
      });
      
      return result;
    } catch (error) {
      this.logger.error('Error in room allocations retrieval service', error.stack);
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_GET_FAILED, error);
    }
  }

  /**
   * Retrieve a room allocation by ID
   * @param id - Room allocation ID
   * @returns Room allocation with relations
   */
  async findOne(id: number): Promise<RoomAllocation> {
    this.logger.log('Getting room allocation by ID', { id });
    
    try {
      const roomAllocation = await this.repository.findById(id);
      
      if (!roomAllocation) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
          null,
          null,
          id.toString()
        );
      }
      
      this.logger.log('Room allocation retrieved successfully', { id });
      return roomAllocation;
    } catch (error) {
      this.logger.error('Error in room allocation retrieval service', error.stack, { id });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Update a room allocation with business validation
   * @param id - Room allocation ID
   * @param dto - Update data
   * @param user - User updating the allocation
   * @returns Updated room allocation
   */
  async update(id: number, dto: UpdateRoomAllocationDto, user: User): Promise<RoomAllocation> {
    this.logger.log('Room allocation update request received', { id, dto, userId: user.id });
    
    try {
      // Validate update request
      await this.validateUpdateRequest(id, dto, user);
      
      // Update the room allocation
      const updatedRoomAllocation = await this.repository.update(id, dto, user.id);
      
      this.logger.log('Room allocation updated successfully', { 
        id, 
        userId: user.id 
      });
      
      return updatedRoomAllocation;
    } catch (error) {
      this.logger.error('Error in room allocation update service', error.stack, { id, dto, userId: user.id });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_UPDATE_FAILED, error);
    }
  }

  /**
   * Delete a room allocation (soft delete)
   * @param id - Room allocation ID
   * @param user - User deleting the allocation
   * @returns Deletion result
   */
  async remove(id: number, user: User): Promise<{ id: number }> {
    this.logger.log('Room allocation deletion request received', { id, userId: user.id });
    
    try {
      // Get the existing allocation to determine which room inventory to update
      const existingAllocation = await this.repository.findById(id);
      
      // Validate deletion request
      await this.validateDeleteRequest(id, user);
      
      // Delete the room allocation
      const result = await this.repository.remove(id, user.id);
      
      // Update room occupancy and status after deallocation
      if (existingAllocation && existingAllocation.programRoomInventoryMapId) {
        await this.roomInventoryService.updateRoomOccupancyAfterDeallocation(
          existingAllocation.programRoomInventoryMapId,
          1 // Only one allocation is being removed
        );
        
        this.logger.log('Room occupancy updated after allocation removal', {
          roomInventoryId: existingAllocation.programRoomInventoryMapId,
          deallocatedCount: 1
        });
      }
      
      this.logger.log('Room allocation deleted successfully', { id, userId: user.id });
      return result;
    } catch (error) {
      this.logger.error('Error in room allocation deletion service', error.stack, { id, userId: user.id });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_DELETE_FAILED, error);
    }
  }

  /**
   * Get room allocations by program room inventory map ID
   * @param programRoomInventoryMapId - Program room inventory map ID
   * @returns Array of room allocations for the specified room
   */
  async findByProgramRoomInventoryMapId(programRoomInventoryMapId: number): Promise<RoomAllocation[]> {
    this.logger.log('Getting room allocations by program room inventory map ID', { programRoomInventoryMapId });
    
    try {
      const allocations = await this.repository.findByProgramRoomInventoryMapId(programRoomInventoryMapId);
      
      this.logger.log('Room allocations retrieved by program room inventory map ID', { 
        programRoomInventoryMapId,
        count: allocations.length 
      });
      
      return allocations;
    } catch (error) {
      this.logger.error('Error getting room allocations by program room inventory map ID', error.stack, { programRoomInventoryMapId });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FIND_BY_ROOM_INVENTORY_FAILED, error);
    }
  }

  /**
   * Get room allocation by registration ID
   * @param registrationId - Registration ID
   * @returns Room allocation for the specified registration
   */
  async findByRegistrationId(registrationId: number): Promise<RoomAllocation | null> {
    this.logger.log('Getting room allocation by registration ID', { registrationId });
    
    try {
      const allocation = await this.repository.findByRegistrationId(registrationId);
      
      this.logger.log('Room allocation retrieved by registration ID', { 
        registrationId,
        found: !!allocation 
      });
      
      return allocation;
    } catch (error) {
      this.logger.error('Error getting room allocation by registration ID', error.stack, { registrationId });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FIND_BY_REGISTRATION_ID_FAILED, error);
    }
  }

  /**
   * Get room occupancy statistics
   * @param programRoomInventoryMapId - Optional filter by program room inventory map
   * @returns Room occupancy statistics
   */
  async getRoomOccupancyStats(programRoomInventoryMapId?: number) {
    this.logger.log('Getting room occupancy statistics', { programRoomInventoryMapId });
    
    try {
      const stats = await this.repository.getRoomOccupancyStats(programRoomInventoryMapId);
      
      this.logger.log('Room occupancy statistics retrieved successfully', { 
        programRoomInventoryMapId,
        stats 
      });
      
      return stats;
    } catch (error) {
      this.logger.error('Error getting room occupancy statistics', error.stack, { programRoomInventoryMapId });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_OCCUPANCY_STATS_FAILED, error);
    }
  }

  /**
   * Create bulk room allocations for multiple registrations
   * @param programId - Program ID from path parameter
   * @param dto - Bulk room allocation data with individual bed positions and remarks
   * @param user - User creating the allocations
   * @returns Array of created room allocations
   */
  async createBulkAllocation(
    programId: number,
    dto: BulkRoomAllocationDto,
    user: User,
  ): Promise<RoomAllocation[]> {
    this.logger.log('Bulk room allocation request received', { 
      programId, 
      dto, 
      userId: user.id 
    });

    try {
      // Validate the bulk allocation request
      await this.validateBulkAllocationRequest(programId, dto, user);
      console.log('Bulk allocation request validated successfully');
      // Create individual room allocations
      const createdAllocations: RoomAllocation[] = [];
      for (const registration of dto.registrations) {
        const allocationDto = {
          programRoomInventoryMapId: dto.roomInventoryId,
          registrationId: registration.registrationId,
          bedPosition: registration.bedPosition,
          remarks: registration.remarks || dto.globalRemarks,
          createdAt: new Date(),
          updatedAt: new Date(),
        };

        const allocation = await this.repository.create(allocationDto, user.id);
        console.log('Created allocation:', allocation);
        createdAllocations.push(allocation);
      }

      // Update room occupancy and status based on the new allocations
      const updatedRoomInventory = await this.roomInventoryService.updateRoomOccupancy(
        dto.roomInventoryId,
        dto.registrations.length,
      );

      this.logger.log('Bulk room allocation completed successfully', {
        programId,
        roomInventoryId: dto.roomInventoryId,
        allocationsCount: createdAllocations.length,
        newRoomStatus: updatedRoomInventory.roomStatus,
        remainingOccupancy: updatedRoomInventory.remainingOccupancy,
        userId: user.id,
      });

      return createdAllocations;
    } catch (error) {
      this.logger.error('Error in bulk room allocation service', error.stack, {
        programId,
        dto,
        userId: user.id,
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_BULK_CREATE_FAILED, error);
    }
  }

  /**
   * Delete bulk room allocations for multiple allocation IDs
   * @param dto - Bulk room allocation delete data with allocation IDs
   * @param user - User deleting the allocations
   * @returns Array of deleted allocation details with updated room information for affected rooms
   */
  async deleteBulkAllocation(
    dto: BulkRoomAllocationDeleteDto,
    user: User,
  ): Promise<{ 
    deletedAllocations: RoomAllocation[]; 
    deletedCount: number;
    updatedRoomsInfo: Array<{
      roomInventoryId: number;
      previousRemainingOccupancy: number;
      newRemainingOccupancy: number;
      previousRoomStatus: string;
      newRoomStatus: string;
    }>;
  }> {
    this.logger.log('Bulk room allocation delete request received', { 
      dto, 
      userId: user.id 
    });

    try {
      // Validate the bulk delete request
      const allocationsToDelete = await this.validateBulkDeleteRequest(dto, user);

      // Group allocations by room inventory ID to update each room separately
      const roomInventoryMap = new Map<number, RoomAllocation[]>();
      allocationsToDelete.forEach(allocation => {
        const roomId = allocation.programRoomInventoryMapId;
        if (!roomInventoryMap.has(roomId)) {
          roomInventoryMap.set(roomId, []);
        }
        const roomAllocations = roomInventoryMap.get(roomId);
        if (roomAllocations) {
          roomAllocations.push(allocation);
        }
      });

      // Get room status before deletion for all affected rooms
      const roomsBeforeDeletion = await Promise.all(
        Array.from(roomInventoryMap.keys()).map(async roomId => ({
          roomId,
          details: await this.roomInventoryService.getRoomOccupancyDetails(roomId)
        }))
      );

      // Perform bulk soft delete
      const deleteResult = await this.repository.bulkRemove(dto.allocationIds, user.id);

      // Recalculate room occupancy and status for all affected rooms
      const updatedRoomsInfo = await Promise.all(
        Array.from(roomInventoryMap.keys()).map(async roomId => {
          const roomBefore = roomsBeforeDeletion.find(r => r.roomId === roomId);
          const updatedRoom = await this.roomInventoryService.recalculateRoomStatus(roomId);
          
          return {
            roomInventoryId: roomId,
            previousRemainingOccupancy: roomBefore?.details.remainingOccupancy || 0,
            newRemainingOccupancy: updatedRoom.remainingOccupancy,
            previousRoomStatus: roomBefore?.details.roomStatus || 'Unknown',
            newRoomStatus: updatedRoom.roomStatus,
          };
        })
      );

      this.logger.log('Bulk room allocation deletion completed successfully', {
        deletedCount: deleteResult.count,
        deletedIds: deleteResult.deletedIds,
        affectedRooms: updatedRoomsInfo.length,
        roomsInfo: updatedRoomsInfo,
        userId: user.id,
      });

      return {
        deletedAllocations: allocationsToDelete,
        deletedCount: deleteResult.count,
        updatedRoomsInfo,
      };
    } catch (error) {
      this.logger.error('Error in bulk room allocation deletion service', error.stack, {
        dto,
        userId: user.id,
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_BULK_DELETE_FAILED, error);
    }
  }

  /**
   * Transfer room allocation to a new room or update bed position in current room
   * @param allocationId - Allocation ID from path parameter
   * @param dto - Transfer data containing new inventory ID (optional) and bed position
   * @param user - User performing the transfer
   * @returns Updated room allocation with new room/bed position
   */
  async transferRoomAllocation(
    allocationId: number,
    dto: TransferRoomAllocationDto,
    user: User,
  ): Promise<RoomAllocation> {
    this.logger.log('Room allocation transfer request received', { 
      allocationId,
      dto, 
      userId: user.id 
    });

    try {
      // Validate the transfer request
      const { allocation } = await this.validateTransferRequest(
        allocationId,
        dto, 
        user
      );

      // Store original values for room occupancy updates
      const originalRoomId = allocation.programRoomInventoryMapId;
      const isRoomTransfer = dto.newInventoryId && dto.newInventoryId !== originalRoomId;

      // Update the allocation
      const updateData: Partial<RoomAllocation> = {
        bedPosition: dto.bedPosition,
        remarks: dto.remarks || allocation.remarks,
      };

      // If transferring to a different room, update the room inventory mapping
      if (isRoomTransfer) {
        if (dto.newInventoryId !== null) {
          updateData.programRoomInventoryMapId = dto.newInventoryId;
        }
      }

      const updatedAllocation = await this.repository.update(allocation.id, updateData, user.id);

      // Update room occupancy for both rooms if it's a room transfer
      if (isRoomTransfer) {
        // Decrease occupancy in the original room
        await this.roomInventoryService.updateRoomOccupancyAfterDeallocation(
          originalRoomId,
          1
        );

        // Increase occupancy in the new room
        await this.roomInventoryService.updateRoomOccupancy(
          dto.newInventoryId as number,
          1
        );

        this.logger.log('Room occupancy updated after transfer', {
          fromRoomId: originalRoomId,
          toRoomId: dto.newInventoryId,
          allocationId: allocation.id,
          userId: user.id,
        });
      }

      this.logger.log('Room allocation transfer completed successfully', {
        allocationId: allocation.id,
        fromRoomId: originalRoomId,
        toRoomId: dto.newInventoryId || originalRoomId,
        newBedPosition: dto.bedPosition,
        isRoomTransfer,
        userId: user.id,
      });

      return updatedAllocation;
    } catch (error) {
      this.logger.error('Error in room allocation transfer service', error.stack, {
        allocationId,
        dto,
        userId: user.id,
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_TRANSFER_FAILED, error);
    }
  }

  /**
   * Transfer multiple room allocations to new rooms or update bed positions
   * @param allocations - Array of allocation transfer data
   * @param user - User performing the transfers
   * @returns Array of updated room allocations
   */
  async transferRoomAllocations(
    data: {
      programId: number;
      subProgramId: number;
      updateAllocations: Array<{ allocationId: number; newRoomInventoryId: number; bedPosition: number }>;
    },
    user: User,
  ): Promise<RoomAllocation[]> {
    this.logger.log('Processing bulk room allocation transfers', { allocations : data.updateAllocations, userId: user.id });

    // Add validation to ensure all three values are mandatory for each object in the updateAllocations array
    for (const allocation of data.updateAllocations) {
      if (!allocation.allocationId || !allocation.newRoomInventoryId || !allocation.bedPosition) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_ALLOCATION_DATA,
          null,
          null,
          'Each allocation must have allocationId, newRoomInventoryId, and bedPosition.'
        );
      }
    }

    const updatedAllocations: RoomAllocation[] = [];
    const allocations = data.updateAllocations;
    for (const allocation of allocations) {
      const existingAllocation = await this.repository.findById(allocation.allocationId);

      if (!existingAllocation) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
          null,
          null,
          allocation?.allocationId?.toString()
        );
      }

      const currentRoomId = existingAllocation.programRoomInventoryMapId;
      const isRoomTransfer = allocation.newRoomInventoryId !== currentRoomId;

      if (isRoomTransfer) {
        const newRoomInventory = await this.roomInventoryService.validateRoomInventoryMapping(
          allocation.newRoomInventoryId,
          data.programId,
          data.subProgramId,
        );

        if (newRoomInventory.remainingOccupancy < 1) {
          throw new InifniBadRequestException(
            ERROR_CODES.INSUFFICIENT_ROOM_CAPACITY,
            null,
            null,
            allocation.newRoomInventoryId.toString()
          );
        }

        const occupiedPositions = (await this.findByProgramRoomInventoryMapId(allocation.newRoomInventoryId))
          .map(a => a.bedPosition);

        if (occupiedPositions.includes(allocation.bedPosition)) {
          throw new InifniBadRequestException(
            ERROR_CODES.BED_POSITION_ALREADY_OCCUPIED,
            null,
            null,
            allocation.bedPosition.toString()
          );
        }
      } else {
        const occupiedPositions = (await this.findByProgramRoomInventoryMapId(currentRoomId))
          .filter(a => a.id !== allocation.allocationId)
          .map(a => a.bedPosition);

        if (occupiedPositions.includes(allocation.bedPosition)) {
          throw new InifniBadRequestException(
            ERROR_CODES.BED_POSITION_ALREADY_OCCUPIED,
            null,
            null,
            allocation.bedPosition.toString()
          );
        }
      }

      const updatedAllocation = await this.repository.update(allocation.allocationId, {
        programRoomInventoryMapId: allocation.newRoomInventoryId,
        bedPosition: allocation.bedPosition,
      }, user.id);

      updatedAllocations.push(updatedAllocation);

      if (isRoomTransfer) {
        await this.roomInventoryService.updateRoomOccupancyAfterDeallocation(currentRoomId, 1);
        await this.roomInventoryService.updateRoomOccupancy(allocation.newRoomInventoryId, 1);
      }
    }

    this.logger.log('Bulk room allocation transfers completed successfully', { updatedCount: updatedAllocations.length });
    return updatedAllocations;
  }

  /**
   * Get registrations by program ID and optional sub-program ID with pagination and filters
   * Retrieves program registrations filtered by program, sub-program, and custom filter criteria
   * Automatically filters for approved registrations only and non-allocated registrations
   * @param dto - Request data containing programId, optional subProgramId, limit, and offset
   * @param searchText - Optional search text for filtering by name, email, or mobile number
   * @param filters - Parsed filter object containing custom filter criteria
   * @returns Promise<{ data: any[]; total: number; pagination: any }> - Paginated registrations data
   */
  async getRegistrationsByProgram(
    dto: GetRegistrationsByProgramDto,
    searchText?: string,
    filters: Record<string, any> = {},
    sort?: Array<Record<string, 'asc' | 'desc'>>,
  ): Promise<{ data: any[]; total: number; pagination: any }> {
    this.logger.log('Fetching registrations by program with filters, search, and sort', { 
      dto, 
      searchText, 
      filters,
      sort
    });

    try {
      // Process and normalize filters before passing to repository
      const processedFilters = this.processFilters(filters);

      // Automatically filter for approved registrations only
      processedFilters.approvalStatus = ApprovalStatusEnum.APPROVED;
      
      // Filter for non-allocated registrations only (those without room allocations)
      processedFilters.hasRoomAllocation = false;
      const searchFields = ['registration.fullName','registration.city','registration.otherCityName']
      const result = await this.programRegistrationRepository.getRegistrationsByProgram(
        dto.programId,
        dto.limit || 10,
        dto.offset || 0,
        searchText,
        processedFilters,
        sort,
        searchFields,
        true
      );

      this.logger.log('Registrations retrieved successfully', {
        programId: dto.programId,
        search: searchText,
        total: result.total,
        count: result.data.length,
        limit: dto.limit,
        offset: dto.offset,
        appliedFilters: processedFilters,
      });

      return result;
    } catch (error) {
      this.logger.error('Error fetching registrations by program', error.stack, { 
        dto, 
        searchText, 
        filters,
        sort 
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_GET_FAILED, error);
    }
  }

  /**
   * Get comprehensive filter configuration for room allocation
   * Provides age and location filter options based on approved registrations
   * 
   * @param programId - Program ID to get filter options for
   * @param subProgramId - Optional sub program ID for more specific filtering
   * @param filters - Optional filter object containing additional criteria
   * @returns Promise<RoomAllocationFilterConfig> Complete filter configuration with age and location options
   */
  async getFilterConfiguration(
    programId: number,
    subProgramId?: number,
    filters: Record<string, any> = {},
  ): Promise<RoomAllocationFilterConfig> {
    this.logger.log('Room allocation filter configuration request received', {
      programId,
      subProgramId,
      filters
    });

    try {
      // Validate program exists
      await this.validateProgramAndSubProgramExistence(programId, subProgramId);
      filters.roomAllocation = false;

      // Get cities from lookup data using repository
      const citiesResponse = await this.lookupDataRepository.findActiveByCategory('CITY_NAME');
      const cities = citiesResponse.data;
      
      // Transform lookup data into filter options
      let availableLocations = cities
        .map((city) => ({
          value: city.label?.trim() || '',
          label: city.label?.trim() || '',
        }))
        .filter((location) => location.value !== '' && location.label !== '')
        .sort((a, b) => a.label.localeCompare(b.label));

      // Check if 'Other' city exists, if not, add it at the end
      const otherCity = availableLocations.find(
        (location) => location.label.toLowerCase() === 'other',
      );
      if (otherCity) {
        availableLocations = availableLocations.filter(
          (location) => location.label.toLowerCase() !== 'other',
        );
        availableLocations.push(otherCity);
      } else {
        availableLocations.push({ label: 'Other', value: 'Other' });
      }

      const rmFilterOptions = await this.userRepository.getUsersByRoleKey(ROLE_KEYS.RELATIONAL_MANAGER);
      const availableRMs = rmFilterOptions
        .map(user => ({
          value: user.id,
          label: user.orgUsrName || user.fullName || `${user.firstName || ''} ${user.lastName || ''}`.trim() || 'Unnamed RM'
        }))
        .filter(rm => rm.label !== 'Unnamed RM')
        .sort((a, b) => a.label.localeCompare(b.label));

      // Create filter configuration using constants with simple typing
      const filterConfig: FilterConfig[] = ROOM_ALLOCATION_FILTER_CONFIG_BASE.map(baseConfig => {
        const config: FilterConfig = { 
          ...baseConfig,
          options: []
        };
        
        // Assign appropriate options based on filter key
        switch (config.key) {
          case 'age':
            config.options = [...ROOM_ALLOCATION_AGE_FILTER_OPTIONS];
            break;
          case 'gender':
            config.options = [...ROOM_ALLOCATION_GENDER_FILTER_OPTIONS];
            break;
          case 'departureDatetime':
            config.options = [...ROOM_ALLOCATION_DEPARTURE_DATE_FILTER_OPTIONS];
            break;
          case 'noOfHdbs':
            config.options = [...ROOM_ALLOCATION_HDB_FILTER_OPTIONS];
            break;
          case 'preferredRoommate':
            config.options = [...ROOM_ALLOCATION_ROOMMATE_PREFERENCE_OPTIONS];
            break;
          case 'rmContact':
            config.options = availableRMs;
            break;
          case 'city':
            config.options = availableLocations;
            break;
          case 'pairedStatus':
            config.options = [...ROOM_ALLOCATION_PAIRED_STATUS_OPTIONS];
            break;
          case 'returnTravelTerminal':
            config.options = [...ROOM_ALLOCATION_RETURN_TERMINAL_OPTIONS];
            break;
          default:
            config.options = [];
            this.logger.warn('Unknown filter key found in configuration', { key: config.key });
        }
        
        return config;
      });

      // Sort filters by order
      const sortedFilters = filterConfig.sort((a, b) => a.order - b.order);

      this.logger.log('Room allocation filter configuration generated successfully', {
        totalFilters: sortedFilters.length,
        programId,
        subProgramId,
        appliedFilters: filters,
        dynamicOptionsLoaded: {
          locations: availableLocations.length,
          rms: availableRMs.length
        }
      });

      return sortedFilters;

    } catch (error) {
      this.logger.error('Error generating room allocation filter configuration', error.stack, {
        programId,
        subProgramId,
        filters
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FILTER_CONFIG_FAILED, error);
    }
  }

  /**
   * Fetch registrations with preferred roommate details
   * @param programId - Program ID to filter registrations
   * @param subProgramId - Optional sub-program ID for filtering
   * @param search - Optional search text to filter registrations by full name
   * @param limit - Number of records per page (default: 10)
   * @param offset - Offset for pagination (default: 0)
   * @returns Array of registrations with preferred roommate details
   */
  async getPreferredRoommateRegistrations(
    programId: number,
    subProgramId?: number,
    limit: number = 10,
    offset: number = 0,
    search?: string
  ): Promise<{ data: any[]; total: number; pagination: any }> {
    if (!programId) {
      throw new InifniBadRequestException(
        ERROR_CODES.PQ_VALIDATION_FAILED,
        null,
        null,
        'programId is required'
      );
    }

    const filters: Record<string, any> = {};
    if (subProgramId) {
      filters.allocatedProgramId = subProgramId;
    }

    const result = await this.programRegistrationRepository.getRegistrationsByProgram(
      programId,
      limit,
      offset,
      search,
      filters,
      [],
      ['registration.fullName']
    );

    return result;
  }

  /**
   * Process and normalize filter values for database queries
   * Handles special cases like JSON string arrays
   * @param filters - Raw filter object from request
   * @returns Processed filter object with normalized values
   */
  private processFilters(filters: Record<string, any>): Record<string, any> {
    const processedFilters = { ...filters };

    // Handle allocatedProgramId filter - convert JSON string to array
    if (processedFilters.allocatedProgramId) {
      let allocatedProgramIds = processedFilters.allocatedProgramId;
      
      // Handle case where allocatedProgramId is a JSON string like "[833]"
      if (typeof allocatedProgramIds === 'string') {
        try {
          allocatedProgramIds = JSON.parse(allocatedProgramIds);
          this.logger.debug('Parsed allocatedProgramId from JSON string', {
            original: processedFilters.allocatedProgramId,
            parsed: allocatedProgramIds,
          });
        } catch (error) {
          this.logger.warn('Failed to parse allocatedProgramId as JSON, treating as single value', {
            allocatedProgramId: processedFilters.allocatedProgramId,
            error: error.message,
          });
          allocatedProgramIds = [allocatedProgramIds];
        }
      }
      
      // Ensure it's an array
      if (!Array.isArray(allocatedProgramIds)) {
        allocatedProgramIds = [allocatedProgramIds];
      }
      
      processedFilters.allocatedProgramId = allocatedProgramIds;
    }    
    return processedFilters;
  }

  // ===== PRIVATE VALIDATION METHODS =====

  /**
   * Validate room allocation creation request
   * Checks business rules and constraints before creation
   * @param dto - Room allocation creation data
   * @param user - User creating the allocation
   */
  private async validateCreateRequest(dto: CreateRoomAllocationDto, user: User): Promise<void> {
    this.logger.debug('Validating room allocation create request', { dto, userId: user.id });
    
    try {
      // Validate required fields
      if (!dto.programRoomInventoryMapId) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_INVENTORY_NOT_FOUND,
          null,
          null,
          dto.programRoomInventoryMapId?.toString() || 'undefined'
        );
      }

      if (!dto.registrationId) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_NOT_FOUND,
          null,
          null,
          dto.registrationId?.toString() || 'undefined'
        );
      }

      // Validate bed position if provided
      if (dto.bedPosition && (dto.bedPosition < 1 || !Number.isInteger(dto.bedPosition))) {
        throw new InifniBadRequestException(
          ERROR_CODES.BED_POSITION_INVALID,
          null,
          null,
          dto.bedPosition.toString()
        );
      }

      this.logger.debug('Room allocation create request validation completed', { dto, userId: user.id });
    } catch (error) {
      this.logger.error('Room allocation create validation failed', error.stack, { dto, userId: user.id });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_VALIDATION_FAILED, error);
    }
  }

  /**
   * Validate room allocation update request
   * Checks business rules and constraints before update
   * @param id - Room allocation ID
   * @param dto - Update data
   * @param user - User updating the allocation
   */
  private async validateUpdateRequest(id: number, dto: UpdateRoomAllocationDto, user: User): Promise<void> {
    this.logger.debug('Validating room allocation update request', { id, dto, userId: user.id });
    
    try {
      // Check if room allocation exists
      const existingAllocation = await this.repository.findById(id);
      if (!existingAllocation) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
          null,
          null,
          id.toString()
        );
      }

      // Validate bed position if being updated
      if (dto.bedPosition && (dto.bedPosition < 1 || !Number.isInteger(dto.bedPosition))) {
        throw new InifniBadRequestException(
          ERROR_CODES.BED_POSITION_INVALID,
          null,
          null,
          dto.bedPosition.toString()
        );
      }

      this.logger.debug('Room allocation update request validation completed', { id, dto, userId: user.id });
    } catch (error) {
      this.logger.error('Room allocation update validation failed', error.stack, { id, dto, userId: user.id });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_VALIDATION_FAILED, error);
    }
  }

  /**
   * Validate room allocation deletion request
   * Checks business rules and constraints before deletion
   * @param id - Room allocation ID
   * @param user - User deleting the allocation
   */
  private async validateDeleteRequest(id: number, user: User): Promise<void> {
    this.logger.debug('Validating room allocation delete request', { id, userId: user.id });
    
    try {
      // Check if room allocation exists
      const existingAllocation = await this.repository.findById(id);
      if (!existingAllocation) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
          null,
          null,
          id.toString()
        );
      }

      this.logger.debug('Room allocation delete request validation completed', { id, userId: user.id });
    } catch (error) {
      this.logger.error('Room allocation delete validation failed', error.stack, { id, userId: user.id });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_VALIDATION_FAILED, error);
    }
  }

  /**
   * Validate bulk room allocation request
   * @param programId - Program ID from path
   * @param dto - Bulk allocation data
   * @param user - User creating the allocations
   */
  private async validateBulkAllocationRequest(
    programId: number,
    dto: BulkRoomAllocationDto,
    user: User,
  ): Promise<void> {
    this.logger.debug('Validating bulk room allocation request', {
      programId,
      dto,
      userId: user.id,
    });

    try {
      // 0. Validate program and sub-program existence
      await this.validateProgramAndSubProgramExistence(programId, dto.subProgramId);

      // 1. Validate array of registrations
      if (!Array.isArray(dto.registrations) || dto.registrations.length === 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_NOT_FOUND,
          null,
          null,
          'array'
        );
      }

      // Extract registration IDs for further validation
      const registrationIds = dto.registrations.map(r => r.registrationId);

      // Check for duplicate registration IDs
      const uniqueRegistrationIds = new Set(registrationIds);
      if (uniqueRegistrationIds.size !== registrationIds.length) {
        const duplicates = registrationIds.filter((id, index) => registrationIds.indexOf(id) !== index);
        throw new InifniBadRequestException(
          ERROR_CODES.DUPLICATE_REGISTRATION_IDS,
          null,
          null,
          duplicates.join(', ')
        );
      }

      // 2. Validate room inventory mapping and get room capacity
      const roomInventoryMap = await this.roomInventoryService.validateRoomInventoryMapping(
        dto.roomInventoryId,
        programId,
        dto.subProgramId,
      );

      // Get room capacity from the related room entity
      const roomCapacity = roomInventoryMap.room?.occupancy;
      if (!roomCapacity) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_CAPACITY_NOT_AVAILABLE,
          null,
          null,
          dto.roomInventoryId.toString()
        );
      }

      // 3. Check remaining room capacity using remainingOccupancy field
      if (dto.registrations.length > roomInventoryMap.remainingOccupancy) {
        throw new InifniBadRequestException(
          ERROR_CODES.INSUFFICIENT_ROOM_CAPACITY,
          null,
          null,
          roomInventoryMap.roomId.toString(),
          dto.registrations.length.toString(),
          roomInventoryMap.remainingOccupancy.toString()
        );
      }

      // 3.1. Check if room is reserved - do not allow allocation to reserved rooms
      if (roomInventoryMap.isReserved) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_ALLOCATION_RESERVED_ROOM,
          null,
          null,
          roomInventoryMap.roomId.toString(),
          roomInventoryMap.reservedFor || 'Unknown purpose'
        );
      }

      // 4. Check for duplicate bed positions
      const bedPositions = dto.registrations.map(r => r.bedPosition);
      const uniqueBedPositions = new Set(bedPositions);
      if (uniqueBedPositions.size !== bedPositions.length) {
        const duplicates = bedPositions.filter((pos, index) => bedPositions.indexOf(pos) !== index);
        throw new InifniBadRequestException(
          ERROR_CODES.DUPLICATE_BED_POSITIONS,
          null,
          null,
          duplicates.join(', ')
        );
      }

      // 5. Validate bed positions are within room capacity (1-based indexing)
      for (const registration of dto.registrations) {
        if (registration.bedPosition > roomCapacity || registration.bedPosition < 1) {
          throw new InifniBadRequestException(
            ERROR_CODES.BED_POSITION_OUT_OF_RANGE,
            null,
            null,
            registration.bedPosition.toString(),
            roomCapacity.toString()
          );
        }
      }

      // 6. Check for bed position conflicts with existing allocations
      const currentOccupancy = await this.findByProgramRoomInventoryMapId(dto.roomInventoryId);
      const occupiedBedPositions = currentOccupancy.map(allocation => allocation.bedPosition).filter(pos => pos !== null);
      
      for (const registration of dto.registrations) {
        if (occupiedBedPositions.includes(registration.bedPosition)) {
          throw new InifniBadRequestException(
            ERROR_CODES.BED_POSITION_ALREADY_OCCUPIED,
            null,
            null,
            registration.bedPosition.toString(),
            dto.roomInventoryId.toString()
          );
        }
      }

      // 7. Validate that all registration IDs exist and belong to valid registrations
      const registrations = await this.registrationRepository.find({
        where: { 
          id: In(registrationIds)
        },
        relations: ['allocatedProgram'],
      });

      this.logger.debug('Registration validation results', {
        requestedCount: registrationIds.length,
        foundCount: registrations.length,
        requestedIds: registrationIds,
        foundIds: registrations.map(r => r.id)
      });

      // Handle case when no registrations are found
      if (registrations.length === 0) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_NOT_FOUND,
          null,
          null,
          registrationIds.join(', ')
        );
      }

      // Handle case when some registrations are missing
      if (registrations.length !== registrationIds.length) {
        const foundIds = registrations.map(r => r.id);
        const missingIds = registrationIds.filter(id => !foundIds.includes(id));
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_NOT_FOUND,
          null,
          null,
          missingIds.join(', ')
        );
      }

      // 8. Validate program ID consistency
      for (const registration of registrations) {
        const allocatedProgramId = registration.allocatedProgramId;
        
        if (dto.subProgramId) {
          // If sub-program is provided, check if registration's allocated program matches sub-program
          if (allocatedProgramId !== dto.subProgramId) {
            throw new InifniBadRequestException(
              ERROR_CODES.ROOM_ALLOCATION_PROGRAM_MISMATCH,
              null,
              null,
              registration.id.toString()
            );
          }
        } else {
          // If no sub-program, check if registration's allocated program matches path program
          if (allocatedProgramId !== programId) {
            throw new InifniBadRequestException(
              ERROR_CODES.ROOM_ALLOCATION_PROGRAM_MISMATCH,
              null,
              null,
              registration.id.toString()
            );
          }
        }
      }

      // 9. Check if any registration is already allocated to a room
      const existingAllocations = await this.repository.findByRegistrationIds(registrationIds);
      if (existingAllocations.length > 0) {
        const allocatedRegistrationIds = existingAllocations.map(a => a.registrationId);
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_ALLOCATION_ALREADY_EXISTS,
          null,
          null,
          allocatedRegistrationIds.join(', ')
        );
      }

      this.logger.debug('Bulk room allocation request validation completed', {
        programId,
        dto,
        userId: user.id,
      });
    } catch (error) {
      this.logger.error('Bulk room allocation validation failed', error.stack, {
        programId,
        dto,
        userId: user.id,
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_VALIDATION_FAILED, error);
    }
  }

  /**
   * Validate that the program and sub-program (if provided) exist and are active
   * @param programId - Main program ID from the path parameter
   * @param subProgramId - Optional sub-program ID from the request body
   */
  private async validateProgramAndSubProgramExistence(
    programId: number,
    subProgramId?: number,
  ): Promise<void> {
    this.logger.debug('Validating program and sub-program existence', {
      programId,
      subProgramId,
    });

    // 1. Validate main program exists and is active
    const program = await this.programRepository.findOne({
      where: {
        id: programId,
        deletedAt: IsNull(),
        isActive: true,
      },
    });

    if (!program) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        programId.toString()
      );
    }

    this.logger.debug('Main program validated successfully', { programId, programName: program.name });

    // 2. Validate sub-program if provided
    if (subProgramId) {
      const subProgram = await this.programRepository.findOne({
        where: {
          id: subProgramId,
          deletedAt: IsNull(),
          isActive: true,
        },
      });

      if (!subProgram) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          subProgramId.toString()
        );
      }

      // 3. Validate relationship between main program and sub-program
      const isValidRelationship = await this.validateProgramRelationship(program, subProgram);
      
      if (!isValidRelationship) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_ALLOCATION_PROGRAM_MISMATCH,
          null,
          null,
          subProgramId.toString()
        );
      }

      this.logger.debug('Sub-program validated successfully', {
        subProgramId,
        subProgramName: subProgram.name,
      });
    }

    this.logger.debug('Program and sub-program validation completed successfully', {
      programId,
      subProgramId,
    });
  }

  /**
   * Validates the relationship between a main program and sub-program
   * @param mainProgram - The main program entity
   * @param subProgram - The sub-program entity to validate
   * @returns True if the relationship is valid, false otherwise
   */
  private async validateProgramRelationship(
    mainProgram: Program,
    subProgram: Program,
  ): Promise<boolean> {
    this.logger.debug('Validating program relationship', {
      mainProgramId: mainProgram.id,
      subProgramId: subProgram.id,
    });

    // Case 1: Sub-program is a direct child of the main program
    if (subProgram.primaryProgramId === mainProgram.id) {
      this.logger.debug('Sub-program is a direct child of main program');
      return true;
    }

    // Case 2: Both programs are in the same group and main program is the primary
    if (
      mainProgram.groupId &&
      subProgram.groupId &&
      mainProgram.groupId === subProgram.groupId &&
      mainProgram.isPrimaryProgram === true
    ) {
      this.logger.debug('Programs are in the same group with valid hierarchy');
      return true;
    }

    // Case 3: Both programs are in the same group (for grouped program scenarios)
    if (
      mainProgram.groupId &&
      subProgram.groupId &&
      mainProgram.groupId === subProgram.groupId
    ) {
      this.logger.debug('Programs are in the same group');
      return true;
    }

    this.logger.debug('No valid relationship found between programs');
    return false;
  }

  /**
   * Validate bulk room allocation deletion request
   * Validates allocation IDs exist and belong to the specified program/sub-program
   * @param dto - Bulk delete data with programId, allocation IDs, and optional subProgramId
   * @param user - User deleting the allocations
   * @returns Array of room allocations to be deleted
   */
  private async validateBulkDeleteRequest(
    dto: BulkRoomAllocationDeleteDto,
    user: User,
  ): Promise<RoomAllocation[]> {
    this.logger.debug('Validating bulk room allocation delete request', {
      dto,
      userId: user.id,
    });

    try {
      // 1. Validate array of allocation IDs
      if (!Array.isArray(dto.allocationIds) || dto.allocationIds.length === 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
          null,
          null,
          'array'
        );
      }

      // Check for duplicate allocation IDs
      const uniqueAllocationIds = new Set(dto.allocationIds);
      if (uniqueAllocationIds.size !== dto.allocationIds.length) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_ALLOCATION_DUPLICATE_IDS,
          null,
          null
        );
      }

      // 2. Validate program and sub-program existence
      await this.validateProgramAndSubProgramExistence(dto.programId, dto.subProgramId);

      // 3. Validate that all allocation IDs exist and are not already deleted
      const existingAllocations = await this.repository.findByAllocationIds(dto.allocationIds);

      if (existingAllocations.length === 0) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
          null,
          null,
          dto.allocationIds.join(', ')
        );
      }

      // Handle case when some allocations are missing
      if (existingAllocations.length !== dto.allocationIds.length) {
        const foundIds = existingAllocations.map(a => a.id);
        const missingIds = dto.allocationIds.filter(id => !foundIds.includes(id));
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
          null,
          null,
          missingIds.join(', ')
        );
      }

      // 4. Validate that allocations belong to the specified program or sub-program
      // Get all registration IDs from allocations
      const registrationIds = existingAllocations.map(a => a.registrationId);
      const registrations = await this.registrationRepository.find({
        where: { 
          id: In(registrationIds)
        },
        relations: ['allocatedProgram'],
      });

      // Check program/sub-program consistency
      for (const registration of registrations) {
        const allocatedProgramId = registration.allocatedProgramId;
        
        if (dto.subProgramId) {
          // If sub-program is provided, check if registration belongs to that sub-program
          if (allocatedProgramId !== dto.subProgramId) {
            // Find the allocation ID for this registration to provide better error message
            const allocation = existingAllocations.find(a => a.registrationId === registration.id);
            throw new InifniBadRequestException(
              ERROR_CODES.ROOM_ALLOCATION_PROGRAM_MISMATCH,
              null,
              null,
              `Allocation ID ${allocation?.id} (Registration ID: ${registration.id}) does not belong to sub-program ${dto.subProgramId}`
            );
          }
        } else {
          // If no sub-program provided, validate against main program
          // Check if registration belongs to the main program or any of its sub-programs
          const belongsToProgram = await this.validateRegistrationBelongsToProgram(registration, dto.programId);
          
          if (!belongsToProgram) {
            const allocation = existingAllocations.find(a => a.registrationId === registration.id);
            throw new InifniBadRequestException(
              ERROR_CODES.ROOM_ALLOCATION_PROGRAM_MISMATCH,
              null,
              null,
              `Allocation ID ${allocation?.id} (Registration ID: ${registration.id}) does not belong to program ${dto.programId}`
            );
          }
        }
      }

      this.logger.debug('Bulk room allocation delete request validation completed', {
        dto,
        userId: user.id,
        allocationsToDelete: existingAllocations.length,
        programValidation: dto.subProgramId ? `Sub-program ${dto.subProgramId}` : `Program ${dto.programId}`,
      });

      return existingAllocations;
    } catch (error) {
      this.logger.error('Bulk room allocation delete validation failed', error.stack, {
        dto,
        userId: user.id,
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_VALIDATION_FAILED, error);
    }
  }

  /**
   * Validate that a registration belongs to the specified program or its sub-programs
   * @param registration - The registration to validate
   * @param programId - The main program ID from the path
   * @returns True if registration belongs to the program hierarchy
   */
  private async validateRegistrationBelongsToProgram(
    registration: ProgramRegistration,
    programId: number,
  ): Promise<boolean> {
    const allocatedProgramId = registration.allocatedProgramId;

    // Case 1: Direct match with the main program
    if (allocatedProgramId === programId) {
      return true;
    }

    // Case 2: Check if the allocated program is a sub-program of the main program
    if (!allocatedProgramId) {
      return false;
    }

    const allocatedProgram = await this.programRepository.findOne({
      where: {
        id: allocatedProgramId,
        deletedAt: IsNull(),
        isActive: true,
      },
    });

    if (!allocatedProgram) {
      return false;
    }

    const mainProgram = await this.programRepository.findOne({
      where: {
        id: programId,
        deletedAt: IsNull(),
        isActive: true,
      },
    });

    if (!mainProgram) {
      return false;
    }

    // Check if there's a valid relationship
    return await this.validateProgramRelationship(mainProgram, allocatedProgram);
  }

  /**
   * Validate room allocation transfer request
   * @param allocationId - Allocation ID from path
   * @param dto - Transfer data
   * @param user - User performing the transfer
   * @returns Validation results with allocation and room inventory details
   */
  private async validateTransferRequest(
    allocationId: number,
    dto: TransferRoomAllocationDto,
    user: User,
  ): Promise<{
    allocation: RoomAllocation;
    currentRoomInventory: any;
    newRoomInventory?: any;
  }> {
    this.logger.debug('Validating room allocation transfer request', {
      allocationId,
      dto,
      userId: user.id,
    });

    try {
      // 1. Find the allocation by ID first to get the current room inventory ID
      const allocation = await this.repository.findById(allocationId);
      
      if (!allocation) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
          null,
          null,
          allocationId.toString()
        );
      }

      const currentInventoryId = allocation.programRoomInventoryMapId;

      // 2. Get current room inventory details to determine the program ID
      const currentRoomInventory = await this.roomInventoryService.getRoomOccupancyDetails(currentInventoryId);
      if (!currentRoomInventory) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_INVENTORY_NOT_FOUND,
          null,
          null,
          currentInventoryId.toString()
        );
      }

      // We need to get the program ID from the allocation's room inventory mapping
      if (!allocation.programRoomInventoryMap) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_INVENTORY_NOT_FOUND,
          null,
          null,
          allocationId.toString()
        );
      }

      const programId = allocation.programRoomInventoryMap.programId;

      // 3. Validate program and sub-program existence if subProgramId is provided
      if (dto.subProgramId) {
        await this.validateProgramAndSubProgramExistence(programId, dto.subProgramId);
      }

      // 4. Validate the registration belongs to the correct program or sub-program
      const registration = await this.registrationRepository.findOne({
        where: { id: allocation.registrationId },
        relations: ['allocatedProgram'],
      });

      if (!registration) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_NOT_FOUND,
          null,
          null,
          allocation.registrationId.toString()
        );
      }

      // Validate program/sub-program consistency for the registration
      const allocatedProgramId = registration.allocatedProgramId;
      
      if (dto.subProgramId) {
        // If sub-program is provided, check if registration's allocated program matches sub-program
        if (allocatedProgramId !== dto.subProgramId) {
          throw new InifniBadRequestException(
            ERROR_CODES.ROOM_ALLOCATION_PROGRAM_MISMATCH,
            null,
            null,
            registration.id.toString()
          );
        }
      } else {
        // If no sub-program, check if registration belongs to the program hierarchy
        const isValidProgram = await this.validateRegistrationBelongsToProgram(registration, programId);
        if (!isValidProgram) {
          throw new InifniBadRequestException(
            ERROR_CODES.ROOM_ALLOCATION_PROGRAM_MISMATCH,
            null,
            null,
            registration.id.toString()
          );
        }
      }

      let newRoomInventory: ProgramRoomInventoryMap | null = null;

      // 5. If new inventory ID is provided, validate the transfer to new room
      if (dto.newInventoryId && dto.newInventoryId !== currentInventoryId) {
        // Validate new room inventory mapping belongs to the same program (with subProgram support)
        newRoomInventory = await this.roomInventoryService.validateRoomInventoryMapping(
          dto.newInventoryId,
          programId,
          dto.subProgramId
        );

        this.logger.debug('New room inventory validation passed', {
          newRoomInventoryId: dto.newInventoryId,
          programId: programId,
          subProgramId: dto.subProgramId,
          newRoomDetails: {
            id: newRoomInventory.id,
            roomId: newRoomInventory.roomId,
            programId: newRoomInventory.programId
          }
        });

        // Check if new room has capacity
        if (newRoomInventory.remainingOccupancy < 1) {
          throw new InifniBadRequestException(
            ERROR_CODES.INSUFFICIENT_ROOM_CAPACITY,
            null,
            null,
            '1',
            newRoomInventory.remainingOccupancy.toString()
          );
        }

        // Get new room capacity and validate bed position
        const newRoomCapacity = newRoomInventory.room?.occupancy;
        if (!newRoomCapacity) {
          throw new InifniBadRequestException(
            ERROR_CODES.ROOM_CAPACITY_NOT_AVAILABLE,
            null,
            null,
            dto.newInventoryId.toString()
          );
        }

        if (dto.bedPosition > newRoomCapacity || dto.bedPosition < 1) {
          throw new InifniBadRequestException(
            ERROR_CODES.BED_POSITION_OUT_OF_RANGE,
            null,
            null,
            dto.bedPosition.toString(),
            newRoomCapacity.toString()
          );
        }

        // Check if bed position is already occupied in new room
        const newRoomAllocations = await this.findByProgramRoomInventoryMapId(dto.newInventoryId);
        const occupiedBedPositions = newRoomAllocations.map(a => a.bedPosition).filter(pos => pos !== null);
        
        if (occupiedBedPositions.includes(dto.bedPosition)) {
          throw new InifniBadRequestException(
            ERROR_CODES.BED_POSITION_ALREADY_OCCUPIED,
            null,
            null,
            dto.bedPosition.toString(),
            dto.newInventoryId.toString()
          );
        }

        this.logger.debug('Room transfer validation completed successfully', {
          fromRoomInventoryId: currentInventoryId,
          toRoomInventoryId: dto.newInventoryId,
          bedPosition: dto.bedPosition,
          allocationId,
          programId: programId,
          subProgramId: dto.subProgramId
        });

      } else {
        // 6. If staying in same room, validate bed position availability
        const currentRoomCapacity = allocation.programRoomInventoryMap.room?.occupancy;
        if (!currentRoomCapacity) {
          throw new InifniBadRequestException(
            ERROR_CODES.ROOM_CAPACITY_NOT_AVAILABLE,
            null,
            null,
            currentInventoryId.toString()
          );
        }

        if (dto.bedPosition > currentRoomCapacity || dto.bedPosition < 1) {
          throw new InifniBadRequestException(
            ERROR_CODES.BED_POSITION_OUT_OF_RANGE,
            null,
            null,
            dto.bedPosition.toString(),
            currentRoomCapacity.toString()
          );
        }

        // Check if bed position is available (excluding current allocation)
        const currentRoomAllocations = await this.findByProgramRoomInventoryMapId(currentInventoryId);
        const occupiedBedPositions = currentRoomAllocations
          .filter(a => a.id !== allocation.id)
          .map(a => a.bedPosition)
          .filter(pos => pos !== null);
        
        if (occupiedBedPositions.includes(dto.bedPosition)) {
          throw new InifniBadRequestException(
            ERROR_CODES.BED_POSITION_ALREADY_OCCUPIED,
            null,
            null,
            dto.bedPosition.toString(),
            currentInventoryId.toString()
          );
        }

        this.logger.debug('Bed position update validation completed successfully', {
          roomInventoryId: currentInventoryId,
          newBedPosition: dto.bedPosition,
          allocationId,
          programId: programId,
          subProgramId: dto.subProgramId
        });
      }

      this.logger.debug('Room allocation transfer request validation completed', {
        allocationId,
        dto,
        userId: user.id,
        isRoomTransfer: !!(dto.newInventoryId && dto.newInventoryId !== currentInventoryId),
        validationSummary: {
          allocationFound: true,
          allocationBelongsToRoom: true,
          registrationBelongsToProgram: true,
          roomInventoryValidated: true,
          bedPositionAvailable: true,
          roomCapacityChecked: true,
          subProgramValidated: !!dto.subProgramId
        }
      });

      return {
        allocation,
        currentRoomInventory,
        newRoomInventory,
      };
    } catch (error) {
      this.logger.error('Room allocation transfer validation failed', error.stack, {
        allocationId,
        dto,
        userId: user.id,
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_VALIDATION_FAILED, error);
    }
  }
}