import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, DataSource, IsNull, ILike, In, EntityManager } from 'typeorm';
import { RoomAllocation } from 'src/common/entities/room-allocation.entity';
import { CreateRoomAllocationDto } from './dto/create-room-allocation.dto';
import { UpdateRoomAllocationDto } from './dto/update-room-allocation.dto';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CommonDataService } from 'src/common/services/commonData.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { RoomInventoryRepository } from 'src/room-inventory/room-inventory.repository';
import { RoomAllocationHistory } from 'src/common/entities/room-allocation-history.entity';
import { formatHistoryRemarks, HistoryAction } from 'src/common/utils/room-allocation-history.util';

/**
 * Repository class for managing room allocation data operations
 * Handles database interactions for room allocations with proper error handling and validation
 */
@Injectable()
export class RoomAllocationRepository {
  constructor(
    @InjectRepository(RoomAllocation)
    private readonly roomAllocationRepo: Repository<RoomAllocation>,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
    private readonly commonDataService: CommonDataService,
    private readonly roomInventoryRepository: RoomInventoryRepository,
  ) {}

  /**
   * Create a new room allocation
   * @param dto - Room allocation creation data
   * @param userId - ID of the user creating the allocation
   * @returns Created room allocation with relations
   */
  async create(dto: CreateRoomAllocationDto, userId: number): Promise<RoomAllocation> {
    this.logger.log('Creating new room allocation', { dto, userId });
    
    try {
      return await this.dataSource.transaction(async (manager) => {
        // Check if registration is already allocated to a room
        const existingAllocation = await manager.findOne(RoomAllocation, {
          where: { 
            registrationId: dto.registrationId,
            deletedAt: IsNull()
          }
        });
        if (existingAllocation) {
          throw new InifniBadRequestException(
            ERROR_CODES.ROOM_ALLOCATION_ALREADY_EXISTS,
            null,
            null,
            `Registration ${dto.registrationId} is already allocated to a room`
          );
        }

        // Check if bed position is already occupied (if specified)
        if (dto.bedPosition) {
          const existingBedAllocation = await manager.findOne(RoomAllocation, {
            where: {
              programRoomInventoryMapId: dto.programRoomInventoryMapId,
              bedPosition: dto.bedPosition,
              deletedAt: IsNull()
            }
          });
          if (existingBedAllocation) {
            throw new InifniBadRequestException(
              ERROR_CODES.BED_POSITION_ALREADY_OCCUPIED,
              null,
              null,
              `${dto.bedPosition} is already occupied`
            );
          }
        }

        const roomAllocation = manager.create(RoomAllocation, {
          ...dto,
          createdById: userId,
          updatedById: userId,
          createdAt: new Date(),
          updatedAt: new Date()
        });

        const savedAllocation = await manager.save(RoomAllocation, roomAllocation);

        await this.createHistoryRecord(
          manager,
          {
            id: savedAllocation.id,
            programRoomInventoryMapId: savedAllocation.programRoomInventoryMapId,
            registrationId: savedAllocation.registrationId,
            bedPosition: savedAllocation.bedPosition,
            remarks: savedAllocation.remarks,
          },
          userId,
          'create',
        );

        // Fetch the created allocation with relations using the transaction manager
        const result = await manager.findOne(RoomAllocation, {
          where: { id: savedAllocation.id },
          relations: [
            'programRoomInventoryMap',
            'programRoomInventoryMap.program',
            'programRoomInventoryMap.room',
            'createdBy',
            'updatedBy',
            'roomAllocationHistories'
          ]
        });
        
        if (!result) {
          throw new InifniNotFoundException(
            ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
            null,
            null,
            `Created room allocation not found`
          );
        }
        
        return result;
      });
    } catch (error) {
      this.logger.error('Error creating room allocation', error.stack, { dto, userId });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_CREATE_FAILED, error);
    }
  }

  /**
   * Create multiple room allocations in bulk for better performance
   * @param allocations - Array of room allocation creation data
   * @param userId - ID of the user creating the allocations
   * @returns Array of created room allocations with relations
   */
  async bulkCreate(
    allocations: CreateRoomAllocationDto[],
    userId: number
  ): Promise<RoomAllocation[]> {
    this.logger.log('Creating bulk room allocations', { count: allocations.length, userId });
    
    try {
      return await this.dataSource.transaction(async (manager) => {
        // Validate all allocations first
        const registrationIds = allocations.map(a => a.registrationId);
        const roomInventoryIds = allocations.map(a => a.programRoomInventoryMapId);
        
        // Check for existing allocations for all registrations at once
        const existingAllocations = await manager.find(RoomAllocation, {
          where: { 
            registrationId: In(registrationIds),
            deletedAt: IsNull()
          }
        });

        if (existingAllocations.length > 0) {
          const conflictingRegistrations = existingAllocations.map(a => a.registrationId);
          throw new InifniBadRequestException(
            ERROR_CODES.ROOM_ALLOCATION_ALREADY_EXISTS,
            null,
            null,
            `Registrations already allocated: ${conflictingRegistrations.join(', ')}`
          );
        }

        // Check for bed position conflicts
        const bedPositionChecks = allocations
          .filter(a => a.bedPosition)
          .map(a => ({ 
            programRoomInventoryMapId: a.programRoomInventoryMapId, 
            bedPosition: a.bedPosition 
          }));

        if (bedPositionChecks.length > 0) {
          const existingBedAllocations = await manager.find(RoomAllocation, {
            where: bedPositionChecks.map(check => ({
              programRoomInventoryMapId: check.programRoomInventoryMapId,
              bedPosition: check.bedPosition,
              deletedAt: IsNull()
            }))
          });

          if (existingBedAllocations.length > 0) {
            const conflictingBeds = existingBedAllocations.map(a => 
              `Room ${a.programRoomInventoryMapId} Bed ${a.bedPosition}`
            );
            throw new InifniBadRequestException(
              ERROR_CODES.BED_POSITION_ALREADY_OCCUPIED,
              null,
              null,
              `Bed positions already occupied: ${conflictingBeds.join(', ')}`
            );
          }
        }

        // Create all room allocation entities
        const roomAllocations = allocations.map(dto => manager.create(RoomAllocation, {
          ...dto,
          createdById: userId,
          updatedById: userId,
          createdAt: new Date(),
          updatedAt: new Date()
        }));

        // Bulk insert using TypeORM's save method (optimized for bulk operations)
        const savedAllocations = await manager.save(RoomAllocation, roomAllocations);

        // Create history records in bulk
        const historyRecords = savedAllocations.map(allocation => 
          manager.create(RoomAllocationHistory, {
            roomAllocationId: allocation.id,
            programRoomInventoryMapId: allocation.programRoomInventoryMapId,
            registrationId: allocation.registrationId,
            bedPosition: allocation.bedPosition ?? null,
            remarks: formatHistoryRemarks('create', allocation.remarks),
            createdById: userId,
            updatedById: userId,
            createdAt: new Date(),
            updatedAt: new Date(),
          })
        );

        await manager.save(RoomAllocationHistory, historyRecords);

        // Return saved allocations with relations (use separate query for better performance)
        const result = await manager.find(RoomAllocation, {
          where: { id: In(savedAllocations.map(a => a.id)) },
          relations: [
            'programRoomInventoryMap',
            'programRoomInventoryMap.program',
            'programRoomInventoryMap.room',
            'createdBy',
            'updatedBy'
          ]
        });
        
        return result;
      });
    } catch (error) {
      this.logger.error('Error creating bulk room allocations', error.stack, { count: allocations.length, userId });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_CREATE_FAILED, error);
    }
  }

  /**
   * Find room allocation by ID with all relations
   * @param id - Room allocation ID
   * @returns Room allocation with relations or null if not found
   */
  async findById(id: number): Promise<RoomAllocation | null> {
    try {
      console.log('Finding room allocation by ID:', id);
      return await this.commonDataService.findOneById(
        this.roomAllocationRepo,
        id,
        true,
        [
          'programRoomInventoryMap',
          'programRoomInventoryMap.program',
          'programRoomInventoryMap.room',
          'createdBy',
          'updatedBy',
          'roomAllocationHistories'
        ]
      );
    } catch (error) {
      this.logger.error(`Error finding room allocation by ID: ${id}`, error.stack);
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Find all room allocations with pagination and filtering
   * @param limit - Number of records per page
   * @param offset - Offset for pagination
   * @param searchText - Search text for filtering
   * @param filters - Additional filters
   * @returns Paginated room allocations with metadata
   */
  async findAll(
    limit: number = 10,
    offset: number = 0,
    searchText: string = '',
    filters: Record<string, any> = {}
  ) {
    try {
      const whereClause: any = { deletedAt: IsNull() };

      // Apply filters
      if (filters.programRoomInventoryMapId) {
        whereClause.programRoomInventoryMapId = filters.programRoomInventoryMapId;
      }

      if (filters.registrationId) {
        whereClause.registrationId = filters.registrationId;
      }

      if (filters.bedPosition) {
        whereClause.bedPosition = filters.bedPosition;
      }

      // Apply search text
      let finalWhereClause: any = whereClause;
      if (searchText?.trim()) {
        const searchTerm = ILike(`%${searchText.trim()}%`);
        finalWhereClause = [
          { ...whereClause, remarks: searchTerm },
          // Add more searchable fields as needed
        ];
      }

      const relations = [
        'programRoomInventoryMap',
        'programRoomInventoryMap.program',
        'programRoomInventoryMap.room',
        'createdBy',
        'updatedBy'
      ];

      const data = await this.commonDataService.get(
        this.roomAllocationRepo,
        undefined,
        finalWhereClause,
        limit,
        offset,
        { id: 'DESC' },
        undefined,
        relations,
      );

      const total = await this.roomAllocationRepo.count({ where: whereClause });

      const paginationInfo = {
        totalPages: Math.ceil(total / limit),
        pageNumber: Math.floor(offset / limit) + 1,
        pageSize: limit,
        totalRecords: total,
        numberOfRecords: data.length,
      };

      return { data, pagination: paginationInfo };
    } catch (error) {
      this.logger.error('Error finding room allocations', error.stack);
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_GET_FAILED, error);
    }
  }

  /**
   * Update a room allocation
   * @param id - Room allocation ID
   * @param dto - Update data
   * @param userId - ID of the user updating the allocation
   * @returns Updated room allocation
   */
  async update(id: number, dto: UpdateRoomAllocationDto, userId: number): Promise<RoomAllocation> {
    this.logger.log('Updating room allocation', { id, dto, userId });
    
    try {
      return await this.dataSource.transaction(async (manager) => {
        const existingAllocation = await manager.findOne(RoomAllocation, {
          where: { id, deletedAt: IsNull() }
        });

        if (!existingAllocation) {
          throw new InifniNotFoundException(
            ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
            null,
            null,
            `Room allocation with ID ${id} not found`
          );
        }

        // Check if new registration is already allocated (if changing registration)
        if (dto.registrationId && dto.registrationId !== existingAllocation.registrationId) {
          const existingRegAllocation = await manager.findOne(RoomAllocation, {
            where: { 
              registrationId: dto.registrationId,
              deletedAt: IsNull()
            }
          });

          if (existingRegAllocation) {
            throw new InifniBadRequestException(
              ERROR_CODES.ROOM_ALLOCATION_ALREADY_EXISTS,
              null,
              null,
              `Registration ${dto.registrationId} is already allocated to a room`
            );
          }
        }

        // Check if new bed position is occupied (if changing bed position)
        if (dto.bedPosition && dto.bedPosition !== existingAllocation.bedPosition) {
          const programRoomInventoryMapId = dto.programRoomInventoryMapId || existingAllocation.programRoomInventoryMapId;
          
          const existingBedAllocation = await manager.findOne(RoomAllocation, {
            where: {
              programRoomInventoryMapId,
              bedPosition: dto.bedPosition,
              deletedAt: IsNull()
            }
          });

          if (existingBedAllocation && existingBedAllocation.id !== id) {
            throw new InifniBadRequestException(
              ERROR_CODES.BED_POSITION_ALREADY_OCCUPIED,
              null,
              null,
              `Bed position ${dto.bedPosition} is already occupied`
            );
          }
        }

        await manager.update(RoomAllocation, id, {
          ...dto,
          updatedById: userId,
        });

        const updatedAllocationState = {
          id,
          programRoomInventoryMapId:
            dto.programRoomInventoryMapId !== undefined
              ? dto.programRoomInventoryMapId
              : existingAllocation.programRoomInventoryMapId,
          registrationId:
            dto.registrationId !== undefined
              ? dto.registrationId
              : existingAllocation.registrationId,
          bedPosition:
            dto.bedPosition !== undefined
              ? dto.bedPosition
              : existingAllocation.bedPosition,
          remarks:
            dto.remarks !== undefined ? dto.remarks : existingAllocation.remarks,
        };

        await this.createHistoryRecord(manager, updatedAllocationState, userId, 'update');

        const result = await manager.findOne(RoomAllocation, {
          where: { id },
          relations: [
            'programRoomInventoryMap',
            'programRoomInventoryMap.program',
            'programRoomInventoryMap.room',
            'createdBy',
            'updatedBy',
            'roomAllocationHistories'
          ]
        });
        if (!result) {
          throw new InifniNotFoundException(
            ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
            null,
            null,
            `Updated room allocation not found`
          );
        }
        return result;
      });
    } catch (error) {
      this.logger.error('Error updating room allocation', error.stack, { id, dto, userId });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_UPDATE_FAILED, error);
    }
  }

  /**
   * Soft delete a room allocation
   * @param id - Room allocation ID
   * @param userId - ID of the user deleting the allocation
   * @returns Deletion result
   */
  async remove(id: number, userId: number): Promise<{ id: number }> {
    this.logger.log('Deleting room allocation', { id, userId });
    
    try {
      return await this.dataSource.transaction(async (manager) => {
        const existingAllocation = await manager.findOne(RoomAllocation, {
          where: { id, deletedAt: IsNull() }
        });

        if (!existingAllocation) {
          throw new InifniNotFoundException(
            ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
            null,
            null,
            `Room allocation with ID ${id} not found`
          );
        }

        await manager.update(RoomAllocation, id, {
          deletedAt: new Date(),
          updatedById: userId,
        });

        await this.createHistoryRecord(
          manager,
          {
            id,
            programRoomInventoryMapId: existingAllocation.programRoomInventoryMapId,
            registrationId: existingAllocation.registrationId,
            bedPosition: existingAllocation.bedPosition,
            remarks: existingAllocation.remarks,
          },
          userId,
          'delete',
        );

        return { id };
      });
    } catch (error) {
      this.logger.error('Error deleting room allocation', error.stack, { id, userId });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_DELETE_FAILED, error);
    }
  }

  /**
   * Find room allocations by program room inventory map ID
   * @param programRoomInventoryMapId - Program room inventory map ID
   * @returns Array of room allocations
   */
  async findByProgramRoomInventoryMapId(programRoomInventoryMapId: number): Promise<RoomAllocation[]> {
    try {
      return await this.roomAllocationRepo.find({
        where: { 
          programRoomInventoryMapId,
          deletedAt: IsNull()
        },
        relations: [
          'programRoomInventoryMap',
          'createdBy',
          'updatedBy'
        ],
        order: { bedPosition: 'ASC' }
      });
    } catch (error) {
      this.logger.error(`Error finding room allocations by program room inventory map ID: ${programRoomInventoryMapId}`, error.stack);
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FIND_BY_ROOM_INVENTORY_FAILED, error);
    }
  }

  /**
   * Find room allocation by registration ID
   * @param registrationId - Registration ID
   * @returns Room allocation or null if not found
   */
  async findByRegistrationId(registrationId: number): Promise<RoomAllocation | null> {
    try {
      return await this.roomAllocationRepo.findOne({
        where: { 
          registrationId,
          deletedAt: IsNull()
        },
        relations: [
          'programRoomInventoryMap',
          'programRoomInventoryMap.program',
          'programRoomInventoryMap.room',
          'createdBy',
          'updatedBy'
        ]
      });
    } catch (error) {
      this.logger.error(`Error finding room allocation by registration ID: ${registrationId}`, error.stack);
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FIND_BY_REGISTRATION_ID_FAILED, error);
    }
  }

  /**
   * Find room allocations by multiple registration IDs
   * @param registrationIds - Array of registration IDs
   * @returns Array of room allocations for the given registration IDs
   */
  async findByRegistrationIds(registrationIds: number[]): Promise<RoomAllocation[]> {
    try {
      if (!registrationIds || registrationIds.length === 0) {
        return [];
      }

      return await this.roomAllocationRepo.find({
        where: { 
          registrationId: In(registrationIds),
          deletedAt: IsNull()
        },
        relations: [
          'programRoomInventoryMap',
          'programRoomInventoryMap.program',
          'programRoomInventoryMap.room',
          'createdBy',
          'updatedBy'
        ]
      });
    } catch (error) {
      this.logger.error(`Error finding room allocations by registration IDs: ${registrationIds.join(', ')}`, error.stack);
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FIND_BY_REGISTRATION_ID_FAILED, error);
    }
  }

  /**
   * Get room occupancy statistics
   * @param programRoomInventoryMapId - Optional filter by program room inventory map
   * @returns Room occupancy statistics
   */
  async getRoomOccupancyStats(programRoomInventoryMapId?: number) {
    try {
      const queryBuilder = this.roomAllocationRepo
        .createQueryBuilder('allocation')
        .leftJoin('allocation.programRoomInventoryMap', 'inventory')
        .select([
          'COUNT(allocation.id) as totalAllocations',
          'COUNT(DISTINCT allocation.programRoomInventoryMapId) as occupiedRooms',
          'AVG(CASE WHEN allocation.bedPosition IS NOT NULL THEN 1 ELSE 0 END) as avgBedOccupancy'
        ])
        .where('allocation.deletedAt IS NULL');

      if (programRoomInventoryMapId) {
        queryBuilder.andWhere('allocation.programRoomInventoryMapId = :programRoomInventoryMapId', {
          programRoomInventoryMapId
        });
      }

      return await queryBuilder.getRawOne();
    } catch (error) {
      this.logger.error('Error getting room occupancy statistics', error.stack);
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_OCCUPANCY_STATS_FAILED, error);
    }
  }

  /**
   * Find room allocations by multiple allocation IDs
   * @param allocationIds - Array of allocation IDs
   * @returns Array of room allocations
   */
  async findByAllocationIds(allocationIds: number[]): Promise<RoomAllocation[]> {
    try {
      if (!allocationIds || allocationIds.length === 0) {
        return [];
      }

      return await this.roomAllocationRepo.find({
        where: { 
          id: In(allocationIds),
          deletedAt: IsNull()
        },
        relations: [
          'programRoomInventoryMap',
          'programRoomInventoryMap.program',
          'programRoomInventoryMap.room',
          'createdBy',
          'updatedBy'
        ]
      });
    } catch (error) {
      this.logger.error(`Error finding room allocations by allocation IDs: ${allocationIds.join(', ')}`, error.stack);
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Bulk soft delete room allocations
   * @param allocationIds - Array of allocation IDs to delete
   * @param userId - ID of the user deleting the allocations
   * @returns Array of deleted allocation IDs
   */
  async bulkRemove(allocationIds: number[], userId: number): Promise<{ deletedIds: number[]; count: number }> {
    this.logger.log('Bulk deleting room allocations', { allocationIds, userId });

    try {
      return await this.dataSource.transaction(async (manager) => {
        // Check which allocations exist and are not already deleted
        const existingAllocations = await manager.find(RoomAllocation, {
          where: {
            id: In(allocationIds),
            deletedAt: IsNull()
          }
        });

        if (existingAllocations.length === 0) {
          throw new InifniNotFoundException(
            ERROR_CODES.ROOM_ALLOCATION_NOT_FOUND,
            null,
            null,
            `None of the provided allocation IDs were found: ${allocationIds.join(', ')}`
          );
        }

        const existingIds = existingAllocations.map(allocation => allocation.id);
        
        // Soft delete all existing allocations
        await manager.update(RoomAllocation,
          { id: In(existingIds) },
          {
            deletedAt: new Date(),
            updatedById: userId,
          }
        );

        for (const allocation of existingAllocations) {
          await this.createHistoryRecord(
            manager,
            {
              id: allocation.id,
              programRoomInventoryMapId: allocation.programRoomInventoryMapId,
              registrationId: allocation.registrationId,
              bedPosition: allocation.bedPosition,
              remarks: allocation.remarks,
            },
            userId,
            'delete',
          );
        }

        return {
          deletedIds: existingIds,
          count: existingIds.length
        };
      });
    } catch (error) {
      this.logger.error('Error bulk deleting room allocations', error.stack, { allocationIds, userId });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_DELETE_FAILED, error);
    }
  }

  private async createHistoryRecord(
    manager: EntityManager,
    allocation: {
      id: number;
      programRoomInventoryMapId: number;
      registrationId: number;
      bedPosition?: number | null;
      remarks?: string | null;
    },
    userId: number,
    action: HistoryAction,
  ): Promise<void> {
    console.log('Creating history record for allocation:', allocation.id, 'Action:', action);
    const historyRecord = manager.create(RoomAllocationHistory, {
      roomAllocationId: allocation.id,
      programRoomInventoryMapId: allocation.programRoomInventoryMapId,
      registrationId: allocation.registrationId,
      bedPosition: allocation.bedPosition ?? null,
      remarks: formatHistoryRemarks(action, allocation.remarks),
      createdById: userId,
      updatedById: userId,
      createdAt: new Date(),
      updatedAt: new Date(),
    });

    await manager.save(RoomAllocationHistory, historyRecord);
  }
}
