import {
  Body,
  Controller,
  HttpCode,
  HttpStatus,
  Param,
  ParseIntPipe,
  Post,
  Req,
  Res,
  UseGuards,
  ValidationPipe,
  Delete,
  Put,
  Get,
  Query,
} from '@nestjs/common';
import { Response } from 'express';
import {
  ApiBearerAuth,
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiQuery,
  ApiResponse,
  ApiSecurity,
  ApiTags,
} from '@nestjs/swagger';
import { RoomAllocationService } from './room-allocation.service';
import { BulkRoomAllocationDto, BulkRoomAllocationDeleteDto, GetRegistrationsByProgramDto } from './dto/bulk-room-allocation.dto';
import { TransferRoomAllocationDto } from './dto/transfer-room-allocation.dto';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';

/**
 * Controller for managing bulk room allocation operations
 * Provides endpoints for allocating, deleting, and transferring room allocations
 * @class RoomAllocationController
 */
@ApiTags('Room Allocation Management')
@Controller('room-allocation')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'operational_manager', 'rm_support', 'shoba', 'mahatria','viewer')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class RoomAllocationController {
  constructor(
    private readonly service: RoomAllocationService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Create bulk room allocations for multiple registrations with individual bed positions
   * Validates all requirements before creating allocations including capacity, bed positions, and duplicates
   * @param dto - Bulk room allocation data with programId, registrations and room details including optional subProgramId for validation
   * @param req - Request object containing user information
   * @param res - Response object for sending result
   * @returns Promise<void> - Success response with created allocations
   */
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: 'Create bulk room allocations' })
  @ApiBody({ 
    type: BulkRoomAllocationDto,
    description: 'Bulk room allocation request payload with programId, registrations, room details, and optional sub-program validation',
    examples: {
      basicAllocation: {
        summary: 'Basic bulk allocation without sub-program',
        description: 'Simple bulk allocation with minimal required data - no sub-program validation',
        value: {
          programId: 123,
          registrations: [
            { registrationId: 123, bedPosition: 1 },
            { registrationId: 124, bedPosition: 2 },
            { registrationId: 125, bedPosition: 3 }
          ],
          roomInventoryId: 1,
          globalRemarks: 'Standard room allocation for program participants'
        }
      },
      withSubProgramValidation: {
        summary: 'Bulk allocation with sub-program validation',
        description: 'Allocation with sub-program ID for additional validation and security',
        value: {
          programId: 123,
          subProgramId: 10,
          registrations: [
            { registrationId: 123, bedPosition: 1 },
            { registrationId: 124, bedPosition: 2 },
            { registrationId: 125, bedPosition: 3 },
            { registrationId: 126, bedPosition: 4 }
          ],
          roomInventoryId: 1,
          globalRemarks: 'Workshop participants - Advanced Track allocation'
        }
      },
      detailedWithIndividualRemarks: {
        summary: 'Detailed allocation with individual remarks and sub-program',
        description: 'Complete allocation with individual bed preferences, remarks, and sub-program validation',
        value: {
          programId: 123,
          subProgramId: 10,
          registrations: [
            { registrationId: 123, bedPosition: 1, remarks: 'Window side preferred - medical requirement for natural light' },
            { registrationId: 124, bedPosition: 2, remarks: 'Near bathroom access needed due to mobility issues' },
            { registrationId: 125, bedPosition: 3, remarks: 'Quiet corner for meditation practice' },
            { registrationId: 126, bedPosition: 4 }
          ],
          roomInventoryId: 1,
          globalRemarks: 'Special accommodations group - Workshop Advanced Track'
        }
      },
      largeGroupAllocation: {
        summary: 'Large group allocation with mixed preferences',
        description: 'Example of allocating a larger group with some individual preferences',
        value: {
          programId: 123,
          subProgramId: 15,
          registrations: [
            { registrationId: 201, bedPosition: 1, remarks: 'Group leader - needs easy access' },
            { registrationId: 202, bedPosition: 2 },
            { registrationId: 203, bedPosition: 3 },
            { registrationId: 204, bedPosition: 4, remarks: 'Prefers upper bunk' },
            { registrationId: 205, bedPosition: 5 },
            { registrationId: 206, bedPosition: 6, remarks: 'Lower bunk only - knee injury' }
          ],
          roomInventoryId: 2,
          globalRemarks: 'Youth program participants - Summer Camp 2025'
        }
      }
    }
  })
  @ApiResponse({
    status: HttpStatus.CREATED,
    description: 'Bulk room allocation created successfully with detailed allocation information',
    schema: {
      type: 'object',
      properties: {
        statusCode: { type: 'number', example: 201 },
        message: { type: 'string', example: 'Successfully allocated 4 registrations to room with individual bed positions' },
        data: {
          type: 'array',
          description: 'Array of created room allocations',
          items: {
            type: 'object',
            properties: {
              id: { type: 'number', description: 'Unique room allocation identifier', example: 1 },
              programRoomInventoryMapId: { type: 'number', description: 'Program room inventory mapping ID', example: 1 },
              registrationId: { type: 'number', description: 'Registration ID that was allocated', example: 123 },
              bedPosition: { type: 'number', description: 'Assigned bed position (1-based indexing)', example: 1 },
              remarks: { type: 'string', description: 'Individual or global remarks for allocation', example: 'Window side preferred' },
              isDeleted: { type: 'boolean', description: 'Soft delete flag', example: false },
              createdAt: { type: 'string', format: 'date-time', description: 'Allocation creation timestamp' },
              updatedAt: { type: 'string', format: 'date-time', description: 'Last modification timestamp' },
              createdBy: {
                type: 'object',
                description: 'User who created the allocation',
                properties: {
                  id: { type: 'number', example: 1 },
                  name: { type: 'string', example: 'John Doe' }
                }
              },
              programRoomInventoryMap: {
                type: 'object',
                description: 'Room inventory mapping details',
                properties: {
                  id: { type: 'number', example: 1 },
                  roomNumber: { type: 'string', example: 'R-101' },
                  capacity: { type: 'number', example: 4 }
                }
              }
            }
          }
        }
      }
    }
  })
  @ApiResponse({
    status: HttpStatus.BAD_REQUEST,
    description: 'Invalid request data - duplicate registration IDs, duplicate bed positions, empty array, bed positions out of range, programId missing, or validation errors',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 400 },
        message: { 
          type: 'string', 
          examples: [
            'Duplicate registration IDs found in the array',
            'Duplicate bed positions found. Each registration must have a unique bed position',
            'Bed position 5 is invalid. Room capacity is 4 (positions 1-4)',
            'Registrations array is required and cannot be empty',
            'programId is required'
          ]
        },
        error: { type: 'string', example: 'Bad Request' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.NOT_FOUND, 
    description: 'Registration IDs not found or room inventory mapping not found',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 404 },
        message: { 
          type: 'string', 
          examples: [
            'Invalid registration IDs: 123, 456',
            'Room inventory mapping with ID 999 not found'
          ]
        },
        error: { type: 'string', example: 'Not Found' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.CONFLICT, 
    description: 'Room capacity insufficient, bed positions already occupied, or registrations already allocated',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 409 },
        message: { 
          type: 'string', 
          examples: [
            'Insufficient room capacity. Requested: 5, Available: 2',
            'Bed position 3 is already occupied',
            'Registrations already allocated to rooms: 123, 124'
          ]
        },
        error: { type: 'string', example: 'Conflict' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.UNPROCESSABLE_ENTITY, 
    description: 'Program/sub-program validation failed',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 422 },
        message: { 
          type: 'string', 
          examples: [
            'Registration ID 789 is not allocated to program 123',
            'Registration ID 456 is not allocated to sub-program 10'
          ]
        },
        error: { type: 'string', example: 'Unprocessable Entity' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.INTERNAL_SERVER_ERROR, 
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR 
  })
  async createBulkAllocation(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: BulkRoomAllocationDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(
        ERROR_CODES.USER_NOTFOUND, 
        null, 
        null, 
        'User not found in request'
      );
    }

    this.logger.log('Bulk room allocation request received', { 
      programId: dto.programId,
      dto, 
      userId: user.id 
    });

    try {
      const roomAllocations = await this.service.createBulkAllocation(dto.programId, dto, user);
      
      await this.responseService.success(
        res,
        `Successfully allocated ${roomAllocations.length} registrations to room with individual bed positions`,
        roomAllocations,
        HttpStatus.CREATED
      );
    } catch (error) {
      this.logger.error('Error in bulk room allocation controller', error.stack, {
        programId: dto.programId,
        dto,
        userId: user.id,
      });
      handleControllerError(res, error);
    }
  }

  /**
   * Fetch registrations by program ID with pagination, search, and filters
   * Retrieves program registrations with room allocation details, supporting search, filtering and pagination
   * @param queryParams - Query parameters with programId, search, limit, offset, and filters
   * @param req - Request object containing user information
   * @param res - Response object for sending result
   * @returns Promise<void> - Success response with paginated registrations data including room allocation status
   */
  @Get('registrations')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Fetch program registrations with room allocation details' })
  @ApiResponse({
    status: HttpStatus.OK,
    description: 'Registrations retrieved successfully with room allocation details and applied filters',
    schema: {
      type: 'object',
      properties: {
        statusCode: { 
          type: 'number', 
          example: 200,
          description: 'HTTP status code'
        },
        message: { 
          type: 'string', 
          example: 'Registrations retrieved successfully',
          description: 'Success message'
        },
        data: {
          type: 'object',
          description: 'Response data containing registrations and pagination information',
          properties: {
            data: {
              type: 'array',
              description: 'Array of registration records with room allocation details',
              items: {
                type: 'object',
                properties: {
                  id: { 
                    type: 'number', 
                    description: 'Unique registration identifier',
                    example: 1001
                  },
                  registrationSeqNumber: { 
                    type: 'string', 
                    description: 'Registration sequence/reference number',
                    example: 'REG-2025-001'
                  },
                  fullName: { 
                    type: 'string', 
                    description: 'Full name of the registered participant',
                    example: 'John Doe'
                  },
                  gender: { 
                    type: 'string', 
                    description: 'Gender of the participant',
                    enum: ['Male', 'Female', 'Other'],
                    example: 'Male'
                  },
                  mobileNumber: { 
                    type: 'string', 
                    description: 'Contact mobile number',
                    example: '+1234567890'
                  },
                  emailAddress: { 
                    type: 'string', 
                    description: 'Contact email address',
                    example: 'john.doe@example.com'
                  },
                  registrationStatus: { 
                    type: 'string', 
                    description: 'Current registration status',
                    example: 'CONFIRMED'
                  },
                  registrationDate: { 
                    type: 'string', 
                    format: 'date-time',
                    description: 'Date and time of registration',
                    example: '2025-10-15T10:30:00Z'
                  },
                  preferredRoomMate: { 
                    type: 'string', 
                    description: 'Name or ID of preferred roommate',
                    example: 'Jane Smith',
                    nullable: true
                  },
                  city: { 
                    type: 'string', 
                    description: 'City of residence',
                    example: 'New York',
                    nullable: true
                  },
                  countryName: { 
                    type: 'string', 
                    description: 'Country of residence',
                    example: 'United States',
                    nullable: true
                  },
                  programId: { 
                    type: 'number', 
                    description: 'Main program ID',
                    example: 123
                  },
                  programTitle: { 
                    type: 'string', 
                    description: 'Main program title',
                    example: 'Annual Meditation Retreat 2025'
                  },
                  allocatedProgramId: { 
                    type: 'number', 
                    description: 'Allocated program/sub-program ID (may differ from main program)',
                    example: 10,
                    nullable: true
                  },
                  allocatedProgramTitle: { 
                    type: 'string', 
                    description: 'Allocated program/sub-program title',
                    example: 'Advanced Workshop Track',
                    nullable: true
                  },
                  allocationId: { 
                    type: 'number', 
                    description: 'Room allocation ID (null if not yet allocated)',
                    example: 456,
                    nullable: true
                  },
                  bedPosition: { 
                    type: 'number', 
                    description: 'Assigned bed position in room (1-based, null if not allocated)',
                    example: 2,
                    nullable: true
                  },
                  allocationRemarks: { 
                    type: 'string', 
                    description: 'Remarks or notes for the room allocation',
                    example: 'Window side preferred',
                    nullable: true
                  },
                  roomInventoryId: { 
                    type: 'number', 
                    description: 'Room inventory mapping ID (null if not allocated)',
                    example: 1,
                    nullable: true
                  },
                  roomNumber: { 
                    type: 'string', 
                    description: 'Room number/identifier (null if not allocated)',
                    example: 'R-101',
                    nullable: true
                  },
                  roomLabel: { 
                    type: 'string', 
                    description: 'Room descriptive label (null if not allocated)',
                    example: 'Ground Floor - East Wing',
                    nullable: true
                  },
                  roomOccupancy: { 
                    type: 'number', 
                    description: 'Room total capacity/occupancy (null if not allocated)',
                    example: 4,
                    nullable: true
                  }
                }
              }
            },
            total: { 
              type: 'number', 
              description: 'Total number of registrations matching the filter criteria',
              example: 45
            },
            pagination: {
              type: 'object',
              description: 'Pagination metadata',
              properties: {
                totalPages: { 
                  type: 'number', 
                  description: 'Total number of pages available',
                  example: 5
                },
                pageNumber: { 
                  type: 'number', 
                  description: 'Current page number (1-based)',
                  example: 1
                },
                pageSize: { 
                  type: 'number', 
                  description: 'Number of records per page',
                  example: 10
                },
                totalRecords: { 
                  type: 'number', 
                  description: 'Total number of records',
                  example: 45
                },
                numberOfRecords: { 
                  type: 'number', 
                  description: 'Number of records in current response',
                  example: 10
                }
              }
            }
          }
        }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.BAD_REQUEST, 
    description: 'Invalid request data - validation errors for programId, search, limit, offset, or malformed filters JSON',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 400 },
        message: { 
          type: 'string', 
          examples: [
            'Validation failed for request parameters',
            'programId must be a positive integer',
            'limit must not be greater than 100',
            'offset must not be less than 0',
            'Invalid filters format - must be valid JSON'
          ]
        },
        error: { type: 'string', example: 'Bad Request' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.NOT_FOUND, 
    description: 'Program not found',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 404 },
        message: { 
          type: 'string', 
          examples: [
            'Program with ID 123 not found or inactive'
          ]
        },
        error: { type: 'string', example: 'Not Found' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.INTERNAL_SERVER_ERROR, 
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR 
  })
  async getRegistrationsByProgram(
    @Query(new ValidationPipe({ transform: true, whitelist: true })) queryParams: GetRegistrationsByProgramDto,
    @Req() req: any,
    @Res() res: Response
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(
        ERROR_CODES.USER_NOTFOUND, 
        null, 
        null, 
        'User not found in request'
      );
    }

    this.logger.log('Fetch registrations by program request received', { 
      queryParams, 
      userId: user.id 
    });

    try {
      // Decode and parse filters if provided
      let parsedFilters = {};
      if (queryParams.filters) {
        try {
          const decodedFilters = decodeURIComponent(queryParams.filters);
          parsedFilters = JSON.parse(decodedFilters);
          this.logger.log('Filters decoded and parsed successfully', { 
            originalFilters: queryParams.filters,
            decodedFilters,
            parsedFilters,
            userId: user.id 
          });
        } catch (error) {
          this.logger.error('Failed to decode or parse filters', error.stack, {
            filters: queryParams.filters,
            userId: user.id,
          });
          throw new InifniBadRequestException(
            ERROR_CODES.PQ_VALIDATION_FAILED,
            null,
            null,
            'Invalid filters format - must be valid URL-encoded JSON'
          );
        }
      }

      const parsedSort = queryParams.sort ? JSON.parse(queryParams.sort) : undefined;

      const data = await this.service.getRegistrationsByProgram(
        queryParams,
        queryParams.search,
        parsedFilters,
        parsedSort
      );
      
      await this.responseService.success(
        res,
        'Registrations retrieved successfully',
        data,
        HttpStatus.OK
      );
    } catch (error) {
      this.logger.error('Error in fetch registrations by program controller', error.stack, {
        queryParams,
        userId: user.id,
      });
      handleControllerError(res, error);
    }
  }

  /**
   * Delete bulk room allocations for multiple allocation IDs
   * Performs soft delete on specified allocations and recalculates room occupancy
   * @param dto - Bulk delete data with allocation IDs
   * @param req - Request object containing user information
   * @param res - Response object for sending result
   * @returns Promise<void> - Success response with deleted allocations count and updated room info
   */
  @Delete()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Delete bulk room allocations' })
  @ApiBody({ 
    type: BulkRoomAllocationDeleteDto,
    description: 'Bulk room allocation delete request payload with programId, allocation IDs, and optional sub-program validation',
    examples: {
      basicBulkDelete: {
        summary: 'Basic bulk deletion without sub-program',
        description: 'Simple bulk deletion with allocation IDs - can be from multiple rooms',
        value: {
          programId: 123,
          allocationIds: [1, 2, 3, 4]
        }
      },
      withSubProgramValidation: {
        summary: 'Bulk deletion with sub-program validation',
        description: 'Deletion with sub-program ID for additional validation and security',
        value: {
          programId: 123,
          subProgramId: 10,
          allocationIds: [5, 6, 7, 8, 9]
        }
      },
      singleAllocationDelete: {
        summary: 'Delete single allocation using bulk endpoint',
        description: 'Can also be used to delete a single allocation',
        value: {
          programId: 123,
          allocationIds: [15]
        }
      },
      largeGroupCancellation: {
        summary: 'Large group cancellation across multiple rooms',
        description: 'Example of canceling a large group booking that spans multiple rooms',
        value: {
          programId: 123,
          subProgramId: 20,
          allocationIds: [100, 101, 102, 103, 104, 105, 106, 107, 108, 109]
        }
      }
    }
  })
  @ApiResponse({
    status: HttpStatus.OK,
    description: 'Bulk room allocation deleted successfully with room occupancy recalculated for all affected rooms',
    schema: {
      type: 'object',
      properties: {
        statusCode: { type: 'number', example: 200 },
        message: { 
          type: 'string', 
          example: 'Successfully deleted 4 room allocations and recalculated room occupancy' 
        },
        data: {
          type: 'object',
          description: 'Deletion result with updated room information',
          properties: {
            deletedAllocations: {
              type: 'array',
              description: 'Array of deleted room allocations',
              items: {
                type: 'object',
                properties: {
                  id: { type: 'number', description: 'Room allocation ID', example: 1 },
                  programRoomInventoryMapId: { type: 'number', description: 'Room inventory mapping ID', example: 1 },
                  registrationId: { type: 'number', description: 'Registration ID', example: 123 },
                  bedPosition: { type: 'number', description: 'Bed position that was allocated', example: 1 },
                  remarks: { type: 'string', description: 'Allocation remarks', example: 'Window side preferred' }
                }
              }
            },
            deletedCount: { 
              type: 'number', 
              description: 'Number of allocations deleted',
              example: 4 
            },
            updatedRoomsInfo: {
              type: 'array',
              description: 'Room occupancy information for all affected rooms before and after deletion',
              items: {
                type: 'object',
                properties: {
                  roomInventoryId: { 
                    type: 'number', 
                    description: 'Room inventory mapping ID',
                    example: 1 
                  },
                  previousRemainingOccupancy: { 
                    type: 'number', 
                    description: 'Remaining occupancy before deletion',
                    example: 0 
                  },
                  newRemainingOccupancy: { 
                    type: 'number', 
                    description: 'Remaining occupancy after deletion',
                    example: 4 
                  },
                  previousRoomStatus: { 
                    type: 'string', 
                    description: 'Room status before deletion',
                    example: 'Fully Occupied' 
                  },
                  newRoomStatus: { 
                    type: 'string', 
                    description: 'Room status after deletion',
                    example: 'Available' 
                  }
                }
              }
            }
          }
        }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.BAD_REQUEST, 
    description: 'Invalid request data - empty array, duplicate allocation IDs, or validation errors',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 400 },
        message: { 
          type: 'string', 
          examples: [
            'Allocation IDs array is required and cannot be empty',
            'Duplicate allocation IDs found in the array',
            'programId is required'
          ]
        },
        error: { type: 'string', example: 'Bad Request' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.NOT_FOUND, 
    description: 'Allocation IDs not found',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 404 },
        message: { 
          type: 'string', 
          examples: [
            'Room allocation IDs not found: 123, 456'
          ]
        },
        error: { type: 'string', example: 'Not Found' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.UNPROCESSABLE_ENTITY, 
    description: 'Program/sub-program validation failed for registrations',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 422 },
        message: { 
          type: 'string', 
          examples: [
            'Registration ID 789 is not allocated to program 123',
            'Registration ID 456 is not allocated to sub-program 10'
          ]
        },
        error: { type: 'string', example: 'Unprocessable Entity' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.INTERNAL_SERVER_ERROR, 
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR 
  })
  async deleteBulkAllocation(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: BulkRoomAllocationDeleteDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(
        ERROR_CODES.USER_NOTFOUND, 
        null, 
        null, 
        'User not found in request'
      );
    }

    this.logger.log('Bulk room allocation delete request received', { 
      dto, 
      userId: user.id 
    });

    try {
      const result = await this.service.deleteBulkAllocation(dto, user);
      
      await this.responseService.success(
        res,
        `Successfully deleted ${result.deletedCount} room allocations and recalculated room occupancy`,
        result,
        HttpStatus.OK
      );
    } catch (error) {
      this.logger.error('Error in bulk room allocation delete controller', error.stack, {
        dto,
        userId: user.id,
      });
      handleControllerError(res, error);
    }
  }

  /**
   * Transfer room allocation to a new room or update bed position
   * Moves allocation to a different room or updates bed position within the same room
   * @param payload - Transfer data containing programId, subProgramId, and updateAllocations
   * @param req - Request object containing user information
   * @param res - Response object for sending result
   * @returns Promise<void> - Success response with updated allocation details
   */
  @Put()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Transfer room allocation to a new room or update bed position' })
  @ApiBody({ 
    type: Object,
    description: 'Payload for transferring room allocations',
    examples: {
      basicTransfer: {
        summary: 'Basic transfer to a new room',
        description: 'Move allocation to a new room with a specific bed position',
        value: {
          programId: 1,
          subProgramId: 2,
          updateAllocations: [
            { allocationId: 99, newRoomInventoryId: 22, bedPosition: 1 }
          ]
        }
      },
      multipleTransfers: {
        summary: 'Multiple transfers to new rooms',
        description: 'Move multiple allocations to new rooms with specific bed positions',
        value: {
          programId: 1,
          subProgramId: 2,
          updateAllocations: [
            { allocationId: 99, newRoomInventoryId: 22, bedPosition: 1 },
            { allocationId: 100, newRoomInventoryId: 23, bedPosition: 2 }
          ]
        }
      }
    }
  })
  @ApiResponse({
    status: HttpStatus.OK,
    description: 'Room allocations transferred successfully',
    schema: {
      type: 'object',
      properties: {
        statusCode: { type: 'number', example: 200 },
        message: { type: 'string', example: 'Room allocations transferred successfully' },
        data: {
          type: 'array',
          description: 'Array of updated room allocations',
          items: {
            type: 'object',
            properties: {
              allocationId: { type: 'number', description: 'Allocation ID', example: 99 },
              newRoomInventoryId: { type: 'number', description: 'New room inventory ID', example: 22 },
              bedPosition: { type: 'number', description: 'Updated bed position', example: 1 },
              updatedAt: { type: 'string', format: 'date-time', description: 'Last modification timestamp', example: '2025-10-24T10:30:00Z' }
            }
          }
        }
      }
    }
  })
  async transferRoomAllocations(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) payload: {
      programId: number;
      subProgramId: number;
      updateAllocations: Array<{ allocationId: number; newRoomInventoryId: number; bedPosition: number }>;
    },
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(
        ERROR_CODES.USER_NOTFOUND, 
        null, 
        null, 
        'User not found in request'
      );
    }

    this.logger.log('Bulk room allocation transfer request received', { payload, userId: user.id });

    try {
      const updatedAllocations = await this.service.transferRoomAllocations(payload, user);
      await this.responseService.success(
        res,
        'Room allocations transferred successfully',
        updatedAllocations,
        HttpStatus.OK
      );
    } catch (error) {
      this.logger.error('Error in bulk room allocation transfer controller', error.stack, { payload, userId: user.id });
      handleControllerError(res, error);
    }
  }

  /**
   * Get filter configuration for room allocation registrations
   * Provides age and location filter options based on approved registrations
   * @param programId - Program ID to get filter options for
   * @param subProgramId - Optional sub program ID for more specific filtering
   * @param filters - Optional URL-encoded JSON string containing additional filter criteria
   * @param res - Express Response object
   * @returns Promise<any[]> Complete filter configuration with age and location options
   */
  @Get('registrations/filter-config')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get room allocation filter configuration' })
  @ApiQuery({ 
    name: 'programId', 
    type: Number, 
    required: true, 
    description: 'Program ID to get filter options for',
    example: 1
  })
  @ApiQuery({ 
    name: 'subProgramId', 
    type: Number, 
    required: false, 
    description: 'Sub program ID for more specific filtering',
    example: 2
  })
  @ApiQuery({ 
    name: 'filters', 
    type: String, 
    required: false, 
    description: 'URL-encoded JSON string containing additional filter criteria',
    example: '{"gender":"Male"}'
  })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Filter configuration retrieved successfully',
    schema: {
      example: {
        success: true,
        message: 'Room allocation filter configuration retrieved successfully',
        data: [
          {
            key: 'age',
            label: 'Age Range',
            type: 'checkbox',
            sortable: true,
            filterable: true,
            order: 1,
            options: [
              { label: '18-25 years', value: '18-25' },
              { label: '26-35 years', value: '26-35' },
              { label: '36-45 years', value: '36-45' },
              { label: '46-55 years', value: '46-55' },
              { label: '56+ years', value: '56+' }
            ]
          },
          {
            key: 'city',
            label: 'Location',
            type: 'checkbox',
            sortable: false,
            filterable: true,
            order: 2,
            options: [
              { label: 'Bangalore', value: 'Bangalore' },
              { label: 'Mumbai', value: 'Mumbai' },
              { label: 'Delhi', value: 'Delhi' }
            ]
          }
        ]
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.BAD_REQUEST, 
    description: SWAGGER_API_RESPONSE.BAD_REQUEST,
    schema: {
      example: {
        success: false,
        message: 'programId is required',
        errorCode: 'VALIDATION_FAILED'
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.UNAUTHORIZED, 
    description: SWAGGER_API_RESPONSE.UNAUTHORIZED
  })
  @ApiResponse({ 
    status: HttpStatus.FORBIDDEN, 
    description: SWAGGER_API_RESPONSE.FORBIDDEN
  })
  @ApiResponse({ 
    status: HttpStatus.INTERNAL_SERVER_ERROR, 
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR
  })
  async getRoomAllocationFilterConfiguration(
    @Res() res: Response,
    @Query('programId', ParseIntPipe) programId: number,
    @Query('subProgramId') subProgramId?: number,
    @Query('filters') filters?: string,
  ) {
    this.logger.log('Room allocation filter configuration request received', {
      programId,
      subProgramId,
      filters,
      timestamp: new Date().toISOString()
    });

    try {
      // Validate programId
      if (!programId) {
        throw new InifniBadRequestException(
          ERROR_CODES.PQ_VALIDATION_FAILED,
          null,
          null,
          'programId is required'
        );
      }

      // Decode and parse filters if provided
      let parsedFilter = {};
      if (filters) {
        try {
          const decodedFilter = decodeURIComponent(filters);
          parsedFilter = JSON.parse(decodedFilter);
          this.logger.log('Filter decoded and parsed successfully', { 
            originalFilter: filters,
            decodedFilter,
            parsedFilter
          });
        } catch (error) {
          this.logger.error('Failed to decode or parse filter', error.stack, {
            filters
          });
          throw new InifniBadRequestException(
            ERROR_CODES.PQ_VALIDATION_FAILED,
            null,
            null,
            'Invalid filter format - must be valid URL-encoded JSON'
          );
        }
      }

      // Get filter configuration from service
      const filterConfig = await this.service.getFilterConfiguration(
        programId,
        subProgramId,
        parsedFilter
      );

      this.logger.log('Room allocation filter configuration retrieved successfully', {
        filtersCount: filterConfig.length,
        programId,
        subProgramId,
        appliedFilters: parsedFilter
      });

      await this.responseService.success(
        res,
        'Room allocation filter configuration retrieved successfully',
        filterConfig,
        HttpStatus.OK
      );

    } catch (error) {
      this.logger.error(
        `Failed to retrieve room allocation filter configuration: ${error.message}`,
        error.stack,
        { requestParams: { programId, subProgramId, filters } }
      );
      
      handleControllerError(res, error);
    }
  }

  /**
   * Fetch registrations with preferred roommate details
   * Delegates business logic to the service layer
   * @param programId - Program ID to filter registrations
   * @param subProgramId - Optional sub-program ID for filtering
   * @param search - Optional search text to filter registrations by full name
   * @returns Array of registrations with preferred roommate details
   */
  @Get('registrations/preferred-roommate')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Fetch registrations with preferred roommate details' })
  @ApiQuery({ 
    name: 'programId', 
    type: Number, 
    required: true, 
    description: 'Program ID to filter registrations',
    example: 1
  })
  @ApiQuery({ 
    name: 'subProgramId', 
    type: Number, 
    required: false, 
    description: 'Sub-program ID for more specific filtering',
    example: 2
  })
  @ApiQuery({ 
    name: 'search', 
    type: String, 
    required: false, 
    description: 'Search text to filter registrations by full name',
    example: 'John'
  })
  @ApiQuery({ 
    name: 'limit', 
    type: Number, 
    required: false, 
    description: 'Number of records to retrieve',
    example: 10
  })
  @ApiQuery({ 
    name: 'offset', 
    type: Number, 
    required: false, 
    description: 'Number of records to skip',
    example: 0
  })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Registrations retrieved successfully with preferred roommate details',
    schema: {
      type: 'object',
      properties: {
        statusCode: { type: 'number', example: 200 },
        message: { type: 'string', example: 'Registrations retrieved successfully' },
        data: {
          type: 'array',
          items: {
            type: 'object',
            properties: {
              id: { type: 'number', example: 1 },
              fullName: { type: 'string', example: 'John Doe' },
              preferredRoomMate: { type: 'string', example: 'Jane Smith', nullable: true },
              programId: { type: 'number', example: 123 },
              subProgramId: { type: 'number', example: 10, nullable: true }
            }
          }
        }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.BAD_REQUEST, 
    description: 'Invalid request data - validation errors for programId or search',
    schema: {
      properties: {
        statusCode: { type: 'number', example: 400 },
        message: { type: 'string', example: 'programId is required' },
        error: { type: 'string', example: 'Bad Request' }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.INTERNAL_SERVER_ERROR, 
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR 
  })
  async getPreferredRoommateRegistrations(
    @Query('programId') programId: number,
    @Query('subProgramId') subProgramId?: number,
    @Query('search') search?: string,
    @Query('limit') limit?: number,
    @Query('offset') offset?: number
  ) {
    const data = await this.service.getPreferredRoommateRegistrations(programId, subProgramId, limit, offset, search,);
    return {
      statusCode: HttpStatus.OK,
      message: 'Registrations retrieved successfully',
      data
    };
  }
}