import { ApiProperty } from '@nestjs/swagger';
import { IsArray, IsInt, IsNotEmpty, IsOptional, IsString, ArrayMinSize, ArrayNotEmpty, Min, Max, ValidateNested } from 'class-validator';
import { Type } from 'class-transformer';

/**
 * Individual room allocation details for each registration
 */
export class RegistrationAllocationDto {
  @ApiProperty({ 
    description: 'Registration ID to allocate to the room',
    example: 123,
    type: 'number'
  })
  @IsInt()
  @IsNotEmpty()
  @Min(1)
  registrationId: number;

  @ApiProperty({ 
    description: 'Bed position within the room (1-based index)',
    example: 1,
    type: 'number',
    minimum: 1
  })
  @IsInt()
  @IsNotEmpty()
  @Min(1)
  bedPosition: number;

  @ApiProperty({ 
    description: 'Individual remarks for this registration allocation',
    example: 'Window side bed preferred',
    type: 'string',
    required: false
  })
  @IsOptional()
  @IsString()
  remarks?: string;
}

/**
 * Data Transfer Object for bulk room allocation
 * Allows allocating multiple registrations to a room at once with individual bed positions and remarks
 */
export class BulkRoomAllocationDto {
  @ApiProperty({ 
    description: 'Program ID - all allocations must belong to this program',
    example: 123,
    type: 'integer',
    required: true
  })
  @IsInt()
  @IsNotEmpty()
  @Min(1)
  programId: number;

  @ApiProperty({ 
    description: 'Sub-program ID (optional)',
    example: 10,
    type: 'integer',
    required: false
  })
  @IsOptional()
  @IsInt()
  @Min(1)
  subProgramId?: number;

  @ApiProperty({ 
    description: 'Array of registration allocation details with bed positions and remarks',
    example: [
      { registrationId: 123, bedPosition: 1, remarks: 'Window side preferred' },
      { registrationId: 124, bedPosition: 2 },
      { registrationId: 125, bedPosition: 3, remarks: 'Near entrance' }
    ],
    type: [RegistrationAllocationDto],
    minItems: 1
  })
  @IsArray()
  @ArrayNotEmpty()
  @ArrayMinSize(1)
  @ValidateNested({ each: true })
  @Type(() => RegistrationAllocationDto)
  registrations: RegistrationAllocationDto[];

  @ApiProperty({ 
    description: 'Program room inventory map ID',
    example: 1,
    type: 'integer'
  })
  @IsInt()
  @IsNotEmpty()
  @Min(1)
  roomInventoryId: number;

  @ApiProperty({ 
    description: 'Global remarks for the bulk room allocation (used when individual remarks are not provided)',
    example: 'Group allocation for workshop participants',
    type: 'string',
    required: false
  })
  @IsOptional()
  @IsString()
  globalRemarks?: string;
}

/**
 * Data Transfer Object for bulk room allocation deletion
 * Allows deleting multiple room allocations at once by allocation IDs
 */
export class BulkRoomAllocationDeleteDto {
  @ApiProperty({ 
    description: 'Program ID - all allocations must belong to this program',
    example: 123,
    type: 'integer'
  })
  @IsInt()
  @IsNotEmpty()
  @Min(1)
  programId: number;

  @ApiProperty({ 
    description: 'Array of room allocation IDs to delete',
    example: [1, 2, 3, 4, 5],
    type: [Number],
    minItems: 1
  })
  @IsArray()
  @ArrayNotEmpty()
  @ArrayMinSize(1)
  @IsInt({ each: true })
  @Min(1, { each: true })
  allocationIds: number[];

  @ApiProperty({ 
    description: 'Sub-program ID (optional) - if provided, validates that allocations belong to this sub-program',
    example: 10,
    type: 'integer',
    required: false
  })
  @IsOptional()
  @IsInt()
  @Min(1)
  subProgramId?: number;
}

/**
 * Data Transfer Object for fetching registrations by program and sub-program
 * Used as query parameters for GET requests
 */
export class GetRegistrationsByProgramDto {
  @ApiProperty({ 
    description: 'Program ID to fetch registrations for',
    example: 123,
    type: 'integer',
    required: true
  })
  @Type(() => Number)
  @IsInt()
  @IsNotEmpty()
  @Min(1)
  programId: number;

  @ApiProperty({ 
    description: 'Number of records to fetch per page',
    example: 10,
    type: 'integer',
    required: false,
    default: 10,
    minimum: 1,
    maximum: 100
  })
  @Type(() => Number)
  @IsOptional()
  @IsInt()
  @Min(1)
  @Max(100)
  limit?: number = 10;

  @ApiProperty({ 
    description: 'Number of records to skip for pagination',
    example: 0,
    type: 'integer',
    required: false,
    default: 0,
    minimum: 0
  })
  @Type(() => Number)
  @IsOptional()
  @IsInt()
  @Min(0)
  offset?: number = 0;

  @ApiProperty({ 
    description: 'URL-encoded JSON string containing filter criteria. Must be encoded using encodeURIComponent. Example filters: {"registrationStatus": "CONFIRMED", "gender": "Male", "hasRoomAllocation": true}',
    example: '%7B%22registrationStatus%22%3A%22CONFIRMED%22%2C%22gender%22%3A%22Male%22%7D',
    type: 'string',
    required: false
  })
  @IsOptional()
  @IsString()
  filters?: string;

  @ApiProperty({ 
    description: 'Search text to filter registrations by name, email, or mobile number',
    example: 'John Doe',
    type: 'string',
    required: false
  })
  @IsOptional()
  @IsString()
  search?: string;

  @ApiProperty({ 
    description: 'Sorting criteria as an array of objects. Example: [{"age":"asc"}]',
    example: '[{"age":"asc"}]',
    type: 'string',
    required: false
  })
  @IsOptional()
  @IsString()
  sort?: string;
}