import {
  Body,
  Controller,
  HttpCode,
  HttpStatus,
  Post,
  Get,
  Put,
  Res,
  UseGuards,
  ValidationPipe,
  Req,
  Query,
  Param,
  ParseIntPipe,
  Delete,
} from '@nestjs/common';
import { Response } from 'express';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiResponse, ApiSecurity, ApiTags, ApiQuery, ApiParam } from '@nestjs/swagger';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { getRegistrationStatusDateTime, getStatusDateTimeLabel } from 'src/common/utils/common.util';
import { registrationMessages, ROLE_VALUES, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { RegistrationService } from './registration.service';
import { CreateRegistrationDto } from './dto/create-registration.dto';
import { UpdateParentalConsentDto, UpdateRegistrationDto } from './dto/update-registration.dto';
import { CancelRegistrationDto } from './dto/cancel-registration.dto';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { MainProgramDimensions, REPORT_CODES, RoleKPIs, SubProgramDimensions } from 'src/common/constants/constants';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';
import { CommunicationTypeEnum } from 'src/common/enum/communication-type.enum';
import { CommunicationTemplatesKeysEnum } from 'src/common/enum/communication-template-keys.enum';
import { RegistrationSortKey } from 'src/common/enum/registration-sort-key.enum';
import { UserProfileExtensionService } from 'src/user-profile-extension/user-profile-extension.service';
import { ViewTypeEnum } from 'src/common/enum/view-type.enum';

@ApiTags('registration')
@Controller('registration')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'viewer','mahatria', 'rm', 'finance_manager', 'relational_manager', 'shoba', 'operational_manger', 'rm_support')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class RegistrationController {
  constructor(
    private readonly service: RegistrationService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
    private readonly userProfileExtensionService: UserProfileExtensionService,
  ) { }

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: registrationMessages.CREATE_REQUEST_RECEIVED })
  @ApiBody({ type: CreateRegistrationDto })
  @ApiResponse({
    status: HttpStatus.CREATED,
    description: SWAGGER_API_RESPONSE.CREATED,
    schema: {
      properties: {
        registrationId: { type: 'number' },
        pendingApproval: { type: 'boolean' },
        approvalType: { type: 'string' },
        registrationLevel: { type: 'string' },
        registrationStatus: { type: 'string' }
      }
    }
  })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async register(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CreateRegistrationDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
    }
    const isSelfRegister = dto.isSelfRegister === true;
    this.logger.log(registrationMessages.CREATE_REQUEST_RECEIVED, dto);

    try {
      const data = await this.service.register(dto, user, isSelfRegister);
      
      // Determine message based on registration outcome
      let message = registrationMessages.REGISTRATION_SUCCESS;
      
      if (data.pendingApproval) {
        message = `${registrationMessages.REGISTRATION_PENDING_APPROVAL} (Type: ${data.approvalType})`;
      } else if (data.waitlisted) {
        message = registrationMessages.REGISTRATION_WAITLISTED;
      }
      
      await this.responseService.success(res, message, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id/statuses')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_STATUSES })
  @ApiParam({ name: 'id', description: 'Registration ID', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getStatuses(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    try {
      const data = await this.service.getRegistrationStatuses(id);
      await this.responseService.success(res, registrationMessages.STATUSES_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.UPDATE_REQUEST_RECEIVED })
  @ApiBody({ type: UpdateRegistrationDto })
    @ApiQuery({ name: 'isAdmin', type: Boolean, required: false, description: 'Set to true if the updater is an admin user' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async updateRegistration(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdateRegistrationDto,
    @Query('isAdmin') isAdmin = 'false',
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
    }
    this.logger.log(registrationMessages.UPDATE_REQUEST_RECEIVED, dto);
    
    try {
      const data = await this.service.update(dto, user, isAdmin === 'true');
      
      // Determine message based on update outcome
      let message = registrationMessages.REGISTRATION_UPDATED;

      if (data.registrationStatus === RegistrationStatusEnum.PENDING) {
        message = registrationMessages.REGISTRATION_PENDING_APPROVAL;
      } else if (data.waitlisted === true) {
        message = registrationMessages.REGISTRATION_WAITLISTED;
      }
      
      await this.responseService.success(res, message, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }


  @Put('cancel')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Cancel a registration' })
  @ApiBody({ type: CancelRegistrationDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async cancelRegistration(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CancelRegistrationDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
    }
    this.logger.log('Cancellation request received', dto);

    try {
      const data = await this.service.cancel(dto, user);
      await this.responseService.success(res, 'Registration cancelled successfully', data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }


  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_ALL })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Filter by program ID' })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Filter by program session ID' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Req() req: any,
    @Res() res: Response,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const user = req.user;
      const userRoles = user?.roles || [];
      
      const data = await this.service.findAll(
        +limit,
        +offset,
        programId ? +programId : null,
        programSessionId ? +programSessionId : null,
        searchText,
        parsedFilters,
        userRoles,
        user?.id || null
      );
      const allowedSaveDraftRoles = ['viewer'];
      if (data) {
        // for each registration make registration status as pending_approval if it is in pendinG state and approovals status is pending
        data.data.forEach(registration => {
          if (registration.registrationStatus === RegistrationStatusEnum.PENDING && registration.approvals[0].approvalStatus === ApprovalStatusEnum.PENDING) {
            registration.registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
          }
          if (!userRoles.some(role => allowedSaveDraftRoles.includes(role.name))) {
             // if travelPlanStatus map contains save_as_draft, then set travelPlanStatus to [] remove that same in travelInfo also
            if (registration.travelPlans && registration.travelPlans.length> 0 && registration.travelPlans.some(tp => tp.travelPlanStatus === 'save_as_draft')) {
              registration.travelPlans = registration.travelPlans?.filter(tp => tp.travelPlanStatus !== 'save_as_draft');
            }
            if (registration.travelInfo && registration.travelInfo.length > 0 && registration.travelInfo.some(ti => ti.travelInfoStatus === 'save_as_draft')) {
              registration.travelInfo = registration.travelInfo?.filter(ti => ti.travelInfoStatus !== 'save_as_draft');
            }
         }
        }
      );
      }
      await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('basic-list')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get basic registration list with minimal fields (registrationId, userId, name). For relational_manager role, returns only registrations assigned to them.' })
  @ApiQuery({ name: 'programId', required: true, type: Number, description: 'Program ID to filter registrations' })
  @ApiQuery({ name: 'searchText', required: false, type: String, description: 'Search text to filter by name, phone number, or email address' })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Basic registration list retrieved successfully',
    schema: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          registrationId: { type: 'number', description: 'Registration ID' },
          userId: { type: 'number', description: 'User ID' },
          name: { type: 'string', description: 'Full name of the registrant' },
          phoneNumber: { type: 'string', description: 'Phone number of the registrant' },
          emailAddress: { type: 'string', description: 'Email address of the registrant' }
        }
      }
    }
  })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getBasicRegistrationsList(
    @Query('programId', ParseIntPipe) programId: number,
    @Req() req: any,
    @Res() res: Response,
    @Query('searchText') searchText?: string,
  ) {
    try {
      const user = req.user;
      const userRoles = user?.roles || [];
      
      const data = await this.service.getBasicRegistrations(userRoles, user?.id, programId, searchText);
      
      await this.responseService.success(res, 'Basic registration list retrieved successfully', data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('filter-config/:programId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get complete three-level filter configuration for a program (cached by frontend per program)' })
  @ApiParam({ name: 'programId', description: 'Program ID', type: Number })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Complete filter configuration retrieved successfully with actual sub-program IDs',
    schema: {
      type: 'object',
      properties: {
        parentOptions: {
          type: 'array',
          description: 'Available parent filter options with nested KPI options containing kpiFilter values'
        },
        sideFilterSets: {
          type: 'object',
          description: 'Complete side filter sets including base sets and contextual filters'
        },
        programMapping: {
          type: 'object',
          description: 'Actual sub-program ID mappings for HDB1, HDB2, HDB3, MSD1, MSD2'
        },
        programId: {
          type: 'number',
          description: 'The program ID this configuration is for'
        },
        lastUpdated: {
          type: 'string',
          description: 'ISO timestamp when configuration was generated'
        }
      }
    }
  })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getFilterConfig(
    @Param('programId', ParseIntPipe) programId: number,
    @Req() req: any,
    @Res() res: Response,
  ) {
    try {
      const user = req.user;
      const userRoles = user?.roles?.map(role => role.name || role) || [];
      
      const filterConfig = await this.service.getThreeLevelFilterConfiguration(programId, userRoles);
      await this.responseService.success(res, 'Complete filter configuration retrieved successfully', filterConfig);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('registration-list-view')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get registrations list with dynamic KPIs driven by parentFilter from filter-hierarchy' })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Filter by program ID' })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Filter by program session ID' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters' })
  @ApiQuery({ name: 'parentFilter', type: String, required: true, description: 'Parent filter key for dynamic KPI generation (e.g., "all", "blessed", "programs")' })
  @ApiQuery({ name: 'downloadType', type: String, required: false, description: 'Download type: "all", "filtered", or report codes like "all_seekers", "registration_form_report", "finance_report", "travel_report", "master_list", "master_travel_and_goodies_report", "payment_list", "birthday_list", "song_preferences_list", "hold_list", "confirmed_list", "travel_onward_and_return_report","goodies_and_ratria_pillar_report","travel_onward_report","travel_return_report", "swap_can_shift_report' })
  @ApiQuery({
    name: 'sortKey',
    type: String,
    required: false,
    description: 'Field name to sort by. Available fields: seekerName, profileUrl, age, gender, location, averageRating, blessedWith, numberOfHDBs, id, fullName, dob, city, noOfHDBs, mobileNumber, emailAddress, blessedDate, cancelledDate, holdDate, unassignedDate, swapDemandDate, swapRequestDate, pendingDate, statusDateTime. Note: Nested relation fields are not supported for sorting.',
  })
  @ApiQuery({
    name: 'sortOrder',
    enum: ['ASC', 'DESC'],
    required: false,
    description: 'Sort direction (ASC or DESC)',
  })
  @ApiQuery({ name: 'view', type: 'string', required: false, description: 'View type: registrations, goodies, or travel', enum: ViewTypeEnum })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Registrations list with dynamic KPIs retrieved successfully. Returns same items/pagination as list-view plus dynamic KPIs based on parentFilter.',
    schema: {
      properties: {
        data: { 
          type: 'array',
          description: 'Flat array of registration records',
          items: {
            type: 'object',
            properties: {
              id: { type: 'number' },
              seekerName: { type: 'string' },
              age: { type: 'number' },
              gender: { type: 'string' },
              location: { type: 'string' },
              blessedWith: { type: 'string' },
              numberOfHDBs: { type: 'number' },
              rmComments: { type: 'string' },
              paymentStatus: { type: 'string' },
              travelPlanStatus: { type: 'string' },
              mobileNumber: { type: 'string' },
              registrationStatus: { type: 'string' }
            }
          }
        },
        tableHeaders: {
          type: 'array',
          description: 'Table column headers configuration',
          items: {
            type: 'object',
            properties: {
              key: { type: 'string', description: 'Data field key' },
              label: { type: 'string', description: 'Display label' },
              sortable: { type: 'boolean', description: 'Whether column is sortable' },
              filterable: { type: 'boolean', description: 'Whether column is filterable' },
              type: { type: 'string', description: 'Data type for formatting' }
            }
          }
        },
        pagination: {
          type: 'object',
          description: 'Pagination information',
          properties: {
            totalPages: { type: 'number' },
            pageNumber: { type: 'number' },
            pageSize: { type: 'number' },
            totalRecords: { type: 'number' },
            numberOfRecords: { type: 'number' }
          }
        },
        kpis: {
          type: 'array',
          description: 'Dynamic KPI array based on parentFilter from filter-hierarchy',
          items: {
            type: 'object',
            properties: {
              label: { type: 'string' },
              value: { type: 'number' },
              kpiCategory: { type: 'string' },
              kpiFilter: { type: 'string' },
              category: { type: 'string' },
              categoryLabel: { type: 'string' }
            }
          }
        },
        kpiCount: {
          type: 'number',
          description: 'Total number of KPIs returned'
        },
        statusCounts: {
          type: 'array',
          description: 'SubProgram-wise registration counts (Only for mahatria and shoba roles)',
          items: {
            type: 'object',
            properties: {
              subProgramId: { type: 'number', description: 'Sub-program ID' },
              subProgramName: { type: 'string', description: 'Sub-program name' },
              totalSeatsCount: { type: 'number', description: 'Total seats available in this sub-program' },
              totalRegistrations: { type: 'number', description: 'Total registrations for this sub-program' },
              approvedCount: { type: 'number', description: 'Number of approved registrations' },
              pendingCount: { type: 'number', description: 'Number of pending registrations' },
              organisationUserCount: { type: 'number', description: 'Number of organisation users registered' }
            }
          }
        },
        globalCounts: {
          type: 'object',
          description: 'Global approval status counts across all registrations (Only for mahatria and shoba roles)',
          properties: {
            holdCount: { type: 'number', description: 'Global number of registrations on hold (rejected status)' },
            ytdCount: { type: 'number', description: 'Global number of Swap Demand registrations (on_hold status)' }
          }
        }
      }
    }
  })
  @ApiResponse({ 
    status: HttpStatus.BAD_REQUEST, 
    description: 'Bad Request - parentFilter is missing, invalid, or not found in filter-hierarchy' 
  })
  @ApiResponse({ 
    status: HttpStatus.SERVICE_UNAVAILABLE, 
    description: 'Service Unavailable - filter-config service unavailable' 
  })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async newGetRegistrationsListView(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Query('parentFilter') parentFilter: string,
    @Query('sortKey') sortKey: RegistrationSortKey = RegistrationSortKey.DEFAULT_SORT_KEY,
    @Query('sortOrder') sortOrder: 'ASC' | 'DESC' = 'ASC',
    @Req() req: any,
    @Res() res: Response,
    @Query('downloadType') downloadType?: string,
    @Query('communicationType') communicationType?: CommunicationTypeEnum,
    @Query('communicationTemplateKey') communicationTemplateKey?: CommunicationTemplatesKeysEnum,
    @Query('communicationListForTemplateKey') communicationListForTemplateKey?: CommunicationTemplatesKeysEnum,
    @Query('view') view : ViewTypeEnum = ViewTypeEnum.REGISTRATIONS,
  ) {
    try {
      // Validation: parentFilter is required
      if (!parentFilter) {
        throw new InifniBadRequestException(ERROR_CODES.PARENT_FILTER_REQUIRED, null, null, '');
      }

      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const user = req.user;
      const userRoles = user?.roles || [];
      
      // Add parentFilter to parsedFilters for service compatibility
      parsedFilters.parentFilter = parentFilter;

      if (communicationTemplateKey && communicationType) {
        const result = await this.service.newGetRegistrationsListView(
          +limit,
          +offset,
          programId ? +programId : null,
          programSessionId ? +programSessionId : null,
          '',
          parsedFilters,
          userRoles,
          user?.id || null,
          downloadType,
          sortKey,
          sortOrder,
          communicationTemplateKey,
        );
        await this.service.sendCommunicationToRegistrations(
          result.data,
          communicationType,
          communicationTemplateKey,
        );

        await this.responseService.success(res, 'Successfully sent communication', result);
        return;
      } else if (
        communicationListForTemplateKey &&
        communicationListForTemplateKey === CommunicationTemplatesKeysEnum.FIRST_TIMER
      ) {
        const result = await this.service.newGetRegistrationsListView(
          +limit,
          +offset,
          programId ? +programId : null,
          programSessionId ? +programSessionId : null,
          searchText,
          parsedFilters,
          userRoles,
          user?.id || null,
          downloadType,
          sortKey,
          sortOrder,
          communicationListForTemplateKey,
        );
        const transformedData = await this.service.transformRegistrationListData(result, userRoles);
        await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, transformedData);
        return;
      }
      
      // If it's a download request, handle different download types first
      if (downloadType) {
        
        if (REPORT_CODES.includes(downloadType)) {
          // Handle report download
          const hasAccess = await this.service.validateReportAccess(userRoles, downloadType);
          if (!hasAccess) {
            throw new InifniBadRequestException(ERROR_CODES.ACCESS_DENIED_TO_REPORT, null, null, 'You do not have permission to access this report');
          }

          // Generate report with current filters
          const s3Url = await this.service.generateReport(
            downloadType,
            {
              programId: programId || undefined,
              programSessionId: programSessionId || undefined,
              searchText,
              filters: parsedFilters,
              userRoles,
              filterRegistrationsByUserId: user?.id || null
            }
          );

          await this.responseService.success(res, 'Report generated successfully', { downloadUrl: s3Url });
          return;
        }
        // If downloadType is not a report code, we need to call service for regular Excel download
      }

      // Call service method only when:
      // 1. Not a download request, OR
      // 2. Download request for regular Excel (not report codes)
      const result = await this.service.newGetRegistrationsListView(
        +limit,
        +offset,
        programId ? +programId : null,
        programSessionId ? +programSessionId : null,
        searchText,
        parsedFilters,
        userRoles,
        user?.id || null,
        downloadType,
        sortKey,
        sortOrder,
        undefined, // communicationTemplateKey
        view
      );

      // Handle regular Excel download if downloadType exists but is not a report code
      if (downloadType && !REPORT_CODES.includes(downloadType)) {
        const excelUrl = await this.service.generateExcelDownload(result.data, userRoles, downloadType);
        
        await this.responseService.success(res, 'Excel file generated successfully', {
          downloadUrl: excelUrl,
          totalRecords: result.pagination.totalRecords,
          generatedAt: new Date().toISOString()
        });
        return;
      }

      // Regular response for non-download requests
      const transformedData = await this.service.transformRegistrationListData(result, userRoles, parentFilter, parsedFilters?.kpiFilter, programId, view, parsedFilters?.kpiCategory);
      // Add kpiCount to the response
      (transformedData as any).kpiCount = Array.isArray(transformedData.kpis) ? transformedData.kpis.length : Object.keys(transformedData.kpis || {}).length;
      
      await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, transformedData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('list-view')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get registrations list with role-based KPIs and data filtering' })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Filter by program ID' })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Filter by program session ID' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters' })
  @ApiQuery({ name: 'downloadType', type: String, required: false, description: 'Download type: "all" or "filtered"' })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Registrations list with role-based KPIs retrieved successfully',
    schema: {
      properties: {
        data: { 
          type: 'array',
          description: 'Flat array of registration records',
          items: {
            type: 'object',
            properties: {
              id: { type: 'number' },
              seekerName: { type: 'string' },
              age: { type: 'number' },
              gender: { type: 'string' },
              location: { type: 'string' },
              blessedWith: { type: 'string' },
              numberOfHDBs: { type: 'number' },
              rmComments: { type: 'string' },
              paymentStatus: { type: 'string' },
              travelPlanStatus: { type: 'string' },
              mobileNumber: { type: 'string' },
              registrationStatus: { type: 'string' }
            }
          }
        },
        tableHeaders: {
          type: 'array',
          description: 'Table column headers configuration',
          items: {
            type: 'object',
            properties: {
              key: { type: 'string', description: 'Data field key' },
              label: { type: 'string', description: 'Display label' },
              sortable: { type: 'boolean', description: 'Whether column is sortable' },
              filterable: { type: 'boolean', description: 'Whether column is filterable' },
              type: { type: 'string', description: 'Data type for formatting' }
            }
          }
        },
        pagination: {
          type: 'object',
          description: 'Pagination information',
          properties: {
            totalPages: { type: 'number' },
            pageNumber: { type: 'number' },
            pageSize: { type: 'number' },
            totalRecords: { type: 'number' },
            numberOfRecords: { type: 'number' }
          }
        },
        kpis: {
          type: 'array',
          description: 'Flat KPI array for easy rendering',
          items: {
            type: 'object',
            properties: {
              label: { type: 'string' },
              value: { type: 'number' },
              kpiCategory: { type: 'string' },
              kpiFilter: { type: 'string' },
              category: { type: 'string' },
              categoryLabel: { type: 'string' }
            }
          }
        }
      }
    }
  })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getRegistrationsListView(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Req() req: any,
    @Res() res: Response,
    @Query('downloadType') downloadType?: string,
    @Query('communicationType') communicationType?: CommunicationTypeEnum,
    @Query('communicationTemplateKey') communicationTemplateKey?: CommunicationTemplatesKeysEnum,
    @Query('communicationListForTemplateKey') communicationListForTemplateKey?: CommunicationTemplatesKeysEnum,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const user = req.user;
      const userRoles = user?.roles || [];
      if(communicationTemplateKey && communicationType) {
        const result = await this.service.getAllRegistrationsWithoutPagination(
          programId ? +programId : null,
          programSessionId ? +programSessionId : null,
          searchText,
          parsedFilters,
          userRoles,
          communicationTemplateKey,
        )
        await this.service.sendCommunicationToRegistrations(result.data, communicationType, communicationTemplateKey);


        await this.responseService.success(res, 'Successfully sent communication');
        return;
      } else if (communicationListForTemplateKey && communicationListForTemplateKey === CommunicationTemplatesKeysEnum.FIRST_TIMER) {
        const result = await this.service.getUniqueRegistrationsForPrograms(
          programId ? +programId : 0,
          parsedFilters?.allocatedProgramId ?? undefined,
          parsedFilters,
          offset,
          limit
        )
        const transformedData = await this.service.transformRegistrationListData(result, userRoles);
      
        await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, transformedData);
        return;
      }
      
      const result = await this.service.getRegistrationsListView(
        +limit,
        +offset,
        programId ? +programId : null,
        programSessionId ? +programSessionId : null,
        searchText,
        parsedFilters,
        userRoles,
        user?.id || null,
        downloadType 
      );
  
      // If it's a download request, generate Excel and return S3 URL
      if (downloadType) {
        const excelUrl = await this.service.generateExcelDownload(result.data, userRoles, downloadType);
        
        await this.responseService.success(res, 'Excel file generated successfully', {
          downloadUrl: excelUrl,
          totalRecords: result.pagination.totalRecords,
          generatedAt: new Date().toISOString()
        });
        return;
      }

      // Regular response for non-download requests
      const transformedData = await this.service.transformRegistrationListData(result, userRoles);
      
      await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, transformedData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
  
  @Get('mahatria-kpis')
  @HttpCode(HttpStatus.OK)
  @Roles('mahatria', 'admin', 'shoba') // Restrict to mahatria and admin roles only
  @ApiOperation({ summary: 'Get Mahatria-specific KPIs for program allocation management with registration data' })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: true, description: 'Program ID for KPI generation' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text for filtering' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters for KPI calculation' })
  @ApiQuery({
    name: 'sortKey',
    type: String,
    required: false,
    description: 'Field name to sort by. Available fields: seekerName, profileUrl, age, gender, location, averageRating, blessedWith, numberOfHDBs, id, fullName, dob, city, noOfHDBs, mobileNumber, emailAddress, blessedDate, cancelledDate, holdDate, unassignedDate, swapDemandDate, swapRequestDate, pendingDate, statusDateTime. Note: Nested relation fields are not supported for sorting.',
  })
  @ApiQuery({
    name: 'sortOrder',
    enum: ['ASC', 'DESC'],
    required: false,
    description: 'Sort direction (ASC or DESC)',
  })
  @ApiResponse({ 
    status: HttpStatus.OK, 
    description: 'Mahatria KPIs with registration data retrieved successfully',
    schema: {
      properties: {
        data: { 
          type: 'array',
          description: 'Array of registration records (filtered for mahatria role)'
        },
        pagination: {
          type: 'object',
          description: 'Pagination information',
          properties: {
            totalPages: { type: 'number' },
            pageNumber: { type: 'number' },
            pageSize: { type: 'number' },
            totalRecords: { type: 'number' },
            numberOfRecords: { type: 'number' }
          }
        },
        kpis: {
          type: 'object',
          properties: {
            mahatria: {
              type: 'object',
              properties: {
                groupedProgramMetrics: {
                  type: 'object',
                  properties: {
                    isGroupedProgram: { type: 'boolean' },
                    primaryProgramId: { type: 'number' },
                    primaryProgramName: { type: 'string' },
                    allocated: {
                      type: 'object',
                      properties: {
                        categoryLabel: { type: 'string' },
                        programs: { type: 'array' },
                        totals: {
                          type: 'object',
                          properties: {
                            holdCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            rejectCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            }
                          }
                        }
                      }
                    },
                    unallocated: {
                      type: 'object',
                      properties: {
                        categoryLabel: { type: 'string' },
                        programs: { type: 'array' },
                        totals: {
                          type: 'object',
                          properties: {
                            totalMaleCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            totalFemaleCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            totalUnallocatedCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            totalMahatriaChoiceCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' },
                                organisationUserCount: { type: 'number' }
                              }
                            },
                            totalOrganisationUserCount: {
                              type: 'object',
                              properties: {
                                label: { type: 'string' },
                                value: { type: 'number' },
                                kpiCategory: { type: 'string' },
                                kpiFilter: { type: 'string' }
                              }
                            }
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.FORBIDDEN, description: 'Access denied - Mahatria role required' })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getMahatriaKPIs(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId', ParseIntPipe) programId: number,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Query('sortKey') sortKey: RegistrationSortKey = RegistrationSortKey.DEFAULT_SORT_KEY,
    @Query('sortOrder') sortOrder: 'ASC' | 'DESC' = 'ASC',
    @Req() req: any,
    @Res() res: Response,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};

      const payloadParseFilters = {...parsedFilters};

      // add in parsed Filters add excludeCancelledRegistrations as true
      parsedFilters.excludeCancelledRegistrations = true;
      const user = req.user;
      
      const data = await this.service.getMahatriaKPIs(
        +limit,
        +offset,
        programId,
        searchText,
        parsedFilters,
        user,
        sortKey,
        sortOrder,
      );

      // Transform data (includes registration status transformation to avoid double looping)
      const transformedData = await this.service.transformMahatriaKPIData(data, user.roles, payloadParseFilters);
      const responseData = {
        data: transformedData,
        pagination: data.pagination,
        kpis: data.kpis,
        filters: data.filters,
      };
      await this.responseService.success(res, 'Mahatria KPIs with registration data retrieved successfully', responseData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('drafts')
  @HttpCode(HttpStatus.OK)
  @Roles('relational_manager', 'shoba', 'admin','rm_support')
  @ApiOperation({ summary: 'Get save-as-draft registrations with search and filters' })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: 'Number of records per page (default: 10)' })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: 'Offset for pagination (default: 0)' })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: 'Filter by program ID' })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Filter by program session ID' })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: 'Search text' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: 'Additional filters' })
  @ApiQuery({ name: 'downloadType', type: String, required: false, description: 'Download type: "drafts_report" for Excel download' })
  @ApiQuery({
    name: 'sortKey',
    type: String,
    required: false,
    description: 'Field name to sort by (matches table header key)',
  })
  @ApiQuery({
    name: 'sortOrder',
    enum: ['ASC', 'DESC'],
    required: false,
    description: 'Sort direction (ASC or DESC)',
  })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getSaveDraftRegistrations(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('searchText') searchText = '',
    @Query('filters') filters = '',
    @Query('downloadType') downloadType = '',
    @Query('sortKey') sortKey = 'id',
    @Query('sortOrder') sortOrder: 'ASC' | 'DESC' = 'ASC',
    @Req() req: any,
    @Res() res: Response,
  ) {
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const user = req.user;
      const userRoles = user?.roles || [];

      // Handle download request
      if (downloadType === 'drafts_report') {
        // Validate report access
        const hasAccess = await this.service.validateReportAccess(userRoles, 'drafts_report');
        if (!hasAccess) {
          throw new InifniBadRequestException(ERROR_CODES.PQ_VALIDATION_FAILED, null, null, 'You do not have permission to access this report');
        }

        // Generate and return download URL
        const s3Url = await this.service.generateReport('drafts_report', {
          programId: programId ? +programId : undefined,
          programSessionId: programSessionId ? +programSessionId : undefined,
          searchText: '',
          filters: parsedFilters,
          userRoles,
          filterRegistrationsByUserId: user?.id || null,
        });

        return await this.responseService.success(res, 'Report generated successfully', { downloadUrl: s3Url });
      }

      const data = await this.service.getSaveDraftRegistrations(
        +limit,
        +offset,
        programId ? +programId : null,
        programSessionId ? +programSessionId : null,
        searchText,
        parsedFilters,
        userRoles,
        user?.id || null,
        sortKey,
        sortOrder,
      );

      const transformedDataDraft = await this.service.transformDraftRegistrationListData(data, userRoles);
      await this.responseService.success(res, registrationMessages.REGISTRATIONS_RETRIEVED, transformedDataDraft);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_BY_ID })
  @ApiParam({ name: 'id', description: 'Registration ID', type: Number })
  @ApiQuery({name: 'isAdmin', type: Boolean, required: false, description: 'Flag to indicate from admin'})
  @ApiQuery({ name: 'selectedOption', type: String, required: false, description: 'Selected option for additional data' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number,@Query('isAdmin') isAdmin: string, @Query('selectedOption') selectedOption: string, @Res() res: Response, @Req() req: any) {
    const user = req.user;
    const userRoles = user.userRoleMaps;
    const isMahatria = userRoles.some(urm => urm.role.name === 'mahatria');
    const isShoba = userRoles.some(urm => urm.role.name === 'shoba');
    const viewer = userRoles.some(urm => urm.role.name === 'viewer');
    const adminContext = isAdmin === 'true';
    try {
      const data = await this.service.findOne(id);
      let finalData: any = data;
      if (finalData) {
        // if registration is in pending state and approval status is pending, then set registration status to pending_approval
        if (finalData.registrationStatus === RegistrationStatusEnum.PENDING && finalData.approvals[0].approvalStatus === ApprovalStatusEnum.PENDING) {
          finalData.registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
        }
        // use stored average rating from database
        if (finalData.averageRating !== null && finalData.averageRating !== undefined) {
          finalData.averageRating = Number(finalData.averageRating).toFixed(2); // format stored rating to string like '4.00'
        } else {
          finalData.averageRating = null;
        }
        
        finalData.isOrganizationUser = finalData?.user?.userType === UserTypeEnum.ORG;
        
        // Add previous rating and review from user profile extension
        if (finalData.user?.id) {
          try {
            const userProfileExtension = await this.userProfileExtensionService.findByUserId(finalData.user.id);
            
            // Add prevRating with rating and review from user profile extension
            const prevRating = {
              rating: userProfileExtension?.rmRating || null,
              review: userProfileExtension?.rmReview || null,
              rmContactId: userProfileExtension?.infinitheismContactId || null,
              rmContactName: userProfileExtension?.contactPerson || null,
              hdbYear: userProfileExtension?.hdbYear || null

            };
            
            finalData.prevRating = prevRating;
          } catch (error) {
            // If user profile extension not found, set prevRating to null values
            finalData.prevRating = {
              rating: null,
              review: null,
              rmContactId: null,
              rmContactName: null,
              hdbYear: null
            };
          }
        } else {
          finalData.prevRating = {
            rating: null,
            review: null,
            rmContactId: null,
            rmContactName: null,
            hdbYear: null
          };
        }
      }
      const latestApproval = finalData.approvals && finalData.approvals.length > 0 ? finalData.approvals[0] : null;
      
      // Sort swapsRequests by id in descending order to get the latest one
      if (finalData.swapsRequests && finalData.swapsRequests.length > 0) {
        finalData.swapsRequests = [...finalData.swapsRequests].sort((a, b) => b.id - a.id);
      }
      const latestSwapRequest = finalData.swapsRequests && finalData.swapsRequests.length > 0
        ? finalData.swapsRequests.sort((a: any, b: any) => b.id - a.id)[0]
        : null;

      // Add statusDateTime and statusDateTimeLabel to finalData
      finalData.statusDateTime = getRegistrationStatusDateTime(finalData, latestApproval, latestSwapRequest, undefined, selectedOption, true);
      finalData.statusDateTimeLabel = getStatusDateTimeLabel(finalData, latestApproval, latestSwapRequest, undefined, selectedOption, true);
      const questionResponses = await this.service.getQuestionResponses(id);
      let filteredResponses = questionResponses;
      if (data && adminContext) {
        // fetch user filled questions for Mahatria's Questions section
        const allowedRoles = ['mahatria', 'admin', 'shoba'];
        if (
          req.user && adminContext &&
          !req.user.userRoleMaps.some((role) => allowedRoles.includes(role.role.name))
        ) {
          // filter out the questions with section 4
          filteredResponses = questionResponses.filter(question => question.sectionKey != "FS_MAHATRIAQUESTIONS");
          }
      } else if (data && (isMahatria || isShoba)) {
          filteredResponses = questionResponses.filter(question => question.sectionKey === 'FS_MAHATRIAQUESTIONS');
      }
      
      finalData = {
        ...data,
        questionResponses: filteredResponses,
      };
      const allowedSaveDraftRoles = ['viewer'];
      if (data && !userRoles.some(role => allowedSaveDraftRoles.includes(role.role.name))) {
        let hasSaveAsDraft =  false
        // if travelPlanStatus map contains save_as_draft, then set travelPlanStatus to [] remove that same in travelInfo also
        if (finalData.travelPlans && finalData.travelPlans.length> 0 && finalData.travelPlans.some(tp => tp.travelPlanStatus === 'save_as_draft')) {
          finalData.travelPlans = finalData.travelPlans?.filter(tp => tp.travelPlanStatus !== 'save_as_draft');
          hasSaveAsDraft = true;
        }
        if (finalData.travelInfo && finalData.travelInfo.length > 0 && finalData.travelInfo.some(ti => ti.travelInfoStatus === 'save_as_draft')) {
          finalData.travelInfo = finalData.travelInfo?.filter(ti => ti.travelInfoStatus !== 'save_as_draft');
           hasSaveAsDraft = true;
        }
        // filter section "FS_TRAVELPLAN" then ignore that questions if any of these status is save_as_draft
        if (hasSaveAsDraft && finalData.questionResponses && finalData.questionResponses.length > 0) {
          finalData.questionResponses = finalData.questionResponses?.filter(qr => qr.sectionKey !== 'FS_TRAVELPLAN');
        }
      }
      // if (viewer) {
      //   delete finalData.swapsRequests;
      //   delete finalData.ratings
      // }
      await this.responseService.success(res, registrationMessages.REGISTRATION_RETRIEVED, finalData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id/questions')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_QA_REQUEST_RECEIVED })
  @ApiParam({ name: 'id', description: 'Registration ID', type: Number })
  @ApiQuery({ name: 'isAdmin', type: Boolean, required: false, description: 'Flag to indicate from admin' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getQuestionResponses(@Param('id', ParseIntPipe) id: number,
    @Query('isAdmin') isAdmin: string,
   @Res() res: Response, @Req() req: any) {
    try {
      const registration = await this.service.findOne(id);
      const adminContext = isAdmin === 'true';
      if (!registration) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_NOT_FOUND, null, null, `Registration with ID ${id} not found`);
      }
      let data = await this.service.getQuestionResponses(id);
      // Only allow 'mahatria', 'admin', or to see section 4 questions
      const allowedRoles = ['mahatria', 'admin', 'shoba'];
      if (
        req.user && adminContext &&
        !req.user.userRoleMaps.some((role) => allowedRoles.includes(role.role.name))
      ) {
        // filter out the questions with section 4
        data = data.filter(question => question.sectionKey != "FS_MAHATRIAQUESTIONS");
      }

      const allowedSaveDraftRoles = ['viewer'];
      if (
        req.user &&
        !req.user.userRoleMaps.some((role) => allowedSaveDraftRoles.includes(role.role.name))
      ) {
        let hasSaveAsDraft =  false
        // if travelPlanStatus map contains save_as_draft, then set travelPlanStatus to [] remove that same in travelInfo also
        if (registration.travelPlans && registration.travelPlans.length> 0 && registration.travelPlans.some(tp => tp.travelPlanStatus === 'save_as_draft')) {
          registration.travelPlans = registration.travelPlans?.filter(tp => tp.travelPlanStatus !== 'save_as_draft');
          hasSaveAsDraft = true;
        }
        if (registration.travelInfo && registration.travelInfo.length > 0 && registration.travelInfo.some(ti => ti.travelInfoStatus === 'save_as_draft')) {
          registration.travelInfo = registration.travelInfo?.filter(ti => ti.travelInfoStatus !== 'save_as_draft');
           hasSaveAsDraft = true;
        }
        if (hasSaveAsDraft && data && data.length > 0) {
         data = data?.filter(question => question.sectionKey != "FS_TRAVELPLAN");
        }
      }
      // filter empty question responses
      data = data.filter(question => question && question.answer && question.answer.trim() !== '');
      await this.responseService.success(res, registrationMessages.QA_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Get dashboard data for a specific program
   * @param programId The ID of the program
   * @param res The response object
   * @returns The dashboard data for the program
   */
  @Get('program/:programId/dashboard')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_PROGRAM_DASHBOARD })
  @ApiParam({ name: 'programId', description: 'Program ID', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findByProgramId(@Param('programId', ParseIntPipe) programId: number, @Res() res: Response, @Req() req: any) {
    try {
      const userRoles = req.user?.roles
      const mainProgramDimensions = RoleKPIs[userRoles[0]] || MainProgramDimensions;
      const subProgramDimensions = [];
      const data = await this.service.getProgramDashboardData({programId, mainProgramDimensions, subProgramId: undefined, subProgramDimensions, rmId: userRoles.includes('relational_manager') ? req.user.id : undefined, userRole: userRoles[0]});
      await this.responseService.success(res, registrationMessages.DASHBOARD_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Get dashboard data for a specific program
   * @param programId The ID of the program
   * @param subProgramId The ID of the sub-program
   * @param res The response object
   */
  @Get('program/:programId/dashboard/sub-program/:subProgramId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.GET_SUB_PROGRAM_DASHBOARD })
  @ApiParam({ name: 'programId', description: 'Program ID', type: Number })
  @ApiParam({ name: 'subProgramId', description: 'Sub Program ID', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findBySubProgramId(@Param('programId', ParseIntPipe) programId: number, @Param('subProgramId', ParseIntPipe) subProgramId: number, @Res() res: Response, @Req() req: any) {
    try {
      const userRoles = req.user?.roles
      const mainProgramDimensions = RoleKPIs[userRoles[0]] || MainProgramDimensions;
      const subProgramDimensions = SubProgramDimensions;
      const data = await this.service.getProgramDashboardData({programId, mainProgramDimensions, subProgramId, subProgramDimensions, rmId: userRoles.includes('relational_manager') ? req.user.id : undefined, userRole: userRoles[0]});
      await this.responseService.success(res, registrationMessages.DASHBOARD_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationMessages.DELETE_BY_ID })
  @ApiParam({ name: 'id', description: registrationMessages.REGISTRATION_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Req() req: any, @Res() res: Response) {
    try {
      const user = req.user;
      if (!user) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
      }
      await this.service.remove(id, user);
      await this.responseService.success(res, registrationMessages.REGISTRATION_DELETED, { id });
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put(':id/parental-form')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Update parental consent form details for a registration' })
  @ApiParam({ name: 'id', description: 'Registration ID', type: Number })
  @ApiBody({ type: UpdateParentalConsentDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  async updateParentalConsentFormStatus(
    @Param('id', ParseIntPipe) id: number,
    @Body() updateParentalConsentDto: UpdateParentalConsentDto,
    @Req() req: any,
    @Res() res: Response
  ) {
    try {
      const user = req.user;
      if (!user) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          'User not found in request',
        );
      }
      const updatedRegistration = await this.service.updateParentalConsentForm(
        id,
        updateParentalConsentDto,
        user,
      );
      await this.responseService.success(
        res,
        'Parental consent form details updated successfully',
        updatedRegistration,
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Clear all registrations for a specific program (environment-specific)
   * Only accessible by admin users
   */
  @Delete('clear-all/:programId')
  @Roles(ROLE_VALUES.ADMIN)
  @ApiOperation({
    summary: 'Clear all registrations for a program (Environment-specific)',
    description: `
      Soft deletes all registrations for a specific program. 
      - Only works in development/qa environments
      - Requires ENABLE_CLEAR_REGISTRATIONS=true environment variable
      - Only accessible by admin role
      - Soft deletes all related records
    `
  })
  @ApiParam({ 
    name: 'programId', 
    type: 'number',
    description: 'ID of the program to clear all registrations for'
  })
  @ApiResponse({
    status: 200,
    description: 'Registrations cleared successfully',
    schema: {
      type: 'object',
      properties: {
        message: { type: 'string' },
        clearedCount: { type: 'number' },
        programId: { type: 'number' },
        environment: { type: 'string' }
      }
    }
  })
  @ApiResponse({ status: 400, description: 'Operation not allowed in this environment or not enabled' })
  @ApiResponse({ status: 403, description: 'Insufficient permissions' })
  @ApiResponse({ status: 404, description: 'Program not found' })
  async clearAllRegistrationsForProgram(
    @Param('programId', ParseIntPipe) programId: number,
    @Req() req: any,
    @Res() res: Response
  ) {
    try {
      const user = req.user;
      if (!user) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
      }
      
      const result = await this.service.clearAllRegistrationsForProgram(programId, user);
      await this.responseService.success(res, result.message, result);
    } catch (error) {
      this.logger.error('Error clearing registrations:', error);
      handleControllerError(res, error);
    }
  }

  /**
   * Update signed URLs for registrations in travel info and travel plan
   * Can update for specific registration or all registrations in a program
   */
  @Put('update-signed-urls')
  @Roles(ROLE_VALUES.ADMIN)
  @ApiOperation({
    summary: 'Update signed URLs for registration travel documents',
    description: `
      Regenerates signed URLs in registration_travel_info and registration_travel_plan tables.
      - Always regenerates signed URLs (since they expire after 7 days)
      - Updates profile picture, video, ID pictures, passport, visa, onward/return journey tickets
      - Can update single registration or all registrations in a program
      - Skips URLs from excluded domains (zoho, entrainment)
      - Only accessible by admin role
      
      Note: Signed URLs are always regenerated because they expire after 7 days (AWS S3 limit).
    `
  })
  @ApiQuery({
    name: 'programId',
    type: 'number',
    description: 'Program ID for which to update signed URLs',
    required: true
  })
  @ApiQuery({
    name: 'registrationId',
    type: 'number',
    description: 'Optional registration ID. If provided, updates only this registration. Otherwise updates all registrations in the program.',
    required: false
  })
  @ApiResponse({
    status: 200,
    description: 'Signed URLs updated successfully',
    schema: {
      type: 'object',
      properties: {
        message: { type: 'string' },
        programId: { type: 'number' },
        registrationId: { type: 'number' },
        updatedCount: {
          type: 'object',
          properties: {
            registration: { type: 'number' },
            travelInfo: { type: 'number' },
            travelPlan: { type: 'number' }
          }
        },
        details: {
          type: 'object',
          properties: {
            registration: { type: 'array', items: { type: 'string' } },
            travelInfo: { type: 'array', items: { type: 'string' } },
            travelPlan: { type: 'array', items: { type: 'string' } }
          }
        }
      }
    }
  })
  @ApiResponse({ status: 400, description: 'Invalid request parameters' })
  @ApiResponse({ status: 403, description: 'Insufficient permissions' })
  @ApiResponse({ status: 404, description: 'Program or registration not found' })
  async updateSignedUrls(
    @Query('programId', ParseIntPipe) programId: number,
    @Query('registrationId') registrationId: string | undefined,
    @Req() req: any,
    @Res() res: Response
  ) {
    try {
      const user = req.user;
      if (!user) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
      }

      const parsedRegistrationId = registrationId ? parseInt(registrationId, 10) : undefined;
      
      if (registrationId && isNaN(parsedRegistrationId!)) {
        throw new InifniBadRequestException(
          ERROR_CODES.PQ_VALIDATION_FAILED,
          null,
          null,
          'Invalid registration ID provided'
        );
      }

      const result = await this.service.updateSignedUrlsForRegistrations(
        programId,
        parsedRegistrationId,
        user
      );
      
      await this.responseService.success(res, result.message, result);
    } catch (error) {
      this.logger.error('Error updating signed URLs:', error);
      handleControllerError(res, error);
    }
  }
}
