import { Injectable } from '@nestjs/common';
import { DataSource, Repository } from 'typeorm';
import { InjectDataSource, InjectRepository } from '@nestjs/typeorm';
import { CreateRegistrationTravelInfoDto } from './dto/create-registration-travel-info.dto';
import { UpdateRegistrationTravelInfoDto } from './dto/update-registration-travel-info.dto';
import { RegistrationTravelInfoRepository } from './registration-travel-info.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CommonDataService } from 'src/common/services/commonData.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { registrationTravelInfoMessages } from 'src/common/constants/strings-constants';
import { RegistrationTravelInfo } from 'src/common/entities';

@Injectable()
export class RegistrationTravelInfoService {
  private repo: RegistrationTravelInfoRepository;

  constructor(
    @InjectDataSource()
    private readonly dataSource: DataSource,
    @InjectRepository(RegistrationTravelInfo)
    private readonly registrationTravelInfoRepo: Repository<RegistrationTravelInfo>,
    private readonly logger: AppLoggerService,
    private readonly commonDataService: CommonDataService,
  ) {
    this.repo = new RegistrationTravelInfoRepository(
      this.registrationTravelInfoRepo,
      this.commonDataService,
      this.logger,
    );
  }

  async create(dto: CreateRegistrationTravelInfoDto) {
    this.logger.log(registrationTravelInfoMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationTravelInfoRepository(
          manager.getRepository(RegistrationTravelInfo),
          this.commonDataService,
          this.logger,
        );
        return await repository.createEntity(dto);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    this.logger.log(registrationTravelInfoMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationTravelInfoMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      return await this.repo.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateRegistrationTravelInfoDto) {
    this.logger.log(registrationTravelInfoMessages.UPDATE_REQUEST_RECEIVED);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationTravelInfoRepository(
          manager.getRepository(RegistrationTravelInfo),
          this.commonDataService,
          this.logger,
        );
        return await repository.updateEntity(id, dto);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    this.logger.log(registrationTravelInfoMessages.DELETE_REQUEST_RECEIVED);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationTravelInfoRepository(
          manager.getRepository(RegistrationTravelInfo),
          this.commonDataService,
          this.logger,
        );
        return await repository.remove(id);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_DELETE_FAILED, error);
    }
  }
}
