import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { RegistrationPaymentDetail } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { CreateRegistrationPaymentDetailDto } from './dto/create-registration-payment-detail.dto';
import { UpdateRegistrationPaymentDetailDto } from './dto/update-registration-payment-detail.dto';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';

@Injectable()
export class RegistrationPaymentDetailRepository {
  constructor(
    @InjectRepository(RegistrationPaymentDetail)
    private readonly repo: Repository<RegistrationPaymentDetail>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  async createEntity(dto: CreateRegistrationPaymentDetailDto) {
    try {
      const entity = this.repo.create({
        ...dto,
        registration: { id: dto.registrationId } as any,
        programSession: { id: dto.programSessionId } as any,
        paymentStatus: PaymentStatusEnum.DRAFT,
        createdBy: { id: dto.createdBy } as any,
        updatedBy: { id: dto.updatedBy } as any,
      });
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    try {
      const data = await this.commonDataService.get(
        this.repo,
        undefined,
        {},
        limit,
        offset,
        { id: 'ASC' },
        undefined,
        ['registration', 'programSession'],
      );
      const total = await this.repo.count();
      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    try {
      const data = await this.commonDataService.findOneById(this.repo, id, false, ['registration', 'programSession']);
      if (!data) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_NOTFOUND, null, null, id.toString());
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_FIND_BY_ID_FAILED, error);
    }
  }

  async updateEntity(id: number, dto: UpdateRegistrationPaymentDetailDto) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_NOTFOUND, null, null, id.toString());
      }
      Object.assign(entity, dto);
      if (dto.registrationId !== undefined) entity.registration = { id: dto.registrationId } as any;
      if (dto.programSessionId !== undefined) entity.programSession = { id: dto.programSessionId } as any;
      if (dto.paymentStatus !== undefined) entity.paymentStatus = dto.paymentStatus;
      entity.updatedBy = { id: dto.updatedBy } as any;
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_NOTFOUND, null, null, id.toString());
      }
      await this.repo.remove(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_DELETE_FAILED, error);
    }
  }
}
