import { Injectable } from '@nestjs/common';
import { DataSource, Repository } from 'typeorm';
import { InjectDataSource, InjectRepository } from '@nestjs/typeorm';
import { CreateRegistrationInvoiceDetailDto } from './dto/create-registration-invoice-detail.dto';
import { UpdateRegistrationInvoiceDetailDto } from './dto/update-registration-invoice-detail.dto';
import { RegistrationInvoiceDetailRepository } from './registration-invoice-detail.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CommonDataService } from 'src/common/services/commonData.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { registrationInvoiceDetailMessages } from 'src/common/constants/strings-constants';
import { RegistrationInvoiceDetail } from 'src/common/entities';

@Injectable()
export class RegistrationInvoiceDetailService {
  private repo: RegistrationInvoiceDetailRepository;

  constructor(
    @InjectDataSource()
    private readonly dataSource: DataSource,
    @InjectRepository(RegistrationInvoiceDetail)
    private readonly registrationInvoiceDetailRepo: Repository<RegistrationInvoiceDetail>,
    private readonly logger: AppLoggerService,
    private readonly commonDataService: CommonDataService,
  ) {
    this.repo = new RegistrationInvoiceDetailRepository(
      this.registrationInvoiceDetailRepo,
      this.commonDataService,
      this.logger,
    );
  }

  async create(dto: CreateRegistrationInvoiceDetailDto) {
    this.logger.log(registrationInvoiceDetailMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationInvoiceDetailRepository(
          manager.getRepository(RegistrationInvoiceDetail),
          this.commonDataService,
          this.logger,
        );
        return await repository.createEntity(dto);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    this.logger.log(registrationInvoiceDetailMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationInvoiceDetailMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      return await this.repo.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateRegistrationInvoiceDetailDto) {
    this.logger.log(registrationInvoiceDetailMessages.UPDATE_REQUEST_RECEIVED);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationInvoiceDetailRepository(
          manager.getRepository(RegistrationInvoiceDetail),
          this.commonDataService,
          this.logger,
        );
        return await repository.updateEntity(id, dto);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    this.logger.log(registrationInvoiceDetailMessages.DELETE_REQUEST_RECEIVED);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationInvoiceDetailRepository(
          manager.getRepository(RegistrationInvoiceDetail),
          this.commonDataService,
          this.logger,
        );
        return await repository.remove(id);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_DELETE_FAILED, error);
    }
  }
}
