import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  ParseIntPipe,
  Post,
  Put,
  Query,
  Res,
  UseGuards,
  ValidationPipe,
} from '@nestjs/common';
import { Response } from 'express';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiSecurity, ApiTags } from '@nestjs/swagger';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { registrationInvoiceDetailMessages, paginationConstants, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { RegistrationInvoiceDetailService } from './registration-invoice-detail.service';
import { CreateRegistrationInvoiceDetailDto } from './dto/create-registration-invoice-detail.dto';
import { UpdateRegistrationInvoiceDetailDto } from './dto/update-registration-invoice-detail.dto';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('registration-invoice-detail')
@Controller('registration-invoice-detail')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class RegistrationInvoiceDetailController {
  constructor(
    private readonly service: RegistrationInvoiceDetailService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) { }
  
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: registrationInvoiceDetailMessages.CREATE_NEW })
  @ApiBody({ type: CreateRegistrationInvoiceDetailDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CreateRegistrationInvoiceDetailDto,
    @Res() res: Response,
  ) {
    this.logger.log(registrationInvoiceDetailMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      const data = await this.service.create(dto);
      await this.responseService.success(res, registrationInvoiceDetailMessages.INVOICE_CREATED, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationInvoiceDetailMessages.GET_ALL })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Res() res: Response,
  ) {
    this.logger.log(registrationInvoiceDetailMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      const data = await this.service.findAll(+limit, +offset);
      await this.responseService.success(res, registrationInvoiceDetailMessages.INVOICES_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationInvoiceDetailMessages.GET_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: registrationInvoiceDetailMessages.INVOICE_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(registrationInvoiceDetailMessages.FIND_ONE_REQUEST_RECEIVED, { id });
    try {
      const data = await this.service.findOne(id);
      await this.responseService.success(res, registrationInvoiceDetailMessages.INVOICE_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationInvoiceDetailMessages.UPDATE_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: registrationInvoiceDetailMessages.INVOICE_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  @ApiBody({ type: UpdateRegistrationInvoiceDetailDto })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdateRegistrationInvoiceDetailDto,
    @Res() res: Response,
  ) {
    this.logger.log(registrationInvoiceDetailMessages.UPDATE_REQUEST_RECEIVED, { id, dto });
    try {
      const data = await this.service.update(id, dto);
      await this.responseService.success(res, registrationInvoiceDetailMessages.INVOICE_UPDATED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: registrationInvoiceDetailMessages.DELETE_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: registrationInvoiceDetailMessages.INVOICE_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(registrationInvoiceDetailMessages.DELETE_REQUEST_RECEIVED, { id });
    try {
      await this.service.remove(id);
      await this.responseService.success(res, registrationInvoiceDetailMessages.INVOICE_DELETED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
