import { Injectable } from '@nestjs/common';
import { RegistrationGroupingRepository } from './registration-grouping.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import {
  CreateRegistrationGroupDto,
  CreateRegistrationPairDto,
  AddRegistrationsToGroupDto,
  AddRegistrationsToPairDto,
} from './dto/create-registration-grouping.dto';
import {
  RemoveRegistrationsDto,
  UpdateRegistrationGroupDto,
  UpdateRegistrationPairDto,
} from './dto/update-registration-grouping.dto';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { ProgramService } from 'src/program/program.service';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { LookupDataRepository } from 'src/lookup-data/lookup-data.repository';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { RegistrationGroup } from 'src/common/entities';

/**
 * Service class for managing registration grouping and pairing business logic
 * Handles validation, business rules, and orchestrates repository operations
 */
@Injectable()
export class RegistrationGroupingService {
  constructor(
    private readonly repository: RegistrationGroupingRepository,
    private readonly logger: AppLoggerService,
    private readonly programService: ProgramService,
    private readonly lookupDataRepository: LookupDataRepository,
  ) {}

  // ========== REGISTRATION PAIR SERVICES ==========

  /**
   * Create a new registration pair with business validation
   * @param dto - Registration pair creation data
   * @returns Created registration pair details
   */
  async createRegistrationPair(dto: CreateRegistrationPairDto) {
    this.logger.log('Registration pair creation request received', { dto });
    
    try {
      // Validate program exists and is active
      const program = await this.programService.findOne(dto.programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          dto.programId.toString()
        );
      }

      // Validate sub-program if provided
      if (dto.subProgramId) {
        const subProgram = await this.programService.findOne(dto.subProgramId);
        if (!subProgram || subProgram.primaryProgramId !== dto.programId) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            null,
            dto.subProgramId.toString()
          );
        }
      }
      
      this.logger.log('Program and sub-program validated successfully', {
        programId: dto.programId,
        subProgramId: dto.subProgramId
      });

      // Validate registrations exist and are not already paired
      // try {
        await this.repository.validateRegistrationsForPairingOrGrouping(
          dto.registrationIds,
          dto.programId,
          dto.subProgramId,
          true,
        );
      // } 
      // catch (validationError) {
      //   console.log('Validation error caught in service:', validationError);
      //   this.logger.error('Registration validation error:', '', {
      //     message: validationError.message,
      //     stack: validationError.stack,
      //     registrationIds: dto.registrationIds,
      //   });
      //   throw new InifniBadRequestException(
      //     ERROR_CODES.REGISTRATION_PAIRING_OR_GROUPING_VALIDATION_FAILED,
      //     validationError,
      //     null,
      //     'Validation failed for registration pairing or grouping',
      //   );
      // }

      const roomOccupancy = await this.repository.getMaxRoomOccupancyWithCount(
        dto.registrationIds.length,
        dto.programId,
        dto.subProgramId,
      );

      if (!roomOccupancy.occupancy || roomOccupancy.occupancy === 0) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_PAIRING_FAILED,
          null,
          null,
          'Room occupancy not found'
        );
      }

      // Check if max allowed pairs exceeded
      const maxAllowedPairs = await this.repository.getPairCountsWithMaxCount(
        dto.registrationIds.length,
        dto.programId,
        dto.subProgramId,
      );

      if (
        maxAllowedPairs &&
        maxAllowedPairs.totalWithMaxCount >= roomOccupancy.totalCountWithOccupancy
      ) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_PAIRING_EXCEEDS_MAX_OCCUPANCY,
          null,
          null,
          `Cannot create more pairs. Max allowed pairs of ${roomOccupancy.occupancy} reached.`
        );
      }

      // Fetch pair code from lookup data
      const lookupPairCode = await this.lookupDataRepository.findActiveByCategoryAndLabel(
        'PAIR_CODE',
        dto.registrationIds.length.toString(),
      );
      
      if (!lookupPairCode) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAIRING_LOOKUP_DATA_NOTFOUND,
          null,
          null,
          dto.registrationIds.length.toString()
        );
      }
      
      this.logger.log('Pair code lookup data found', { lookupPairCode });

      // Get next sequence number
      const sequenceNumber = await this.repository.getNextPairingSequenceNumber(
        dto.programId,
        lookupPairCode.key,
      );
      
      dto.pairCode = lookupPairCode.key;
      await this.repository.createRegistrationPair(dto, sequenceNumber);
      
      this.logger.log('Registration pair created successfully', {
        registrationIds: dto.registrationIds,
        programId: dto.programId,
        pairCode: lookupPairCode.key,
        sequenceNumber
      });

      return {
        message: 'Registration pair created successfully',
        registrationIds: dto.registrationIds,
        programId: dto.programId,
        pairCode: lookupPairCode.key,
        sequenceNumber: sequenceNumber,
      };
    } catch (error) {
      this.logger.error('Error creating registration pair', error.stack, { dto });
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAIRING_SAVE_FAILED, error);
    }
  }

  /**
   * Find registration pair by ID
   * @param id - Registration pair ID
   * @returns Registration pair with relations
   */
  async findRegistrationPairById(id: number) {
    this.logger.log('Getting registration pair by ID', { id });
    
    try {
      const pair = await this.repository.findRegistrationPairById(id);
      
      if (!pair) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_PAIRING_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      
      this.logger.log('Registration pair retrieved successfully', { id });
      return pair;
    } catch (error) {
      this.logger.error('Error finding registration pair', error.stack, { id });
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAIRING_GET_FAILED, error);
    }
  }

  /**
   * Delete a registration pair (soft delete)
   * @param id - Registration pair ID
   * @param deletedBy - ID of user deleting the pair
   * @returns Deletion result
   */
  async deleteRegistrationPair(id: number, deletedBy: number) {
    this.logger.log('Delete registration pair request received', { id, deletedBy });
    
    try {
      const pair = await this.repository.findRegistrationPairById(id);
      
      if (!pair) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_PAIRING_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      
      await this.repository.deleteRegistrationPair(id, deletedBy);
      
      this.logger.log('Registration pair deleted successfully', { id, deletedBy });
      return { message: 'Registration pair deleted successfully', id };
    } catch (error) {
      this.logger.error('Error deleting registration pair', error.stack, { id, deletedBy });
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAIRING_DELETE_FAILED, error);
    }
  }

  // ========== REGISTRATION GROUP SERVICES ==========

  /**
   * Create a new registration group with business validation
   * @param dto - Registration group creation data
   * @returns Created registration group details
   */
  async createRegistrationGroup(dto: CreateRegistrationGroupDto) {
    this.logger.log('Registration group creation request received', { dto });
    
    try {
      // Validate program exists
      const program = await this.programService.findOne(dto.programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          dto.programId.toString()
        );
      }

      // Validate sub-program if provided
      if (dto.subProgramId) {
        const subProgram = await this.programService.findOne(dto.subProgramId);
        if (!subProgram || subProgram.primaryProgramId !== dto.programId) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            dto.subProgramId.toString(),
            'Sub-program not found or does not belong to the specified program',
          );
        }
      }
      this.logger.log('Program and sub-program validated');

      // Validate registrations exist and are not already paired
      try {
        await this.repository.validateRegistrationsForPairingOrGrouping(
          dto.registrationIds,
          dto.programId,
          dto.subProgramId,
          false,
        );
      } catch (validationError) {
        this.logger.error('Registration validation error:', '', {
          message: validationError.message,
          stack: validationError.stack,
          registrationIds: dto.registrationIds,
        });
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_PAIRING_OR_GROUPING_VALIDATION_FAILED,
          validationError,
          null,
          'Validation failed for registration pairing or grouping',
        );
      }

      const result = await this.repository.createRegistrationGroup(dto);
      
      this.logger.log('Registration group created successfully', {
        groupId: result.id,
        programId: dto.programId,
        registrationCount: dto.registrationIds.length
      });

      return {
        message: 'Registration group created successfully',
        data: result,
      };
    } catch (error) {
      this.logger.error('Error creating registration group', error.stack, { dto });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_SAVE_FAILED, error);
    }
  }

  /**
   * Find registration group by ID
   * @param id - Registration group ID
   * @returns Registration group with relations
   */
  async findRegistrationGroupById(id: number): Promise<RegistrationGroup> {
    this.logger.log('Getting registration group by ID', { id });
    
    try {
      const result = await this.repository.findRegistrationGroupById(id);
      
      if (!result) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_GROUPING_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      
      this.logger.log('Registration group retrieved successfully', { id });
      return result;
    } catch (error) {
      this.logger.error('Error finding registration group', error.stack, { id });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_GET_FAILED, error);
    }
  }

  /**
   * Delete a registration group (soft delete)
   * @param id - Registration group ID
   * @param deletedBy - ID of user deleting the group
   * @returns Deletion result
   */
  async deleteRegistrationGroup(id: number, deletedBy: number) {
    this.logger.log('Delete registration group request received', { id, deletedBy });
    
    try {
      const group = await this.repository.findRegistrationGroupById(id);
      
      if (!group) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_GROUPING_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      
      await this.repository.deleteRegistrationGroup(id, deletedBy);
      
      this.logger.log('Registration group deleted successfully', { id, deletedBy });
      return { message: 'Registration group deleted successfully', id };
    } catch (error) {
      this.logger.error('Error deleting registration group', error.stack, { id, deletedBy });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_DELETE_FAILED, error);
    }
  }

  /**
   * Add registrations to an existing group
   * @param groupId - Registration group ID
   * @param dto - Registrations to add
   * @returns Success message
   */
  async addRegistrationsToGroup(groupId: number, dto: AddRegistrationsToGroupDto) {
    this.logger.log('Add registrations to group request received', { groupId, dto });
    
    try {
      const group = await this.repository.findRegistrationGroupById(groupId);
      
      if (!group) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_GROUPING_NOTFOUND,
          null,
          null,
          groupId.toString()
        );
      }

      // Validate registrations
      await this.repository.validateRegistrationsForPairingOrGrouping(
        dto.registrationIds,
        group.programId,
        group.subProgramId,
        false,
      );

      await this.repository.addRegistrationsToGroup(groupId, dto);
      
      this.logger.log('Registrations added to group successfully', {
        groupId,
        addedCount: dto.registrationIds.length
      });

      return {
        message: 'Registrations added to group successfully',
        groupId,
        addedCount: dto.registrationIds.length,
      };
    } catch (error) {
      this.logger.error('Error adding registrations to group', error.stack, { groupId, dto });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_ADD_FAILED, error);
    }
  }

  /**
   * Remove registrations from a group
   * @param groupId - Registration group ID
   * @param dto - Registrations to remove
   * @returns Success message
   */
  async removeRegistrationsFromGroup(groupId: number, dto: RemoveRegistrationsDto) {
    this.logger.log('Remove registrations from group request received', { groupId, dto });
    
    try {
      const group = await this.repository.findRegistrationGroupById(groupId);
      
      if (!group) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_GROUPING_NOTFOUND,
          null,
          null,
          groupId.toString()
        );
      }

      await this.repository.removeRegistrationsFromGroup(groupId, dto);
      
      this.logger.log('Registrations removed from group successfully', {
        groupId,
        removedCount: dto.registrationIds?.length ?? 0
      });

      return {
        message: 'Registrations removed from group successfully',
        groupId,
        removedCount: dto.registrationIds?.length ?? 0,
      };
    } catch (error) {
      this.logger.error('Error removing registrations from group', error.stack, { groupId, dto });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_REMOVE_FAILED, error);
    }
  }

  // ========== PRIVATE VALIDATION METHODS ==========

  /**
   * Validate registrations for pairing
   * @param dto - Registration pair creation data
   */
  private async validateRegistrationsForPairing(dto: CreateRegistrationPairDto): Promise<void> {
    this.logger.debug('Validating registrations for pairing', { dto });
    
    try {
      await this.repository.validateRegistrationsForPairingOrGrouping(
        dto.registrationIds,
        dto.programId,
        dto.subProgramId,
        true,
      );
      
      this.logger.debug('Registration pairing validation completed successfully', { dto });
    } catch (validationError) {
      this.logger.error('Registration pairing validation failed', validationError.stack, {
        registrationIds: dto.registrationIds,
        programId: dto.programId
      });
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_PAIRING_OR_GROUPING_VALIDATION_FAILED,
        validationError,
        null,
        'Validation failed for registration pairing'
      );
    }
  }

  /**
   * Validate registrations for grouping
   * @param dto - Registration group creation data
   */
  private async validateRegistrationsForGrouping(dto: CreateRegistrationGroupDto): Promise<void> {
    this.logger.debug('Validating registrations for grouping', { dto });
    
    try {
      await this.repository.validateRegistrationsForPairingOrGrouping(
        dto.registrationIds,
        dto.programId,
        dto.subProgramId,
        false,
      );
      
      this.logger.debug('Registration grouping validation completed successfully', { dto });
    } catch (validationError) {
      this.logger.error('Registration grouping validation failed', validationError.stack, {
        registrationIds: dto.registrationIds,
        programId: dto.programId
      });
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_PAIRING_OR_GROUPING_VALIDATION_FAILED,
        validationError,
        null,
        'Validation failed for registration grouping'
      );
    }
  }
}
