import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, DataSource, EntityManager, IsNull, In, FindOptionsWhere } from 'typeorm';
import {
  RegistrationPairMap,
  RegistrationGroupMap,
  ProgramRegistration,
  Program,
} from 'src/common/entities';
import { RegistrationGroup } from 'src/common/entities/registration-group.entity';
import { RegistrationPair } from 'src/common/entities/registration-pair.entity';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import {
  CreateRegistrationPairDto,
  CreateRegistrationGroupDto,
  AddRegistrationsToGroupDto,
  AddRegistrationsToPairDto,
} from './dto/create-registration-grouping.dto';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import {
  RemoveRegistrationsDto,
  UpdateRegistrationGroupDto,
  UpdateRegistrationPairDto,
} from './dto/update-registration-grouping.dto';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';

/**
 * Repository class for managing registration grouping and pairing data operations
 * Handles database interactions with proper error handling and validation
 */
@Injectable()
export class RegistrationGroupingRepository {
  constructor(
    @InjectRepository(RegistrationPair)
    private readonly pairRepo: Repository<RegistrationPair>,
    @InjectRepository(RegistrationPairMap)
    private readonly pairMapRepo: Repository<RegistrationPairMap>,
    @InjectRepository(RegistrationGroup)
    private readonly groupRepo: Repository<RegistrationGroup>,
    @InjectRepository(RegistrationGroupMap)
    private readonly groupMapRepo: Repository<RegistrationGroupMap>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepo: Repository<ProgramRegistration>,
    @InjectRepository(Program)
    private readonly programRepo: Repository<Program>,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
  ) {}

  // ========== REGISTRATION PAIR METHODS ==========

  /**
   * Create a new registration pair with mappings
   * @param dto - Registration pair creation data
   * @param seqNumber - Sequence number for the pair
   * @returns Created registration pair with relations
   */
  async createRegistrationPair(
    dto: CreateRegistrationPairDto,
    seqNumber: number,
  ): Promise<RegistrationPair> {
    this.logger.log('Creating registration pair', { dto, seqNumber });
    
    try {
      return await this.dataSource.transaction(async (manager) => {
        // Create registration pair
        const pair = manager.create(RegistrationPair, {
          pairCode: dto.pairCode,
          seqNumber: seqNumber,
          programId: dto.programId,
          subProgramId: dto.subProgramId,
          remarks: dto.remarks,
          createdById: dto.createdBy,
          createdAt: new Date(),
        });

        const savedPair = await manager.save(pair);
        this.logger.log('Created registration pair', { pairId: savedPair.id });

        // Create pair mappings
        await this.createPairMappings(manager, savedPair.id, dto.registrationIds, dto.createdBy);

        // Fetch the created pair with relations
        const result = await manager.findOne(RegistrationPair, {
          where: { id: savedPair.id },
          relations: ['program', 'subProgram', 'createdBy', 'registrationPairMaps'],
        });

        if (!result) {
          throw new InifniNotFoundException(
            ERROR_CODES.REGISTRATION_PAIRING_NOTFOUND,
            null,
            null,
            savedPair.id.toString()
          );
        }

        return result;
      });
    } catch (error) {
      this.logger.error('Error creating registration pair', error.stack, { dto, seqNumber });
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAIRING_SAVE_FAILED, error);
    }
  }

  /**
   * Find registration pair by ID with all relations
   * @param id - Registration pair ID
   * @returns Registration pair with relations or null if not found
   */
  async findRegistrationPairById(id: number): Promise<RegistrationPair | null> {
    try {
      return await this.pairRepo.findOne({
        where: { id, deletedAt: IsNull() },
        relations: ['program', 'subProgram', 'createdBy', 'updatedBy', 'registrationPairMaps'],
      });
    } catch (error) {
      this.logger.error('Error finding registration pair by ID', error.stack, { id });
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAIRING_GET_FAILED, error);
    }
  }

  /**
   * Soft delete a registration pair and its mappings
   * @param id - Registration pair ID
   * @param deletedBy - ID of user deleting the pair
   */
  async deleteRegistrationPair(id: number, deletedBy: number): Promise<void> {
    this.logger.log('Deleting registration pair', { id, deletedBy });
    
    try {
      await this.dataSource.transaction(async (manager) => {
        // Check if pair exists
        const existingPair = await manager.findOne(RegistrationPair, {
          where: { id, deletedAt: IsNull() }
        });

        if (!existingPair) {
          throw new InifniNotFoundException(
            ERROR_CODES.REGISTRATION_PAIRING_NOTFOUND,
            null,
            null,
            id.toString()
          );
        }

        // Soft delete pair mappings
        await manager.update(
          RegistrationPairMap,
          { registrationPairId: id, deletedAt: IsNull() },
          { deletedAt: new Date(), updatedById: deletedBy },
        );

        // Soft delete pair
        await manager.update(RegistrationPair, id, {
          deletedAt: new Date(),
          updatedById: deletedBy,
        });

        this.logger.log('Deleted registration pair successfully', { id, deletedBy });
      });
    } catch (error) {
      this.logger.error('Error deleting registration pair', error.stack, { id, deletedBy });
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAIRING_DELETE_FAILED, error);
    }
  }

  // ========== REGISTRATION GROUP METHODS ==========

  /**
   * Create a new registration group with mappings
   * @param dto - Registration group creation data
   * @returns Created registration group with relations
   */
  async createRegistrationGroup(dto: CreateRegistrationGroupDto): Promise<RegistrationGroup> {
    this.logger.log('Creating registration group', { dto });
    
    try {
      return await this.dataSource.transaction(async (manager) => {
        // Create registration group
        const group = manager.create(RegistrationGroup, {
          groupName: dto.groupName,
          programId: dto.programId,
          subProgramId: dto.subProgramId,
          remarks: dto.remarks,
          createdById: dto.createdBy,
          createdAt: new Date(),
        });

        const savedGroup = await manager.save(group);
        this.logger.log('Created registration group', { groupId: savedGroup.id });

        // Create group mappings
        await this.createGroupMappings(manager, savedGroup.id, dto.registrationIds, dto.createdBy);

        // Fetch the created group with relations
        const result = await manager.findOne(RegistrationGroup, {
          where: { id: savedGroup.id },
          relations: ['program', 'subProgram', 'createdBy', 'registrationGroupMaps'],
        });

        if (!result) {
          throw new InifniNotFoundException(
            ERROR_CODES.REGISTRATION_GROUPING_NOTFOUND,
            null,
            null,
            savedGroup.id.toString()
          );
        }

        return result;
      });
    } catch (error) {
      this.logger.error('Error creating registration group', error.stack, { dto });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_SAVE_FAILED, error);
    }
  }

  /**
   * Find registration group by ID with all relations
   * @param id - Registration group ID
   * @returns Registration group with relations or null if not found
   */
  async findRegistrationGroupById(id: number): Promise<RegistrationGroup | null> {
    try {
      return await this.groupRepo.findOne({
        where: { id, deletedAt: IsNull() },
        relations: ['program', 'subProgram', 'createdBy', 'updatedBy', 'registrationGroupMaps'],
      });
    } catch (error) {
      this.logger.error('Error finding registration group by ID', error.stack, { id });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_GET_FAILED, error);
    }
  }

  /**
   * Soft delete a registration group and its mappings
   * @param id - Registration group ID
   * @param deletedBy - ID of user deleting the group
   */
  async deleteRegistrationGroup(id: number, deletedBy: number): Promise<void> {
    this.logger.log('Deleting registration group', { id, deletedBy });
    
    try {
      await this.dataSource.transaction(async (manager) => {
        // Check if group exists
        const group = await manager.findOne(RegistrationGroup, {
          where: { id, deletedAt: IsNull() }
        });

        if (!group) {
          throw new InifniNotFoundException(
            ERROR_CODES.REGISTRATION_GROUPING_NOTFOUND,
            null,
            null,
            id.toString()
          );
        }

        // Soft delete group mappings
        await manager.update(
          RegistrationGroupMap,
          { registrationGroupId: id, deletedAt: IsNull() },
          { deletedAt: new Date(), updatedById: deletedBy },
        );

        // Soft delete group
        await manager.update(RegistrationGroup, id, {
          deletedAt: new Date(),
          updatedById: deletedBy,
        });

        this.logger.log('Deleted registration group successfully', { id, deletedBy });
      });
    } catch (error) {
      this.logger.error('Error deleting registration group', error.stack, { id, deletedBy });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_DELETE_FAILED, error);
    }
  }

  /**
   * Add registrations to an existing group
   * @param groupId - Registration group ID
   * @param dto - Registrations to add
   */
  async addRegistrationsToGroup(groupId: number, dto: AddRegistrationsToGroupDto): Promise<void> {
    this.logger.log('Adding registrations to group', { groupId, dto });
    
    try {
      await this.dataSource.transaction(async (manager) => {
        // Verify group exists
        const group = await manager.findOne(RegistrationGroup, {
          where: { id: groupId, deletedAt: IsNull() }
        });

        if (!group) {
          throw new InifniNotFoundException(
            ERROR_CODES.REGISTRATION_GROUPING_NOTFOUND,
            null,
            null,
            groupId.toString()
          );
        }

        // Validate registrations
        await this.validateRegistrationsForPairingOrGrouping(
          dto.registrationIds,
          group.programId,
          group.subProgramId,
          false,
        );

        // Create group mappings
        await this.createGroupMappings(manager, groupId, dto.registrationIds, dto.updatedBy);

        this.logger.log('Added registrations to group successfully', {
          groupId,
          count: dto.registrationIds.length
        });
      });
    } catch (error) {
      this.logger.error('Error adding registrations to group', error.stack, { groupId, dto });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_ADD_FAILED, error);
    }
  }

  /**
   * Remove registrations from a group
   * @param groupId - Registration group ID
   * @param dto - Registrations to remove
   */
  async removeRegistrationsFromGroup(groupId: number, dto: RemoveRegistrationsDto): Promise<void> {
    this.logger.log('Removing registrations from group', { groupId, dto });
    
    try {
      await this.dataSource.transaction(async (manager) => {
        // Verify group exists
        const group = await manager.findOne(RegistrationGroup, {
          where: { id: groupId, deletedAt: IsNull() }
        });

        if (!group) {
          throw new InifniNotFoundException(
            ERROR_CODES.REGISTRATION_GROUPING_NOTFOUND,
            null,
            null,
            groupId.toString()
          );
        }

        // Soft delete group mappings
        await manager.update(
          RegistrationGroupMap,
          {
            registrationGroupId: groupId,
            registrationId: In(dto.registrationIds ?? []),
            deletedAt: IsNull(),
          },
          {
            deletedAt: new Date(),
            updatedById: dto.updatedBy,
          },
        );

        this.logger.log('Removed registrations from group successfully', {
          groupId,
          count: dto.registrationIds?.length ?? 0
        });
      });
    } catch (error) {
      this.logger.error('Error removing registrations from group', error.stack, { groupId, dto });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_REMOVE_FAILED, error);
    }
  }

  // ========== HELPER METHODS ==========

  /**
   * Validate registrations for pairing or grouping
   * Checks if registrations exist, belong to the program, and are not already paired/grouped
   * @param registrationIds - Array of registration IDs to validate
   * @param programId - Program ID
   * @param subProgramId - Optional sub-program ID
   * @param isPairing - True for pairing validation, false for grouping validation
   */
  async validateRegistrationsForPairingOrGrouping(
    registrationIds: number[],
    programId: number,
    subProgramId?: number,
    isPairing: boolean = false,
  ): Promise<void> {
    this.logger.debug('Validating registrations for pairing/grouping', {
      registrationIds,
      programId,
      subProgramId,
      isPairing
    });

    try {
      // Check if registrations exist and belong to the program
      const whereClause: FindOptionsWhere<ProgramRegistration> = {
        id: In(registrationIds),
        program: { id: programId },
        deletedAt: IsNull(),
      };

      if (subProgramId) {
        whereClause.allocatedProgramId = subProgramId;
      }

      const registrations = await this.registrationRepo.find({
        where: whereClause,
      });

      if (registrations.length !== registrationIds.length) {
        const foundIds = registrations.map(r => r.id);
        const missingIds = registrationIds.filter(id => !foundIds.includes(id));
        
        this.logger.error('Registration validation failed', JSON.stringify({
          requestedIds: registrationIds,
          foundIds,
          missingIds
        }));
        
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_NOT_FOUND,
          null,
          null,
          `Some registrations not found or do not belong to the specified program: ${missingIds.join(', ')}`
        );
      }

      if (isPairing) {
        // Check if any registration is already paired
        const existingPairMaps = await this.pairMapRepo.find({
          where: {
            registrationId: In(registrationIds),
            deletedAt: IsNull(),
          },
        });

        if (existingPairMaps.length > 0) {
          const pairedIds = existingPairMaps.map(pm => pm.registrationId);
          
          this.logger.error('Some registrations are already paired: ' + JSON.stringify({ pairedIds }));
          
          throw new InifniBadRequestException(
            ERROR_CODES.PAIRING_DUPLICATE,
            null,
            null,
            `Registrations already paired: ${pairedIds.join(', ')}`
          );
        }
      } else {
        // Check if any registration is already grouped
        const existingGroupMaps = await this.groupMapRepo.find({
          where: {
            registrationId: In(registrationIds),
            deletedAt: IsNull(),
          },
        });

        if (existingGroupMaps.length > 0) {
          const groupedIds = existingGroupMaps.map(gm => gm.registrationId);
          
          this.logger.error('Some registrations are already grouped: ' + JSON.stringify({ groupedIds }));
          
          throw new InifniBadRequestException(
            ERROR_CODES.GROUPING_DUPLICATE,
            null,
            null,
            `Registrations already grouped: ${groupedIds.join(', ')}`
          );
        }
      }

      this.logger.debug('Registration validation completed successfully', {
        registrationIds,
        isPairing
      });
    } catch (error) {
      this.logger.error('Error validating registrations for pairing/grouping', error.stack, {
        registrationIds,
        programId,
        subProgramId,
        isPairing
      });
      handleKnownErrors(ERROR_CODES.REGISTRATION_GROUPING_ADD_FAILED, error);
    }
  }

  /**
   * Create pair mappings for a registration pair
   * @param manager - Entity manager for transaction
   * @param pairId - Registration pair ID
   * @param registrationIds - Array of registration IDs
   * @param userId - User creating the mappings
   */
  private async createPairMappings(
    manager: EntityManager,
    pairId: number,
    registrationIds: number[],
    userId: number,
  ): Promise<void> {
    try {
      const mappings = registrationIds.map((registrationId) =>
        manager.create(RegistrationPairMap, {
          registrationPairId: pairId,
          registrationId,
          createdById: userId,
          createdAt: new Date(),
        }),
      );

      await manager.save(mappings);
      
      this.logger.log('Created pair mappings', {
        pairId,
        count: mappings.length
      });
    } catch (error) {
      this.logger.error('Error creating pair mappings', error.stack, {
        pairId,
        registrationIds,
        userId
      });
      throw error;
    }
  }

  /**
   * Create group mappings for a registration group
   * @param manager - Entity manager for transaction
   * @param groupId - Registration group ID
   * @param registrationIds - Array of registration IDs
   * @param userId - User creating the mappings
   */
  private async createGroupMappings(
    manager: EntityManager,
    groupId: number,
    registrationIds: number[],
    userId: number,
  ): Promise<void> {
    try {
      const mappings = registrationIds.map((registrationId) =>
        manager.create(RegistrationGroupMap, {
          registrationGroupId: groupId,
          registrationId,
          createdById: userId,
          createdAt: new Date(),
        }),
      );

      await manager.save(mappings);
      
      this.logger.log('Created group mappings', {
        groupId,
        count: mappings.length
      });
    } catch (error) {
      this.logger.error('Error creating group mappings', error.stack, {
        groupId,
        registrationIds,
        userId
      });
      throw error;
    }
  }

  /**
   * Get next sequence number for pairing
   * @param programId - Program ID
   * @param pairCode - Pair code
   * @returns Next sequence number
   */
  async getNextPairingSequenceNumber(programId: number, pairCode: string): Promise<number> {
    try {
      const result = await this.pairRepo
        .createQueryBuilder('pair')
        .select('MAX(pair.seqNumber)', 'max')
        .where('pair.programId = :programId', { programId })
        .andWhere('pair.pairCode = :pairCode', { pairCode })
        .andWhere('pair.deletedAt IS NULL')
        .getRawOne();

      const maxSeq = result?.max ? parseInt(result.max, 10) : 0;
      return maxSeq + 1;
    } catch (error) {
      this.logger.error('Error getting next pairing sequence number', error.stack, {
        programId,
        pairCode
      });
      throw error;
    }
  }

  /**
   * Get maximum room occupancy with count for a program
   * Retrieves the count of rooms with the specified occupancy for the program
   * @param occupancy - Room occupancy value to match
   * @param programId - Program ID
   * @param subProgramId - Optional sub-program ID
   * @returns Object containing occupancy and total count of rooms with that occupancy
   */
  async getMaxRoomOccupancyWithCount(
    occupancy: number,
    programId: number,
    subProgramId?: number,
  ): Promise<{ occupancy: number; totalCountWithOccupancy: number }> {
    try {
      let qb = this.dataSource
        .createQueryBuilder()
        .select('COUNT(*)', 'total_count')
        .from('program_room_inventory_map', 'prm')
        .innerJoin('room', 'room', 'room.id = prm.room_id')
        .where('prm.program_id = :programId', { programId })
        .andWhere('prm.deleted_at IS NULL')
        .andWhere('room.occupancy = :occupancy', { occupancy });

      if (subProgramId !== undefined) {
        qb = qb.andWhere('prm.sub_program_id = :subProgramId', { subProgramId });
      }

      const result = await qb.getRawOne();
      const totalWithOccupancy = parseInt(result?.total_count || '0', 10);

      return {
        occupancy: occupancy,
        totalCountWithOccupancy: totalWithOccupancy,
      };
    } catch (error) {
      this.logger.error('Error getting max room occupancy with count', error.stack, {
        occupancy,
        programId,
        subProgramId
      });
      throw error;
    }
  }

  /**
   * Get count of pairs with maximum count for a program
   * Retrieves the total number of pairs that have the specified registration count
   * @param pairCount - Registration count per pair to match
   * @param programId - Program ID
   * @param subProgramId - Optional sub-program ID
   * @returns Object containing pair count and total number of pairs with that count
   */
  async getPairCountsWithMaxCount(
    pairCount: number,
    programId: number,
    subProgramId?: number,
  ): Promise<{ pairCount: number; totalWithMaxCount: number }> {
    try {
      // Build subquery to get pairs with the specific count
      const subQuery = this.pairMapRepo
        .createQueryBuilder('rpm')
        .innerJoin('rpm.registrationPair', 'rp')
        .select('rpm.registrationPairId')
        .where('rp.programId = :programId', { programId })
        .andWhere('rp.deletedAt IS NULL')
        .andWhere('rpm.deletedAt IS NULL')
        .groupBy('rpm.registrationPairId')
        .having('COUNT(*) = :pairCount', { pairCount });

      if (subProgramId !== undefined) {
        subQuery.andWhere('rp.subProgramId = :subProgramId', { subProgramId });
      }

      // Count the results from the subquery
      const result = await this.dataSource
        .createQueryBuilder()
        .select('COUNT(*)', 'total')
        .from(`(${subQuery.getQuery()})`, 'pair_counts')
        .setParameters(subQuery.getParameters())
        .getRawOne();

      const totalWithGivenPairCount = parseInt(result?.total || '0', 10);

      return {
        pairCount: pairCount,
        totalWithMaxCount: totalWithGivenPairCount,
      };
    } catch (error) {
      this.logger.error('Error getting pair counts with max count', error.stack, {
        pairCount,
        programId,
        subProgramId
      });
      throw error;
    }
  }
}
