import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  HttpCode,
  HttpStatus,
  UseGuards,
  Req,
  ValidationPipe,
  Res,
} from '@nestjs/common';
import {
  ApiTags,
  ApiOperation,
  ApiResponse,
  ApiParam,
  ApiQuery,
  ApiBody,
  ApiBearerAuth,
  ApiSecurity,
} from '@nestjs/swagger';
import { RegistrationGroupingService } from './registration-grouping.service';
import {
  CreateRegistrationGroupDto,
  CreateRegistrationPairDto,
  AddRegistrationsToGroupDto,
  AddRegistrationsToPairDto,
} from './dto/create-registration-grouping.dto';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { AppLoggerService } from 'src/common/services/logger.service';
import {
  RemoveRegistrationsDto,
  UpdateRegistrationGroupDto,
  UpdateRegistrationPairDto,
} from './dto/update-registration-grouping.dto';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { ResponseService } from 'src/common/response-handling/response-handler';

@ApiTags('registration-grouping')
@Controller('registration-grouping')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'mahatria', 'shoba', 'operational_manger', 'rm_support')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class RegistrationGroupingController {
  constructor(
    private readonly service: RegistrationGroupingService,
    private readonly logger: AppLoggerService,
    private readonly responseService: ResponseService,
  ) {}

  // ========== REGISTRATION PAIR ENDPOINTS ==========

  @Post('pair')
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: 'Create a new registration pair' })
  @ApiBody({ type: CreateRegistrationPairDto })
  @ApiResponse({
    status: HttpStatus.CREATED,
    description: 'Registration pair created successfully',
  })
  @ApiResponse({
    status: HttpStatus.BAD_REQUEST,
    description: SWAGGER_API_RESPONSE.BAD_REQUEST,
  })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  async createRegistrationPair(
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: CreateRegistrationPairDto,
    @Req() req: any,
    @Res() res: any,
  ) {
    try {
      const user = req.user;
      if (!user || !user.id) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          '',
          'User not found in request',
        );
      }
      dto.createdBy = user.id;

      const result = await this.service.createRegistrationPair(dto);
      this.responseService.success(res, 'Registration pair created successfully', result, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('pair/:id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get registration pair by ID' })
  @ApiParam({ name: 'id', description: 'Registration pair ID', type: Number })
  @ApiResponse({
    status: HttpStatus.OK,
    description: SWAGGER_API_RESPONSE.OK,
  })
  @ApiResponse({
    status: HttpStatus.NOT_FOUND,
    description: SWAGGER_API_RESPONSE.NOT_FOUND,
  })
  async getRegistrationPairById(
    @Param('id') id: number,
    @Res() res: any,
  ) {
    try {
      this.logger.log(`Get registration pair by ID: ${id}`);
      const result = await this.service.findRegistrationPairById(id);
      if (!result) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_PAIRING_NOTFOUND,
          null,
          id.toString(),
          'Registration pair not found',
        );
      }
      this.responseService.success(res, 'Registration pair retrieved successfully', result);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Delete('pair/:id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Delete registration pair' })
  @ApiParam({ name: 'id', description: 'Registration pair ID', type: Number })
  @ApiResponse({
    status: HttpStatus.OK,
    description: SWAGGER_API_RESPONSE.OK,
  })
  @ApiResponse({
    status: HttpStatus.NOT_FOUND,
    description: SWAGGER_API_RESPONSE.NOT_FOUND,
  })
  async deleteRegistrationPair(
    @Param('id') id: number,
    @Req() req: any,
    @Res() res: any,
  ) {
    try {
      this.logger.log(`Delete registration pair ${id} request received`);
      const user = req.user;
      if (!user || !user.id) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          '',
          'User not found in request',
        );
      }

      await this.service.deleteRegistrationPair(id, user.id);
      this.responseService.success(res, 'Registration pair deleted successfully');
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  // ========== REGISTRATION GROUP ENDPOINTS ==========
  @Post('group')
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: 'Create a new registration group' })
  @ApiBody({ type: CreateRegistrationGroupDto })
  @ApiResponse({
    status: HttpStatus.CREATED,
    description: 'Registration group created successfully',
  })
  @ApiResponse({
    status: HttpStatus.BAD_REQUEST,
    description: SWAGGER_API_RESPONSE.BAD_REQUEST,
  })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  async createRegistrationGroup(
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: CreateRegistrationGroupDto,
    @Req() req: any,
    @Res() res: any,
  ) {
    try {
      const user = req.user;
      if (!user || !user.id) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          '',
          'User not found in request',
        );
      }
      dto.createdBy = user.id;

      const result = await this.service.createRegistrationGroup(dto);
      this.responseService.success(res, 'Registration group created successfully', result, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('group/:id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get registration group by ID' })
  @ApiParam({ name: 'id', description: 'Registration group ID', type: Number })
  @ApiResponse({
    status: HttpStatus.OK,
    description: SWAGGER_API_RESPONSE.OK,
  })
  @ApiResponse({
    status: HttpStatus.NOT_FOUND,
    description: SWAGGER_API_RESPONSE.NOT_FOUND,
  })
  async getRegistrationGroupById(@Param('id') id: number, @Res() res: any) {
    try {
      this.logger.log(`Get registration group by ID: ${id}`);
      const result = await this.service.findRegistrationGroupById(id);
      this.responseService.success(res, 'Registration group retrieved successfully', result);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Delete('group/:id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Delete registration group' })
  @ApiParam({ name: 'id', description: 'Registration group ID', type: Number })
  @ApiResponse({
    status: HttpStatus.OK,
    description: SWAGGER_API_RESPONSE.OK,
  })
  @ApiResponse({
    status: HttpStatus.NOT_FOUND,
    description: SWAGGER_API_RESPONSE.NOT_FOUND,
  })
  async deleteRegistrationGroup(@Param('id') id: number, @Req() req: any, @Res() res: any) {
    try {
      const user = req.user;
      if (!user || !user.id) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          '',
          'User not found in request',
        );
      }

      this.logger.log(`Delete registration group ${id} request received`);

      await this.service.deleteRegistrationGroup(id, user.id);
      this.responseService.success(res, 'Registration group deleted successfully');
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put('group/:id/add-registrations')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Add registrations to group' })
  @ApiParam({ name: 'id', description: 'Registration group ID', type: Number })
  @ApiBody({ type: AddRegistrationsToGroupDto })
  @ApiResponse({
    status: HttpStatus.OK,
    description: 'Registrations added to group successfully',
  })
  async addRegistrationsToGroup(
    @Param('id') id: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: AddRegistrationsToGroupDto,
    @Req() req: any,
    @Res() res: any,
  ) {
    try {
      const user = req.user;
      if (!user || !user.id) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          '',
          'User not found in request',
        );
      }
      dto.updatedBy = user.id;

      await this.service.addRegistrationsToGroup(id, dto);
      this.responseService.success(res, 'Registrations added to group successfully');
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put('group/:id/remove-registrations')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Remove registrations from group' })
  @ApiParam({ name: 'id', description: 'Registration group ID', type: Number })
  @ApiBody({ type: RemoveRegistrationsDto })
  @ApiResponse({
    status: HttpStatus.OK,
    description: 'Registrations removed from group successfully',
  })
  async removeRegistrationsFromGroup(
    @Param('id') id: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: RemoveRegistrationsDto,
    @Req() req: any,
    @Res() res: any,
  ) {
    try {
      const user = req.user;
      if (!user || !user.id) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          '',
          'User not found in request',
        );
      }
      dto.updatedBy = user.id;

      await this.service.removeRegistrationsFromGroup(id, dto);
      this.responseService.success(res, 'Registrations removed from group successfully');
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
