import { Injectable } from '@nestjs/common';
import { DataSource, Repository } from 'typeorm';
import { InjectDataSource, InjectRepository } from '@nestjs/typeorm';
import { CreateRegistrationCustomResponseDto } from './dto/create-registration-custom-response.dto';
import { UpdateRegistrationCustomResponseDto } from './dto/update-registration-custom-response.dto';
import { RegistrationCustomResponseRepository } from './registration-custom-response.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CommonDataService } from 'src/common/services/commonData.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { registrationCustomResponseMessages } from 'src/common/constants/strings-constants';
import { RegistrationCustomResponse } from 'src/common/entities';

@Injectable()
export class RegistrationCustomResponseService {
  private repo: RegistrationCustomResponseRepository;

  constructor(
    @InjectDataSource()
    private readonly dataSource: DataSource,
    @InjectRepository(RegistrationCustomResponse)
    private readonly registrationCustomResponseRepo: Repository<RegistrationCustomResponse>,
    private readonly logger: AppLoggerService,
    private readonly commonDataService: CommonDataService,
  ) {
    this.repo = new RegistrationCustomResponseRepository(
      this.registrationCustomResponseRepo,
      this.commonDataService,
      this.logger,
    );
  }

  async create(dto: CreateRegistrationCustomResponseDto) {
    this.logger.log(registrationCustomResponseMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationCustomResponseRepository(
          manager.getRepository(RegistrationCustomResponse),
          this.commonDataService,
          this.logger,
        );
        return await repository.createEntity(dto);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    this.logger.log(registrationCustomResponseMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationCustomResponseMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      return await this.repo.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateRegistrationCustomResponseDto) {
    this.logger.log(registrationCustomResponseMessages.UPDATE_REQUEST_RECEIVED);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationCustomResponseRepository(
          manager.getRepository(RegistrationCustomResponse),
          this.commonDataService,
          this.logger,
        );
        return await repository.updateEntity(id, dto);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    this.logger.log(registrationCustomResponseMessages.DELETE_REQUEST_RECEIVED);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationCustomResponseRepository(
          manager.getRepository(RegistrationCustomResponse),
          this.commonDataService,
          this.logger,
        );
        return await repository.remove(id);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_DELETE_FAILED, error);
    }
  }
}
