import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { RegistrationCustomResponse } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { CreateRegistrationCustomResponseDto } from './dto/create-registration-custom-response.dto';
import { UpdateRegistrationCustomResponseDto } from './dto/update-registration-custom-response.dto';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';

@Injectable()
export class RegistrationCustomResponseRepository {
  constructor(
    @InjectRepository(RegistrationCustomResponse)
    private readonly repo: Repository<RegistrationCustomResponse>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  async createEntity(dto: CreateRegistrationCustomResponseDto) {
    try {
      const entity = this.repo.create({
        ...dto,
        registration: { id: dto.registrationId } as any,
        question: { id: dto.questionId } as any,
        createdBy: { id: dto.createdBy } as any,
        updatedBy: { id: dto.updatedBy } as any,
      });
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    try {
      const data = await this.commonDataService.get(
        this.repo,
        undefined,
        {},
        limit,
        offset,
        { id: 'ASC' },
        undefined,
        ['registration', 'question'],
      );
      const total = await this.repo.count();
      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    try {
      const data = await this.commonDataService.findOneById(this.repo, id, false, ['registration', 'question']);
      if (!data) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_NOTFOUND, null, null, id.toString());
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_FIND_BY_ID_FAILED, error);
    }
  }

  async updateEntity(id: number, dto: UpdateRegistrationCustomResponseDto) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_NOTFOUND, null, null, id.toString());
      }
      Object.assign(entity, dto);
      if (dto.registrationId !== undefined) entity.registration = { id: dto.registrationId } as any;
      if (dto.questionId !== undefined) entity.question = { id: dto.questionId } as any;
      entity.updatedBy = { id: dto.updatedBy } as any;
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_NOTFOUND, null, null, id.toString());
      }
      await this.repo.remove(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CUSTOM_RESPONSE_DELETE_FAILED, error);
    }
  }
}
