import {
  Injectable,
} from "@nestjs/common";
import { FormSection, Question, User } from "src/common/entities";
import { CreateQuestionDto } from "./dto/create-question.dto";
import { UpdateQuestionDto } from "./dto/update-question.dto";
import { CommonDataService } from "src/common/services/commonData.service";
import { QuestionRepository } from "./question.repository";
import { ILike, IsNull, Repository, DataSource } from "typeorm";
import {
  userConstMessages,
  questionConstMessages,
  FORM_SECTION_MESSAGES,
} from "src/common/constants/strings-constants";
import { InjectRepository, InjectDataSource } from "@nestjs/typeorm";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import InifniBadRequestException from "src/common/exceptions/infini-badrequest-exception";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { AppLoggerService } from "src/common/services/logger.service";

@Injectable()
export class QuestionService {
  constructor(
    private readonly questionRepository: QuestionRepository,
    @InjectRepository(User) private readonly userRepository: Repository<User>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
    @InjectRepository(FormSection)
    private readonly formSectionRepository: Repository<FormSection>,
    @InjectDataSource()
    private readonly dataSource: DataSource,
  ) {}

  /**
   * Creates a new question.
   * Validates the existence of the creator and updater users.
   * Ensures the question is unique by its label.
   * @param createDto - Data transfer object containing question details.
   * @returns The created question.
   * @throws NotFoundException if the creator or updater user does not exist.
   * @throws BadRequestException if the question already exists.
   */
  async create(createDto: CreateQuestionDto) {
    return await this.dataSource.transaction(async (manager) => {
      const questionRepo = manager.getRepository(Question);
      const userRepo = manager.getRepository(User);
      const formSectionRepo = manager.getRepository(FormSection);

      this.logger.log(questionConstMessages.CREATING_QUESTION(createDto));
      const { label, type, config, status, createdBy, updatedBy, formSectionId } = createDto;

      // Validate creator and updater users
      const [creator, updater] = await Promise.all([
        userRepo.findOne({ where: { id: createdBy } }),
        userRepo.findOne({ where: { id: updatedBy } }),
      ]);

      if (!creator) {
        // throw new NotFoundException(userConstMessages.USER_NOT_FOUND_ID(createdBy));
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(createdBy));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_CREATOR_NOTFOUND,
          null,
          null,
          createdBy.toString(),
        );
      }
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        // throw new NotFoundException(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString(),
        );
      }

      // Validate form section if provided
      let formSection: FormSection | undefined = undefined;
      if (formSectionId !== undefined && formSectionId !== null) {
        const foundFormSection = await formSectionRepo.findOne({ 
          where: { id: formSectionId },
        });
        if (!foundFormSection) {
          this.logger.error(FORM_SECTION_MESSAGES.FORM_SEC_NOT_FOUND(formSectionId));
          throw new InifniNotFoundException(
            ERROR_CODES.FORM_SECTION_NOTFOUND,
            null,
            null,
            formSectionId.toString(),
          );
        }
        formSection = foundFormSection;
      }

      // Check if a question with the same label and type already exists
      const existingQuestion = await this.questionRepository.findOneByLabelAndType(label, type);
      if (existingQuestion) {
        this.logger.error(questionConstMessages.DUPLICATE_QUESTION_FOUND(label));
        // throw new BadRequestException(questionConstMessages.DUPLICATE_QUESTION);
        throw new InifniBadRequestException(
          ERROR_CODES.QUESTION_DUPLICATE_BADREQUEST,
          null,
          null,
          label,
        );
      }

      const question = new Question({
        label,
        type,
        config,
        status,
        createdBy: creator,
        updatedBy: updater,
        formSection,

      });
      this.logger.log(questionConstMessages.CREATING_QUESTION(question));
      try {
        return await questionRepo.save(question);
      } catch (error) {
        this.logger.error('Error saving question in create', error?.stack, { error, createDto });
        handleKnownErrors(ERROR_CODES.QUESTION_SAVE_FAILED, error);
      }
    });
  }

  /**
   * Retrieves all questions with optional pagination and search functionality.
   * @param limit - Number of records per page.
   * @param offset - Offset for pagination.
   * @param searchText - Optional search text to filter questions by label.
   * @param parsedFilters - Additional filters for the query.
   * @returns A paginated list of questions and metadata.
   */
  async findAll(
    limit: number,
    offset: number,
    searchText: string,
    parsedFilters: Record<string, any>,
    formSectionId: number | null
  ) {
    this.logger.log(
      questionConstMessages.FINDING_ALL_QUESTIONS(limit, offset, searchText)
    );
    try {
      const whereClause: any = {};
      whereClause.deletedAt = IsNull();
      if (parsedFilters?.status) {
        whereClause.status = parsedFilters.status;
      }
      if (parsedFilters?.createdBy) {
        whereClause.createdBy = { id: parsedFilters.createdBy };
      }
      if (searchText) {
        whereClause.label = ILike(`%${searchText}%`);
      }

      if (formSectionId !== undefined && formSectionId !== null) {
        whereClause.formSection = { id: formSectionId };
      }

      this.logger.log(
        questionConstMessages.RETRIEVING_QUESTIONS_WHERE_CLAUSE(whereClause)
      );
      const data = await this.commonDataService.get(
        this.questionRepository["questionRepo"],
        ["id", "label", "type", "status","config"],
        whereClause,
        limit,
        offset,
        { id: "ASC" },
        searchText ? { label: searchText } : undefined,
        [
          "formSection",
          "questionOptionMaps",
          "questionOptionMaps.option",
        ]
      );

      const total = await this.questionRepository.count(whereClause);
      const totalPages = Math.ceil(total / limit);

      return {
        data,
        pagination: {
          totalPages,
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      this.logger.error('Error in findAll', error?.stack, { error, limit, offset, searchText });
      handleKnownErrors(ERROR_CODES.COMMON_DATA_SERVICE_GET_FAILED, error);
    }
  }

  /**
   * Retrieves a single question by its ID.
   * @param id - ID of the question to retrieve.
   * @returns The question data if found.
   * @throws NotFoundException if the question does not exist.
   */
  async findOne(id: number) {
    this.logger.log(questionConstMessages.FINDING_QUESTION_BY_ID(id));
    try {
      const question = await this.questionRepository.findOneById(id);
      if (!question) {
        // throw new NotFoundException(questionConstMessages.QUESTION_NOT_FOUND_ID(id));
        this.logger.error(questionConstMessages.QUESTION_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }
      return question;
    } catch (error) {
      this.logger.error('Error in findOne', error?.stack, { error, id });
      handleKnownErrors(ERROR_CODES.QUESTION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Updates an existing question by its ID.
   * Validates the existence of the updater user.
   * Ensures the updated question is unique by its label.
   * @param id - ID of the question to update.
   * @param updateDto - Data transfer object containing updated question details.
   * @returns The updated question.
   * @throws NotFoundException if the question or updater user does not exist.
   * @throws BadRequestException if a duplicate question exists.
   */
  async update(id: number, updateDto: UpdateQuestionDto) {
    return await this.dataSource.transaction(async (manager) => {
      const questionRepo = manager.getRepository(Question);
      const userRepo = manager.getRepository(User);
      const formSectionRepo = manager.getRepository(FormSection);

      this.logger.log(questionConstMessages.UPDATING_QUESTION(id));
      try {
        const question = await this.questionRepository.findOneById(id);
        if (!question) {
          this.logger.error(questionConstMessages.QUESTION_NOT_FOUND_ID(id));
          // throw new NotFoundException(questionConstMessages.QUESTION_NOT_FOUND_ID(id));
          throw new InifniNotFoundException(
            ERROR_CODES.QUESTION_NOTFOUND,
            null,
            null,
            id.toString(),
          );
        }

        const { label, type, config, status, formSectionId, updatedBy } = updateDto;

        const updater = await userRepo.findOne({
          where: { id: updatedBy },
        });
        if (!updater) {
          this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
          // throw new NotFoundException(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
          throw new InifniNotFoundException(
            ERROR_CODES.QUESTION_UPDATOR_NOTFOUND,
            null,
            null,
            updatedBy.toString(),
          );
        }

        if (label && type) {
          const existingQuestion = await this.questionRepository.findOneByLabelAndType(label, type);
          if (existingQuestion && existingQuestion.id !== id) {
            this.logger.error(
              questionConstMessages.DUPLICATE_QUESTION_FOUND(label),
            );
            // throw new BadRequestException(questionConstMessages.DUPLICATE_QUESTION);
            throw new InifniBadRequestException(
              ERROR_CODES.QUESTION_DUPLICATE_BADREQUEST,
              null,
              null,
              label,
            );
          }
        }

        // Validate form section if provided
        let formSection: FormSection | undefined = undefined;
        if (formSectionId !== undefined && formSectionId !== null) {
          const foundFormSection = await formSectionRepo.findOne({ 
            where: { id: formSectionId },
          });
          if (!foundFormSection) {
            this.logger.error(FORM_SECTION_MESSAGES.FORM_SEC_NOT_FOUND(formSectionId));
            throw new InifniNotFoundException(
              ERROR_CODES.FORM_SECTION_NOTFOUND,
              null,
              null,
              formSectionId.toString(),
            );
          }
          formSection = foundFormSection;
        }
        Object.assign(question, updateDto, { updatedBy: updater, formSection });
        return await questionRepo.save(question);
      } catch (error) {
        this.logger.error('Error updating question', error?.stack, { error, updateDto, id });
        handleKnownErrors(ERROR_CODES.QUESTION_SAVE_FAILED, error);
      }
    });
  }

  /**
   * Deletes a question by its ID.
   * @param id - ID of the question to delete.
   * @returns The deleted question.
   * @throws NotFoundException if the question does not exist.
   * @throws BadRequestException if the deletion fails.
   */
  async remove(id: number, userId: User) {
    return await this.dataSource.transaction(async (manager) => {
      const questionRepo = manager.getRepository(Question);

      this.logger.log(questionConstMessages.REMOVING_QUESTION(id));
      try {
        const question = await this.questionRepository.findOneById(id);
        if (!question) {
          this.logger.error(questionConstMessages.QUESTION_NOT_FOUND_ID(id));
          // throw new NotFoundException(questionConstMessages.QUESTION_NOT_FOUND_ID(id));
          throw new InifniNotFoundException(
            ERROR_CODES.QUESTION_NOTFOUND,
            null,
            null,
            id.toString(),
          );
        }

        question.updatedBy = userId;
        return await this.questionRepository.softDelete(question.id);
      } catch (error) {
        this.logger.error('Error removing question', error?.stack, { error, id });
        handleKnownErrors(ERROR_CODES.QUESTION_DELETE_FAILED, error);
      }
    });
  }
}