import {
  Injectable
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { IsNull, Repository } from "typeorm";
import { Question, QuestionOptionMap } from "src/common/entities";
import { CommonDataService } from "src/common/services/commonData.service";
import { questionConstMessages } from "src/common/constants/strings-constants";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { AppLoggerService } from "src/common/services/logger.service";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";

@Injectable()
export class QuestionRepository {
  constructor(
    @InjectRepository(Question)
    private readonly questionRepo: Repository<Question>,
    @InjectRepository(QuestionOptionMap)
    private readonly questionOptionMapRepo: Repository<QuestionOptionMap>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Finds a question by its ID.
   * @param id - The ID of the question.
   * @returns The question entity if found, or null.
   */
  async findOneById(id: number): Promise<Question | null> {
    try {
      return await this.commonDataService.findOneById(
        this.questionRepo,
        id,
        true,
        ["formSection","questionOptionMaps","questionOptionMaps.option",]
      );
    } catch (error) {
      this.logger.error('Error in findOneById', error?.stack, { error, id });
      handleKnownErrors(ERROR_CODES.QUESTION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Finds a question by its label and type.
   * @param label - The label of the question.
   * @param type - The type of the question.
   * @returns The question entity if found, or null.
   */
  async findOneByLabelAndType(
    label: string,
    type: string
  ): Promise<Question | null> {
    try {
      const where = { label, type, deletedAt: IsNull() };
      const results = await this.commonDataService.findByData(
        this.questionRepo,
        where
      );
      return results.length > 0 ? results[0] : null;
    } catch (error) {
      this.logger.error('Error in findOneByLabelAndType', error?.stack, { error, label, type });
      handleKnownErrors(ERROR_CODES.QUESTION_FIND_BY_LABEL_TYPE_FAILED, error);
    }
  }

  /**
   * Saves a question entity to the database.
   * @param question - The question entity to save.
   * @returns The saved question entity.
   */
  async save(question: Question): Promise<Question> {
    try {
      return await this.commonDataService.save(this.questionRepo, question);
    } catch (error) {
      this.logger.error('Error in save', error?.stack, { error, question });
      handleKnownErrors(ERROR_CODES.QUESTION_SAVE_FAILED, error);
    }
  }

  /**
   * Counts the number of questions matching the given criteria.
   * @param whereClause - The criteria to match.
   * @returns The count of matching questions.
   */
  async count(whereClause: Record<string, any>): Promise<number> {
    try {
      const results = await this.commonDataService.findByData(
        this.questionRepo,
        whereClause
      );
      return results.length;
    } catch (error) {
      this.logger.error('Error in count', error?.stack, { error, whereClause });
      handleKnownErrors(ERROR_CODES.QUESTION_COUNT_FAILED, error);
    }
  }

  /**
   * Soft deletes a question by its ID and its associated question-option map records.
   * @param id - The ID of the question to delete.
   * @throws NotFoundException if the question is not found.
   */
  async softDelete(id: number): Promise<void> {
    try {
      // Check if the question exists
      const question = await this.questionRepo.findOne({
        where: { id, deletedAt: IsNull() },
      });
      if (!question) {
        // throw new NotFoundException(`Question with ID ${id} not found.`);
        this.logger.error(questionConstMessages.QUESTION_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.QUESTION_NOTFOUND,
          null,
          null,
          id.toString()
        )
      }

      this.logger.log(questionConstMessages.SOFT_DELETING_QUESTION(id));
      // Soft delete the question
      await this.questionRepo.softDelete(id);
      this.logger.log(questionConstMessages.SOFT_DELETED_QUESTION(id));

      this.logger.log(
        questionConstMessages.SOFT_DELETING_ASSOCIATED_RECORDS(id)
      );
      // Soft delete associated question-option map records
      await this.questionOptionMapRepo.softDelete({ question: { id } });
      this.logger.log(
        questionConstMessages.SOFT_DELETED_ASSOCIATED_RECORDS(id)
      );
    } catch (error) {
      this.logger.error('Error in softDelete', error?.stack, { error, id });
      handleKnownErrors(ERROR_CODES.QUESTION_DELETE_FAILED, error);
    }
  }

    /**
 * Retrieves all questions without pagination or filters.
 * This method is useful for scenarios where all questions are needed without any limit or filter.
 * @return An array of questions.
 */
async findAllNoLimit(
  stats?: string,
): Promise<Question[]> {
  this.logger.log(
    questionConstMessages.FIND_ALL_QUESTIONS_REQUEST_RECEIVED
  );
  try {
    const whereClause: any = { deletedAt: IsNull() };

    if (stats) {
      whereClause.status = stats;
    }

    // Only include system user created questions or isAiUser created questions
    whereClause.createdBy =   [{ isSystemUser: true },{ isAiUser: true }];
    this.logger.log(
      questionConstMessages.RETRIEVING_QUESTIONS_WHERE_CLAUSE(whereClause)
    );
    const data = await this.questionRepo.find({
      select: ["id", "label", "type", "status", "config"],
      where: whereClause,
      order: { id: "ASC" },
      relations: ["formSection", "questionOptionMaps", "questionOptionMaps.option"],
    });

    return data;
  } catch (error) {
    this.logger.error('Error in findAllNoLimit', error?.stack, { error, stats });
    handleKnownErrors(ERROR_CODES.COMMON_DATA_SERVICE_GET_FAILED, error);
  }
}
}
