import { Injectable } from '@nestjs/common';
import { CreateQuestionOptionDto } from './dto/create-question-option.dto';
import { UpdateQuestionOptionDto } from './dto/update-question-option.dto';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { User } from 'src/common/entities';
import { QuestionOptionRepository } from './question-option.repository';
import { questionOptionConstMessages } from 'src/common/constants/strings-constants';
import { DataSource } from 'typeorm';
import { InjectDataSource } from '@nestjs/typeorm';

@Injectable()
export class QuestionOptionService {
  constructor(
    private readonly questionOptionRepository: QuestionOptionRepository,
    private readonly logger: AppLoggerService,
    @InjectDataSource()
    private readonly dataSource: DataSource,
  ) {}

  async create(dto: CreateQuestionOptionDto) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(questionOptionConstMessages.CREATE_MAPPING_REQUEST_RECEIVED, dto);
      try {
        return await this.questionOptionRepository.createMappings(dto, manager);
      } catch (error) {
        this.logger.error('Error creating question-option mappings', '', { error });
        handleKnownErrors(ERROR_CODES.QUESTION_OPTION_CREATION_FAILED, error);
      }
    });
  }

  async findAll(limit: number, offset: number, searchText: string) {
    this.logger.log(questionOptionConstMessages.FINDING_ALL_MAPPINGS, {
      limit,
      offset,
      searchText,
    });
    try {
      return await this.questionOptionRepository.findAllMappings(limit, offset, searchText);
    } catch (error) {
      this.logger.error('Error finding all question-option mappings', '', { error });
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_FIND_ALL_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(questionOptionConstMessages.FINDING_MAPPING_BY_ID(id));
    try {
      return await this.questionOptionRepository.findMappingById(id);
    } catch (error) {
      this.logger.error('Error finding question-option mapping by ID', '', { id, error });
      handleKnownErrors(ERROR_CODES.QUESTION_OPTION_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateQuestionOptionDto) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(questionOptionConstMessages.UPDATING_MAPPING(id), dto);
      try {
        return await this.questionOptionRepository.updateMapping(id, dto, manager);
      } catch (error) {
        this.logger.error('Error updating question-option mapping', '', { id, error });
        handleKnownErrors(ERROR_CODES.QUESTION_OPTION_UPDATE_FAILED, error);
      }
    });
  }

  async remove(id: number, user: User) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(questionOptionConstMessages.REMOVING_MAPPING(id));
      try {
        return await this.questionOptionRepository.softDeleteMapping(id, user, manager);
      } catch (error) {
        this.logger.error('Error removing question-option mapping', '', { id, error });
        handleKnownErrors(ERROR_CODES.QUESTION_OPTION_DELETE_FAILED, error);
      }
    });
  }
}
