import { Injectable } from '@nestjs/common';
import { ProgramRepository } from './program.repository';
import { ProgramTypeRepository } from '../program-type/program-type.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CreateProgramDto, GroupedProgramDto } from './dto/create-program.dto';
import { UpdateProgramDto } from './dto/update-program.dto';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { EntityManager } from 'typeorm';

@Injectable()
export class ProgramValidationService {
  constructor(
    private readonly programRepository: ProgramRepository,
    private readonly programTypeRepository: ProgramTypeRepository,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Validates invoice sender address data
   */
  private async validateInvoiceAddress(addressDto: any): Promise<void> {
    if (!addressDto) return;

    // Validate required fields based on your business rules
    if (!addressDto.type) {
      throw new InifniBadRequestException(
       ERROR_CODES.INVALID_ADDRESS_TYPE,
        null,
        null,
        'Address type is required'
      );
    }

    // Add more validation as needed
    if (addressDto.pincode && !/^\d{6}$/.test(addressDto.pincode)) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_PINCODE,
        null,
        null,
        'Pincode must be 6 digits'
      );
    }
  }

  /**
   * Enhanced validateCreateProgram with address validation
   */
  async validateCreateProgram(createDto: CreateProgramDto, manager?: EntityManager): Promise<void> {
    // Validate program type exists and supports grouping if needed
    const programType = await this.programTypeRepository.findOneById(createDto.typeId);
    if (!programType) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
        null,
        null,
        createDto.typeId.toString()
      );
    }

    // Validate venue address if provided
    if (createDto.venueAddress) {
      await this.validateInvoiceAddress(createDto.venueAddress);
    }

    // If grouped programs are provided, validate that program type supports it
    if (createDto.groupedPrograms && createDto.groupedPrograms.length > 0) {
      if (!programType.isGroupedProgram) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_TYPE_DOES_NOT_SUPPORT_GROUPING,
          null,
          null,
          `Program type '${programType.name}' does not support grouped programs`
        );
      }

      // Validate grouped programs structure and their addresses
      await this.validateGroupedPrograms(createDto.groupedPrograms);
    }

    // Validate business rules
    await this.validateBusinessRules(createDto, manager);
  }

  /**
   * Enhanced validateUpdateProgram with address validation
   */
async validateUpdateProgram(
  id: number, 
  updateDto: UpdateProgramDto, 
  manager?: EntityManager
): Promise<void> {
  // Get existing program
  const existingProgram = await this.programRepository.findOneById(id, manager);
  if (!existingProgram) {
    throw new InifniBadRequestException(
      ERROR_CODES.PROGRAM_NOTFOUND,
      null,
      null,
      id.toString()
    );
  }

  // Validate venue address if provided
  if (updateDto.venueAddress) {
    await this.validateInvoiceAddress(updateDto.venueAddress);
  }

  // If updating grouped programs, validate constraints
  if (updateDto.groupedPrograms && updateDto.groupedPrograms.length > 0) {
    if (!existingProgram.isPrimaryProgram) {
      throw new InifniBadRequestException(
        ERROR_CODES.CANNOT_UPDATE_GROUPED_PROGRAMS_ON_NON_PRIMARY,
        null,
        null,
        'Cannot update grouped programs on a non-primary program'
      );
    }

    // Validate that program type supports grouping
    const programType = await this.programTypeRepository.findOneById(existingProgram.typeId);
    if (!programType || !programType.isGroupedProgram) {
      throw new InifniBadRequestException(
        'PROGRAM_TYPE_DOES_NOT_SUPPORT_GROUPING',
        null,
        null,
        'Program type does not support grouped programs'
      );
    }

    // Get existing child programs for validation
    const existingChildPrograms = await this.programRepository.findGroupedPrograms(
      existingProgram.groupId, 
      manager
    );
    const existingChildIds = existingChildPrograms
      .filter(p => !p.isPrimaryProgram)
      .map(p => p.id);

    // Validate grouped programs structure including addresses and IDs
    await this.validateGroupedProgramsForUpdate(
      updateDto.groupedPrograms, 
      existingChildIds, 
      updateDto.deletedProgramIds || [],
      manager
    );
  }

  // Validate deletedProgramIds if provided
  if (updateDto.deletedProgramIds && updateDto.deletedProgramIds.length > 0) {
    await this.validateDeletedProgramIds(
      updateDto.deletedProgramIds,
      existingProgram.groupId,
      manager
    );
  }

  // Validate business rules for update
  // await this.validateBusinessRulesForUpdate(id, updateDto, existingProgram, manager);
}

/**
 * Validates grouped programs structure for updates with explicit deletion support
 */
private async validateGroupedProgramsForUpdate(
  groupedPrograms: GroupedProgramDto[], 
  existingChildIds: number[],
  deletedProgramIds: number[] = [],
  manager?: EntityManager
): Promise<void> {
  // Check for duplicate display orders
  const displayOrders = groupedPrograms.map(gp => gp.groupDisplayOrder);
  const uniqueDisplayOrders = new Set(displayOrders);
  
  if (displayOrders.length !== uniqueDisplayOrders.size) {
    throw new InifniBadRequestException(
      ERROR_CODES.DUPLICATE_DISPLAY_ORDER,
      null,
      null,
      'Duplicate display orders found in grouped programs'
    );
  }

  // Validate display order starts from 2 (1 is reserved for primary)
  const invalidOrders = displayOrders.filter(order => order < 2);
  if (invalidOrders.length > 0) {
    throw new InifniBadRequestException(
      ERROR_CODES.INVALID_GROUP_DISPLAY_ORDER,
      null,
      null,
      'Group display order must start from 2 (1 is reserved for primary program)'
    );
  }

  // Check for duplicate names within the group
  const names = groupedPrograms.map(gp => gp.name.toLowerCase().trim());
  const uniqueNames = new Set(names);
  
  if (names.length !== uniqueNames.size) {
    throw new InifniBadRequestException(
      ERROR_CODES.DUPLICATE_PROGRAM_NAMES_IN_GROUP,
      null,
      null,
      'Duplicate program names found in grouped programs'
    );
  }

  // Check for duplicate codes within the group (if provided)
  // const codes = groupedPrograms
  //   .filter(gp => gp.code)
  //   .map(gp => gp.code!.toLowerCase().trim());
  
  // if (codes.length > 0) {
  //   const uniqueCodes = new Set(codes);
  //   if (codes.length !== uniqueCodes.size) {
  //     throw new InifniBadRequestException(
  //       ERROR_CODES.DUPLICATE_PROGRAM_CODES_IN_GROUP,
  //       null,
  //       null,
  //       'Duplicate program codes found in grouped programs'
  //     );
  //   }
  // }

  // Validate IDs for updates
  const providedIds = groupedPrograms
    .filter(gp => gp.id)
    .map(gp => gp.id!);

  // Check for duplicate IDs in the request
  const uniqueProvidedIds = new Set(providedIds);
  if (providedIds.length !== uniqueProvidedIds.size) {
    throw new InifniBadRequestException(
      ERROR_CODES.DUPLICATE_IDS_IN_REQUEST,
      null,
      null,
      'Duplicate program IDs found in grouped programs update request'
    );
  }

  // Filter out programs that are marked for deletion
  const availableChildIds = existingChildIds.filter(id => !deletedProgramIds.includes(id));
  
  // Validate that all provided IDs exist in the available child programs (not marked for deletion)
  const invalidIds = providedIds.filter(id => !availableChildIds.includes(id));
  if (invalidIds.length > 0) {
    throw new InifniBadRequestException(
      ERROR_CODES.INVALID_PROGRAM_IDS,
      null,
      null,
      `Invalid program IDs provided (either don't exist or marked for deletion): ${invalidIds.join(', ')}`
    );
  }

  // Check that programs marked for deletion are not also being updated
  const conflictingIds = providedIds.filter(id => deletedProgramIds.includes(id));
  if (conflictingIds.length > 0) {
    throw new InifniBadRequestException(
      ERROR_CODES.CONFLICTING_PROGRAM_IDS,
      null,
      null,
      `Programs cannot be both updated and deleted: ${conflictingIds.join(', ')}`
    );
  }

  // Validate venue addresses for each grouped program
  for (const groupedProgram of groupedPrograms) {
    if (groupedProgram.venueAddress) {
      await this.validateInvoiceAddress(groupedProgram.venueAddress);
    }
  }

  // Validate business rules for each grouped program
  for (const groupedProgram of groupedPrograms) {
    await this.validateSingleProgramBusinessRules(groupedProgram);
  }
}

  /**
   * Validates deletion constraints
   */
  async validateDeleteProgram(id: number, manager?: EntityManager): Promise<void> {
    const program = await this.programRepository.findOneById(id, manager);
    if (!program) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        id.toString()
      );
    }

    // Add any deletion constraints here
    // For example: check if program has active registrations, etc.
    
    this.logger.log(`Validation passed for deleting program ${id}`);
  }

  /**
   * Validates grouped programs structure
   */
  private async validateGroupedPrograms(groupedPrograms: GroupedProgramDto[]): Promise<void> {
    // Check for duplicate display orders
    const displayOrders = groupedPrograms.map(gp => gp.groupDisplayOrder);
    const uniqueDisplayOrders = new Set(displayOrders);
    
    if (displayOrders.length !== uniqueDisplayOrders.size) {
      throw new InifniBadRequestException(
        ERROR_CODES.DUPLICATE_DISPLAY_ORDER,
        null,
        null,
        'Duplicate display orders found in grouped programs'
      );
    }

    // Validate display order starts from 2 (1 is reserved for primary)
    const invalidOrders = displayOrders.filter(order => order < 2);
    if (invalidOrders.length > 0) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_GROUP_DISPLAY_ORDER,
        null,
        null,
        'Group display order must start from 2 (1 is reserved for primary program)'
      );
    }

    // Check for duplicate names within the group
    const names = groupedPrograms.map(gp => gp.name.toLowerCase().trim());
    const uniqueNames = new Set(names);
    
    if (names.length !== uniqueNames.size) {
      throw new InifniBadRequestException(
        ERROR_CODES.DUPLICATE_PROGRAM_NAMES_IN_GROUP,
        null,
        null,
        'Duplicate program names found in grouped programs'
      );
    }

    // Check for duplicate codes within the group (if provided)
    // const codes = groupedPrograms
    //   .filter(gp => gp.code)
    //   .map(gp => gp.code!.toLowerCase().trim());
    
    // if (codes.length > 0) {
    //   const uniqueCodes = new Set(codes);
    //   if (codes.length !== uniqueCodes.size) {
    //     throw new InifniBadRequestException(
    //       ERROR_CODES.DUPLICATE_PROGRAM_CODES_IN_GROUP,
    //       null,
    //       null,
    //       'Duplicate program codes found in grouped programs'
    //     );
    //   }
    // }

    // Validate venue addresses for each grouped program
    for (const groupedProgram of groupedPrograms) {
      if (groupedProgram.venueAddress) {
        await this.validateInvoiceAddress(groupedProgram.venueAddress);
      }
    }
  }

  /**
   * Validates business rules for program creation
   */
  private async validateBusinessRules(createDto: CreateProgramDto, manager?: EntityManager): Promise<void> {
    // Validate dates
    if (createDto.startsAt && createDto.endsAt) {
      if (new Date(createDto.startsAt) > new Date(createDto.endsAt)) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_DATE_RANGE,
          null,
          null,
          'Start date cannot be after end date'
        );
      }
    }

    // Validate registration dates
    if (createDto.registrationStartsAt && createDto.registrationEndsAt) {
      if (new Date(createDto.registrationStartsAt) > new Date(createDto.registrationEndsAt)) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_REGISTRATION_DATE_RANGE,
          null,
          null,
          'Registration start date cannot be after registration end date'
        );
      }
    }

    // Validate seats
    if (createDto.totalSeats !== undefined && createDto.availableSeats !== undefined) {
      if (createDto.availableSeats > createDto.totalSeats) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_SEAT_CONFIGURATION,
          null,
          null,
          'Available seats cannot exceed total seats'
        );
      }
    }

    // Validate times
    if (createDto.startsAt && createDto.endsAt) {
      if (createDto.startsAt >= createDto.endsAt) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_TIME_RANGE,
          null,
          null,
          'Start time must be before end time'
        );
      }
    }

    // Validate checkin/checkout dates if applicable
    if (createDto.hasCheckinCheckout && createDto.checkinAt && createDto.checkoutAt) {
      if (new Date(createDto.checkinAt) > new Date(createDto.checkoutAt)) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_CHECKIN_CHECKOUT_DATE_RANGE,
          null,
          null,
          'Check-in date cannot be after check-out date'
        );
      }
    }

    // Validate grouped program constraints
    if (createDto.groupedPrograms && createDto.groupedPrograms.length > 0) {
      // Validate each grouped program follows same business rules
      for (const groupedProgram of createDto.groupedPrograms) {
        await this.validateSingleProgramBusinessRules(groupedProgram);
      }
    }
  }

  /**
   * Validates business rules for a single program (used for grouped programs too)
   */
  private async validateSingleProgramBusinessRules(programData: any): Promise<void> {
    // Validate dates
    if (programData.startAt && programData.endAt) {
      if (new Date(programData.startAt) > new Date(programData.endAt)) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_DATE_RANGE,
          null,
          null,
          `Start date cannot be after end date for program: ${programData.name}`
        );
      }
    }

    // // Validate times
    // if (programData.startTime && programData.endTime) {
    //   if (programData.startTime >= programData.endTime) {
    //     throw new InifniBadRequestException(
    //       'INVALID_TIME_RANGE',
    //       null,
    //       null,
    //       `Start time must be before end time for program: ${programData.name}`
    //     );
    //   }
    // }

    // Validate seats
    if (programData.totalSeats !== undefined && programData.totalSeats < 0) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_SEAT_COUNT,
        null,
        null,
        `Total seats cannot be negative for program: ${programData.name}`
      );
    }

    // Validate checkin/checkout dates if applicable
    if (programData.hasCheckinCheckout && programData.checkinAt && programData.checkoutAt) {
      if (new Date(programData.checkinAt) > new Date(programData.checkoutAt)) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_CHECKIN_CHECKOUT_DATE_RANGE,
          null,
          null,
          `Check-in date cannot be after check-out date for program: ${programData.name}`
        );
      }
    }
  }

  /**
   * Validates business rules for program updates
   */
//   private async validateBusinessRulesForUpdate(
//     id: number, 
//     updateDto: UpdateProgramDto, 
//     existingProgram: any,
//     manager?: EntityManager
//   ): Promise<void> {
//     // Apply same business rules as creation for updated fields
//     const mergedData = { ...existingProgram, ...updateDto };
    
//     // Validate dates
//     if (mergedData.startDate && mergedData.endDate) {
//       if (new Date(mergedData.startDate) > new Date(mergedData.endDate)) {
//         throw new InifniBadRequestException(
//           'INVALID_DATE_RANGE',
//           null,
//           null,
//           'Start date cannot be after end date'
//         );
//       }
//     }

//     // Validate registration dates
//     if (mergedData.registrationStartDate && mergedData.registrationEndDate) {
//       if (new Date(mergedData.registrationStartDate) > new Date(mergedData.registrationEndDate)) {
//         throw new InifniBadRequestException(
//           'INVALID_REGISTRATION_DATE_RANGE',
//           null,
//           null,
//           'Registration start date cannot be after registration end date'
//         );
//       }
//     }

//     // Validate seats
//     if (mergedData.totalSeats !== undefined && mergedData.availableSeats !== undefined) {
//       if (mergedData.availableSeats > mergedData.totalSeats) {
//         throw new InifniBadRequestException(
//           'INVALID_SEAT_CONFIGURATION',
//           null,
//           null,
//           'Available seats cannot exceed total seats'
//         );
//       }
//     }

//     // Validate times
//     if (mergedData.startTime && mergedData.endTime) {
//       if (mergedData.startTime >= mergedData.endTime) {
//         throw new InifniBadRequestException(
//           'INVALID_TIME_RANGE',
//           null,
//           null,
//           'Start time must be before end time'
//         );
//       }
//     }

//     // Validate that group hierarchy properties are not being changed incorrectly
//     if (existingProgram.isPrimaryProgram && updateDto.isPrimaryProgram === false) {
//       throw new InifniBadRequestException(
//         'CANNOT_CHANGE_PRIMARY_STATUS',
//         null,
//         null,
//         'Cannot change primary program status'
//       );
//     }

//     // Prevent changing groupId directly
//     if (updateDto.hasOwnProperty('groupId') && updateDto.groupId !== existingProgram.groupId) {
//       throw new InifniBadRequestException(
//         'CANNOT_CHANGE_GROUP_ID',
//         null,
//         null,
//         'Group ID cannot be changed directly'
//       );
//     }

//     // Prevent changing primaryProgramId directly
//     if (updateDto.hasOwnProperty('primaryProgramId') && updateDto.primaryProgramId !== existingProgram.primaryProgramId) {
//       throw new InifniBadRequestException(
//         'CANNOT_CHANGE_PRIMARY_PROGRAM_ID',
//         null,
//         null,
//         'Primary program ID cannot be changed directly'
//       );
//     }

//     // Additional validation for grouped programs
//     if (existingProgram.groupId && !existingProgram.isPrimaryProgram) {
//       // Child programs cannot be updated with grouped programs
//       if (updateDto.groupedPrograms && updateDto.groupedPrograms.length > 0) {
//         throw new InifniBadRequestException(
//           'CHILD_PROGRAM_CANNOT_HAVE_GROUPED_PROGRAMS',
//           null,
//           null,
//           'Child programs cannot have their own grouped programs'
//         );
//       }
//     }

//     // Validate grouped program updates follow business rules
//     if (updateDto.groupedPrograms && updateDto.groupedPrograms.length > 0) {
//       for (const groupedProgram of updateDto.groupedPrograms) {
//         await this.validateSingleProgramBusinessRules(groupedProgram);
//       }
//     }
//   }

  /**
   * Validates that a program can be safely deleted
   */
  async validateDeletionSafety(programId: number, manager?: EntityManager): Promise<{
    canDelete: boolean;
    reasons: string[];
  }> {
    const reasons: string[] = [];
    
    try {
      const program = await this.programRepository.findOneById(programId, manager);
      if (!program) {
        return { canDelete: false, reasons: ['Program not found'] };
      }

      // Check if program has active sessions with registrations
      // This would require additional repository methods to check registrations
      // For now, we'll assume it's safe to delete

      // Check if this is a primary program with active child programs
      if (program.isPrimaryProgram && program.groupId) {
        const childPrograms = await this.programRepository.findGroupedPrograms(program.groupId, manager);
        const activeChildren = childPrograms.filter(cp => !cp.isPrimaryProgram && cp.isActive);
        
        if (activeChildren.length > 0) {
          this.logger.warn(`Primary program ${programId} has ${activeChildren.length} active child programs that will be deleted`);
          // This is just a warning, not blocking deletion
        }
      }

      // Add more validation rules here as needed
      // Examples:
      // - Check for active registrations
      // - Check for ongoing sessions
      // - Check for payment dependencies
      // - Check for integration dependencies

      return { canDelete: true, reasons: [] };
    } catch (error) {
      this.logger.error('Error validating deletion safety:', error);
      return { canDelete: false, reasons: ['Validation error occurred'] };
    }
  }

  /**
   * Validates program hierarchy constraints
   */
  async validateProgramHierarchy(programData: any, manager?: EntityManager): Promise<void> {
    // Validate that a program cannot be both primary and have a primaryProgramId
    if (programData.isPrimaryProgram && programData.primaryProgramId) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_PROGRAM_HIERARCHY,
        null,
        null,
        'A program cannot be both primary and have a primary program ID'
      );
    }

    // Validate that if groupId is provided, either isPrimaryProgram is true or primaryProgramId is provided
    if (programData.groupId) {
      if (!programData.isPrimaryProgram && !programData.primaryProgramId) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_GROUP_CONFIGURATION,
          null,
          null,
          'Programs in a group must either be primary or have a primary program ID'
        );
      }
    }

    // Validate that if primaryProgramId is provided, the primary program exists and is actually primary
    if (programData.primaryProgramId) {
      const primaryProgram = await this.programRepository.findOneById(programData.primaryProgramId, manager);
      if (!primaryProgram) {
        throw new InifniBadRequestException(
          ERROR_CODES.PRIMARY_PROGRAM_NOT_FOUND,
          null,
          null,
          `Primary program with ID ${programData.primaryProgramId} not found`
        );
      }

      if (!primaryProgram.isPrimaryProgram) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_PRIMARY_PROGRAM,
          null,
          null,
          `Program with ID ${programData.primaryProgramId} is not a primary program`
        );
      }

      // Ensure the groupId matches the primary program's groupId
      if (programData.groupId && programData.groupId !== primaryProgram.groupId) {
        throw new InifniBadRequestException(
          ERROR_CODES.GROUP_ID_MISMATCH,
          null,
          null,
          'Group ID must match the primary program\'s group ID'
        );
      }
    }
  }

  /**
   * Validates that program type change is allowed
   */
  async validateProgramTypeChange(
    programId: number, 
    newTypeId: number, 
    manager?: EntityManager
  ): Promise<void> {
    const existingProgram = await this.programRepository.findOneById(programId, manager);
    if (!existingProgram) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        programId.toString()
      );
    }

    // If program is part of a group, validate type change compatibility
    if (existingProgram.groupId) {
      const newProgramType = await this.programTypeRepository.findOneById(newTypeId);
      if (!newProgramType) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
          null,
          null,
          newTypeId.toString()
        );
      }

      // If changing to a type that doesn't support grouping, but program is in a group
      if (!newProgramType.isGroupedProgram) {
        throw new InifniBadRequestException(
          ERROR_CODES.CANNOT_CHANGE_TO_NON_GROUPED_TYPE,
          null,
          null,
          'Cannot change to a program type that does not support grouping while program is in a group'
        );
      }
    }
  }

  /**
   * Validates group display order constraints
   */
  validateGroupDisplayOrder(
    displayOrder: number, 
    isPrimaryProgram: boolean, 
    existingOrders: number[]
  ): void {
    // Primary program must have display order 1
    if (isPrimaryProgram && displayOrder !== 1) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_PRIMARY_DISPLAY_ORDER,
        null,
        null,
        'Primary program must have display order 1'
      );
    }

    // Child programs must have display order > 1
    if (!isPrimaryProgram && displayOrder <= 1) {
      throw new InifniBadRequestException(
       ERROR_CODES.INVALID_CHILD_DISPLAY_ORDER,
        null,
        null,
        'Child programs must have display order greater than 1'
      );
    }

    // Check for duplicate display orders
    if (existingOrders.includes(displayOrder)) {
      throw new InifniBadRequestException(
        ERROR_CODES.DUPLICATE_DISPLAY_ORDER,
        null,
        null,
        `Display order ${displayOrder} is already in use`
      );
    }
  }

  /**
   * Validates that the programs marked for deletion actually exist and belong to the group
   */
  private async validateDeletedProgramIds(
    deletedProgramIds: number[],
    groupId: string | null,
    manager?: EntityManager
  ): Promise<void> {
    if (!groupId) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_DELETION_REQUEST,
        null,
        null,
        'Cannot delete programs from a non-grouped program'
      );
    }

    // Check for duplicate IDs in deletion list
    const uniqueDeletedIds = new Set(deletedProgramIds);
    if (deletedProgramIds.length !== uniqueDeletedIds.size) {
      throw new InifniBadRequestException(
        ERROR_CODES.DUPLICATE_IDS_IN_DELETION_REQUEST,
        null,
        null,
        'Duplicate program IDs found in deletion request'
      );
    }

    // Get existing child programs
    const existingChildPrograms = await this.programRepository.findGroupedPrograms(groupId, manager);
    const existingChildIds = existingChildPrograms
      .filter(p => !p.isPrimaryProgram)
      .map(p => p.id);

    // Validate that all deleted IDs exist in the group
    const invalidDeletedIds = deletedProgramIds.filter(id => !existingChildIds.includes(id));
    if (invalidDeletedIds.length > 0) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_DELETED_PROGRAM_IDS,
        null,
        null,
        `Invalid program IDs for deletion (don't exist in group): ${invalidDeletedIds.join(', ')}`
      );
    }

    // Validate that we're not trying to delete the primary program
    const primaryProgram = existingChildPrograms.find(p => p.isPrimaryProgram);
    if (primaryProgram && deletedProgramIds.includes(primaryProgram.id)) {
      throw new InifniBadRequestException(
        ERROR_CODES.CANNOT_DELETE_PRIMARY_PROGRAM,
        null,
        null,
        'Cannot delete the primary program using deletedProgramIds. Delete the entire group instead.'
      );
    }
  }
}