import { ApiProperty } from '@nestjs/swagger';
import {
  IsString,
  IsOptional,
  IsNumber,
  IsEnum,
  IsBoolean,
  IsDateString,
  Min,
  Max,
  IsArray,
  ValidateNested,
  MaxLength,
} from 'class-validator';
import { Type } from 'class-transformer';
import { CreateProgramSessionDto } from 'src/program-session/dto/create-program-session.dto';
import { CreateAddressDto } from 'src/address/dto/create-address.dto';
import {
  ModeOfOperationEnum,
  OnlineTypeEnum,
  ProgramStatusEnum,
  RegistrationLevelEnum,
  FrequencyEnum,
} from 'src/common/enum/program.enums';
import { TdsApplicabilityEnum } from 'src/common/enum/tds-applicability.enum';
import { SubProgramTypeEnum } from 'src/common/enum/sub-program-type.enum';

export class GroupedProgramDto {

  @ApiProperty({ description: 'Program ID (for updates)', example: 123, required: false })
  @IsOptional()
  @IsNumber()
  id?: number;
  
  @ApiProperty({ description: 'Program name', example: 'Session 2' })
  @IsString()
  name: string;

  @ApiProperty({ description: 'Program code', example: 'AYW2', required: false })
  @IsOptional()
  @IsString()
  @MaxLength(4, { message: 'Code must be at most 4 characters long' })
  code?: string;

  @ApiProperty({ description: 'Program description', required: false })
  @IsOptional()
  @IsString()
  description?: string;

  @ApiProperty({ description: 'Start at', required: false })
  @IsOptional()
  @IsDateString()
  startsAt?: Date;

  @ApiProperty({ description: 'End at', required: false })
  @IsOptional()
  @IsDateString()
  endsAt?: Date;

  @ApiProperty({ description: 'Duration', example: '8 hours', required: false })
  @IsOptional()
  @IsString()
  duration?: string;

  @ApiProperty({ description: 'Requires payment', required: false })
  @IsOptional()
  @IsBoolean()
  requiresPayment?: boolean;

  @ApiProperty({ description: 'Requires attendance for all sessions', required: false })
  @IsOptional()
  @IsBoolean()
  requiresAttendanceAllSessions?: boolean;

  @ApiProperty({ description: 'Allows minors', required: false })
  @IsOptional()
  @IsBoolean()
  allowsMinors?: boolean;

  @ApiProperty({ description: 'Allows proxy registration', required: false })
  @IsOptional()
  @IsBoolean()
  allowsProxyRegistration?: boolean;

  @ApiProperty({ description: 'Requires approval', required: false })
  @IsOptional()
  @IsBoolean()
  requiresApproval?: boolean;

  @ApiProperty({ description: 'Registration level', enum: RegistrationLevelEnum, required: false })
  @IsOptional()
  @IsEnum(RegistrationLevelEnum)
  registrationLevel?: RegistrationLevelEnum;

  @ApiProperty({ description: 'Total seats', example: 50, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  totalSeats?: number;

  @ApiProperty({
    description: 'Registration starts at',
    example: '2024-01-01T00:00:00Z',
    required: false,
  })
  @IsOptional()
  @IsDateString()
  registrationStartsAt?: Date;

  @ApiProperty({
    description: 'Registration ends at',
    example: '2024-01-14T23:59:59Z',
    required: false,
  })
  @IsOptional()
  @IsDateString()
  registrationEndsAt?: Date;

  @ApiProperty({ description: 'Base price', example: 199.99, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  basePrice?: number;

  @ApiProperty({ description: 'Venue', required: false })
  @IsOptional()
  @IsString()
  venue?: string;

  @ApiProperty({ description: 'Group display order', example: 2 })
  @IsNumber()
  @Min(1)
  groupDisplayOrder: number;

  @ApiProperty({ description: 'Number of sessions', example: 0, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  noOfSession?: number;

  @ApiProperty({ description: 'Meta information (JSON)', required: false, type: Object })
  @IsOptional()
  meta?: any;

  // Allow override of any inherited property
  @ApiProperty({ description: 'Mode of operation', enum: ModeOfOperationEnum, required: false })
  @IsOptional()
  @IsEnum(ModeOfOperationEnum)
  modeOfOperation?: ModeOfOperationEnum;

  @ApiProperty({ description: 'Online type', enum: OnlineTypeEnum, required: false })
  @IsOptional()
  @IsEnum(OnlineTypeEnum)
  onlineType?: OnlineTypeEnum;

  @ApiProperty({ description: 'Max session duration days', required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  maxSessionDurationDays?: number;

  @ApiProperty({ description: 'Has multiple sessions', required: false })
  @IsOptional()
  @IsBoolean()
  hasMultipleSessions?: boolean;

  @ApiProperty({ description: 'Frequency', enum: FrequencyEnum, required: false })
  @IsOptional()
  @IsEnum(FrequencyEnum)
  frequency?: FrequencyEnum;

  @ApiProperty({ description: 'Limited seats', required: false })
  @IsOptional()
  @IsBoolean()
  limitedSeats?: boolean;

  @ApiProperty({ description: 'Program status', enum: ProgramStatusEnum, required: false })
  @IsOptional()
  @IsEnum(ProgramStatusEnum)
  status?: ProgramStatusEnum;

  @ApiProperty({ description: 'Is active', required: false })
  @IsOptional()
  @IsBoolean()
  isActive?: boolean;

  @ApiProperty({ description: 'Program fee', example: 1000.0, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  programFee?: number;

  @ApiProperty({ description: 'GST percentage', example: 18.0, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  gstPercentage?: number;
  @ApiProperty({ description: 'CGST rate', example: 9, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  cgst?: number;

  @ApiProperty({ description: 'SGST rate', example: 9, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  sgst?: number;

  @ApiProperty({ description: 'IGST rate', example: 18, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  igst?: number;



  @ApiProperty({ description: 'TDS percentage', example: 10, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  tdsPercent?: number;

  @ApiProperty({ enum: TdsApplicabilityEnum, required: false })
  @IsOptional()
  @IsEnum(TdsApplicabilityEnum)
  tdsApplicability?: TdsApplicabilityEnum;

  @ApiProperty({ description: 'Invoice sender name', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderName?: string;

  @ApiProperty({ description: 'Invoice sender PAN', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderPan?: string;

  @ApiProperty({ description: 'Invoice sender CIN', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderCin?: string;

  @ApiProperty({ description: 'Venue address', required: false, type: CreateAddressDto })
  @IsOptional()
  @ValidateNested()
  @Type(() => CreateAddressDto)
  venueAddress?: CreateAddressDto;

  @ApiProperty({ description: 'Invoice sender address', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderAddress?: string;


  @ApiProperty({ description: 'GST number', example: '27AAACG1234R1Z5', required: false })
  @IsOptional()
  @IsString()
  gstNumber?: string;

  @ApiProperty({ description: 'Currency', example: 'INR', required: false })
  @IsOptional()
  @IsString()
  currency?: string;


  @ApiProperty({ description: 'Helpline number', required: false })
  @IsOptional()
  @IsString()
  helplineNumber?: string;

  @ApiProperty({ description: 'Email sender name', required: false })
  @IsOptional()
  @IsString()
  emailSenderName?: string;

  @ApiProperty({ description: 'Venue name in emails', required: false })
  @IsOptional()
  @IsString()
  venueNameInEmails?: string;

  @ApiProperty({ description: 'Launch date', required: false })
  @IsOptional()
  @IsDateString()
  launchDate?: Date;

  @ApiProperty({ description: 'Seeker can share experience', required: false })
  @IsOptional()
  @IsBoolean()
  seekerCanShareExperience?: boolean;


  @ApiProperty({ description: 'Total bed count', example: 100, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  totalBedCount?: number;

  // @ApiProperty({ description: 'Is residence required', required: false })
  // @IsOptional()
  // @IsBoolean()
  // isResidenceRequired?: boolean;

  @ApiProperty({ description: 'Is travel involved', required: false })
  @IsOptional()
  @IsBoolean()
  isTravelInvolved?: boolean;

  @ApiProperty({ description: 'Has checkin/checkout', required: false })
  @IsOptional()
  @IsBoolean()
  hasCheckinCheckout?: boolean;

  @ApiProperty({ description: 'Check-in at', required: false })
  @IsOptional()
  @IsDateString()
  checkinAt?: Date;

  @ApiProperty({ description: 'Check-out at', required: false })
  @IsOptional()
  @IsDateString()
  checkoutAt?: Date;

  @ApiProperty({ description: 'Banner image URL', required: false })
  @IsOptional()
  @IsString()
  bannerImageUrl?: string;

  @ApiProperty({ description: 'Banner animation URL', required: false })
  @IsOptional()
  @IsString()
  bannerAnimationUrl?: string;

  // subProgramType enum
  @ApiProperty({ description: 'Sub program type', required: false, enum: SubProgramTypeEnum })
  @IsOptional()
  @IsEnum(SubProgramTypeEnum)
  subProgramType?: SubProgramTypeEnum;

  @ApiProperty({ description: 'Check-in-ends-at', required: false })
  @IsOptional()
  @IsDateString()
  checkinEndsAt?: Date;

  @ApiProperty({ description: 'Check-out-ends-at', required: false })
  @IsOptional()
  @IsDateString()
  checkoutEndsAt?: Date;

  //logo_url
  @ApiProperty({ description: 'Logo image URL', required: false })
  @IsOptional()
  @IsString()
  logoUrl?: string;
}

export class CreateProgramDto {
  @ApiProperty({ description: 'Program type ID', example: 1 })
  @IsNumber()
  typeId: number;

  @ApiProperty({ description: 'Program name', example: 'HDB' })
  @IsString()
  name: string;

  @ApiProperty({ description: 'Program code', example: 'AYW2', required: false })
  @IsOptional()
  @IsString()
  @MaxLength(4, { message: 'Code must be at most 4 characters long' })
  code?: string;

  @ApiProperty({ description: 'Program description', required: false })
  @IsOptional()
  @IsString()
  description?: string;

  @ApiProperty({
    description: 'Mode of operation',
    enum: ModeOfOperationEnum,
    required: false,
  })
  @IsOptional()
  @IsEnum(ModeOfOperationEnum)
  modeOfOperation?: ModeOfOperationEnum;

  @ApiProperty({
    description: 'Online type',
    enum: OnlineTypeEnum,
    required: false,
  })
  @IsOptional()
  @IsEnum(OnlineTypeEnum)
  onlineType?: OnlineTypeEnum;

  @ApiProperty({ description: 'Max session duration days', required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  maxSessionDurationDays?: number;

  @ApiProperty({ description: 'Has multiple sessions', required: false })
  @IsOptional()
  @IsBoolean()
  hasMultipleSessions?: boolean;

  @ApiProperty({ description: 'Frequency', enum: FrequencyEnum, required: false })
  @IsOptional()
  @IsEnum(FrequencyEnum)
  frequency?: FrequencyEnum;

  @ApiProperty({ description: 'Start at', required: false })
  @IsOptional()
  @IsDateString()
  startsAt?: Date;

  @ApiProperty({ description: 'End at', required: false })
  @IsOptional()
  @IsDateString()
  endsAt?: Date;

  @ApiProperty({ description: 'Duration', example: '8 hours', required: false })
  @IsOptional()
  @IsString()
  duration?: string;

  @ApiProperty({ description: 'Requires payment', required: false })
  @IsOptional()
  @IsBoolean()
  requiresPayment?: boolean;

  @ApiProperty({ description: 'Requires attendance for all sessions', required: false })
  @IsOptional()
  @IsBoolean()
  requiresAttendanceAllSessions?: boolean;

  @ApiProperty({ description: 'Allows minors', required: false })
  @IsOptional()
  @IsBoolean()
  allowsMinors?: boolean;

  @ApiProperty({ description: 'Allows proxy registration', required: false })
  @IsOptional()
  @IsBoolean()
  allowsProxyRegistration?: boolean;

  @ApiProperty({ description: 'Requires approval', required: false })
  @IsOptional()
  @IsBoolean()
  requiresApproval?: boolean;

  @ApiProperty({ description: 'Registration level', enum: RegistrationLevelEnum, required: false })
  @IsOptional()
  @IsEnum(RegistrationLevelEnum)
  registrationLevel?: RegistrationLevelEnum;

  @ApiProperty({ description: 'Total seats', example: 50, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  totalSeats?: number;

  @ApiProperty({ description: 'Waitlist trigger count', example: 5, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  waitlistTriggerCount?: number;

  @ApiProperty({ description: 'Available seats', example: 45, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  availableSeats?: number;

  @ApiProperty({ description: 'Waitlist applicable', required: false })
  @IsOptional()
  @IsBoolean()
  waitlistApplicable?: boolean;

  @ApiProperty({ description: 'Max capacity', required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  maxCapacity?: number;

  @ApiProperty({ description: 'Meta information (JSON)', required: false, type: Object })
  @IsOptional()
  meta?: any;


  @ApiProperty({
    description: 'Registration starts at',
    example: '2024-01-01T00:00:00Z',
    required: false,
  })
  @IsOptional()
  @IsDateString()
  registrationStartsAt?: Date;

  @ApiProperty({
    description: 'Registration ends at',
    example: '2024-01-14T23:59:59Z',
    required: false,
  })
  @IsOptional()
  @IsDateString()
  registrationEndsAt?: Date;

  @ApiProperty({ description: 'Base price', example: 199.99, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  basePrice?: number;

  @ApiProperty({ description: 'Program fee', example: 1000.0, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  programFee?: number;

  @ApiProperty({ description: 'GST percentage', example: 18.0, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  gstPercentage?: number;
  
  @ApiProperty({ description: 'CGST rate', example: 9, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  cgst?: number;

  @ApiProperty({ description: 'SGST rate', example: 9, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  sgst?: number;

  @ApiProperty({ description: 'IGST rate', example: 18, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  igst?: number;


  @ApiProperty({ description: 'TDS percentage', example: 10, required: false })
  @IsOptional()
  @IsNumber({ maxDecimalPlaces: 2 })
  @Min(0)
  @Max(100)
  tdsPercent?: number;


  @ApiProperty({ description: 'GST number', example: '27AAACG1234R1Z5', required: false })
  @IsOptional()
  @IsString()
  gstNumber?: string;

  @ApiProperty({ enum: TdsApplicabilityEnum, required: false })
  @IsOptional()
  @IsEnum(TdsApplicabilityEnum)
  tdsApplicability?: TdsApplicabilityEnum;

  @ApiProperty({ description: 'Invoice sender name', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderName?: string;

  @ApiProperty({ description: 'Invoice sender PAN', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderPan?: string;

  @ApiProperty({ description: 'Invoice sender CIN', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderCin?: string;

  @ApiProperty({ description: 'Venue address', required: false, type: CreateAddressDto })
  @IsOptional()
  @ValidateNested()
  @Type(() => CreateAddressDto)
  venueAddress?: CreateAddressDto;

  @ApiProperty({ description: 'Invoice sender address', required: false })
  @IsOptional()
  @IsString()
  invoiceSenderAddress?: string;

  @ApiProperty({ description: 'Helpline number', required: false })
  @IsOptional()
  @IsString()
  helplineNumber?: string;

  @ApiProperty({ description: 'Email sender name', required: false })
  @IsOptional()
  @IsString()
  emailSenderName?: string;

  @ApiProperty({ description: 'Venue name in emails', required: false })
  @IsOptional()
  @IsString()
  venueNameInEmails?: string;

  @ApiProperty({ description: 'Seeker can share experience', required: false })
  @IsOptional()
  @IsBoolean()
  seekerCanShareExperience?: boolean;

  @ApiProperty({ description: 'Total bed count', example: 100, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  totalBedCount?: number;

  @ApiProperty({ description: 'Launch date', required: false })
  @IsOptional()
  @IsDateString()
  launchDate?: Date;

  @ApiProperty({ description: 'Currency', example: 'INR', required: false })
  @IsOptional()
  @IsString()
  currency?: string;

  @ApiProperty({
    description: 'Program status',
    enum: ProgramStatusEnum,
    required: false,
  })
  @IsOptional()
  @IsEnum(ProgramStatusEnum)
  status?: ProgramStatusEnum;

  @ApiProperty({ description: 'Is active', required: false })
  @IsOptional()
  @IsBoolean()
  isActive?: boolean;

  @ApiProperty({ description: 'Program', required: false })
  @IsOptional()
  @IsString()
  program?: string;

  @ApiProperty({ description: 'Number of sessions', example: 0, required: false })
  @IsOptional()
  @IsNumber()
  @Min(0)
  noOfSession?: number;

  // @ApiProperty({ description: 'Is residence required', required: false })
  // @IsOptional()
  // @IsBoolean()
  // isResidenceRequired?: boolean;

  @ApiProperty({ description: 'Limited seats', required: false })
  @IsOptional()
  @IsBoolean()
  limitedSeats?: boolean;

  @ApiProperty({ description: 'Venue', required: false })
  @IsOptional()
  @IsString()
  venue?: string;

  @ApiProperty({ description: 'Is travel involved', required: false })
  @IsOptional()
  @IsBoolean()
  isTravelInvolved?: boolean;

  @ApiProperty({ description: 'Has checkin/checkout', required: false })
  @IsOptional()
  @IsBoolean()
  hasCheckinCheckout?: boolean;

  @ApiProperty({ description: 'Check-in at', required: false })
  @IsOptional()
  @IsDateString()
  checkinAt?: Date;

  @ApiProperty({ description: 'Check-out at', required: false })
  @IsOptional()
  @IsDateString()
  checkoutAt?: Date;

  @ApiProperty({ description: 'Banner image URL', required: false })
  @IsOptional()
  @IsString()
  bannerImageUrl?: string;

  // subProgramType
  @ApiProperty({ description: 'Sub program type', required: false, enum: SubProgramTypeEnum })
  @IsOptional()
  @IsEnum(SubProgramTypeEnum)
  subProgramType?: SubProgramTypeEnum;

  @ApiProperty({ description: 'Banner animation URL', required: false })
  @IsOptional()
  @IsString()
  bannerAnimationUrl?: string;

  @ApiProperty({ description: 'Check-in-ends-at', required: false })
  @IsOptional()
  @IsDateString()
  checkinEndsAt?: Date;

  @ApiProperty({ description: 'Check-out-ends-at', required: false })
  @IsOptional()
  @IsDateString()
  checkoutEndsAt?: Date;

  @ApiProperty({ description: 'Logo image URL', required: false })
  @IsOptional()
  @IsString()
  logoUrl?: string;

  @ApiProperty({
    description: 'Grouped programs (only for primary programs)',
    type: [GroupedProgramDto],
    required: false,
  })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => GroupedProgramDto)
  groupedPrograms?: GroupedProgramDto[];

  @ApiProperty({
    description: 'Program sessions',
    type: [CreateProgramSessionDto],
    required: false,
  })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => CreateProgramSessionDto)
  programSessions?: CreateProgramSessionDto[];

  @ApiProperty({ description: 'Elder minimum age', example: 60, required: false })
  @IsOptional()
  @IsNumber()
  elderMinAge?: number;

  @ApiProperty({ description: 'Child maximum age', example: 12, required: false })
  @IsOptional()
  @IsNumber()
  childMaxAge?: number;

  @ApiProperty({ description: 'Created by user ID', example: 1 })
  @IsNumber()
  createdBy: number;

  @ApiProperty({ description: 'Updated by user ID', example: 1 })
  @IsNumber()
  updatedBy: number;
}
