import {
  Injectable
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { IsNull, Repository } from "typeorm";
import { ProgramType } from "src/common/entities";
import { CommonDataService } from "src/common/services/commonData.service";
import { programTypeConstMessages } from "src/common/constants/strings-constants";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { AppLoggerService } from "src/common/services/logger.service";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";

@Injectable()
export class ProgramTypeRepository {
  constructor(
    @InjectRepository(ProgramType)
    private readonly programTypeRepo: Repository<ProgramType>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Finds a program type by its ID.
   * @param id - The ID of the program type.
   * @returns The program type entity if found, or null.
   */
  async findOneById(id: number): Promise<ProgramType | null> {
    try {
      return await this.commonDataService.findOneById(
        this.programTypeRepo,
        id,
        true,
        ["createdBy", "updatedBy", "programs", "programs.sessions", "programs.programQuestionMaps"]
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Finds a program type by its name.
   * @param name - The name of the program type.
   * @returns The program type entity if found, or null.
   */
  async findOneByName(name: string): Promise<ProgramType | null> {
    try {
      const where = { name, deletedAt: IsNull() };
      const results = await this.commonDataService.findByData(
        this.programTypeRepo,
        where
      );
      return results.length > 0 ? results[0] : null;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_FIND_BY_NAME_FAILED, error);
    }
  }

  /**
   * Saves a program type entity to the database.
   * @param programType - The program type entity to save.
   * @returns The saved program type entity.
   */
  async save(programType: ProgramType): Promise<ProgramType> {
    try {
      return await this.commonDataService.save(this.programTypeRepo, programType);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_SAVE_FAILED, error);
    }
  }

  /**
   * Counts the number of program types matching the given criteria.
   * @param whereClause - The criteria to match.
   * @returns The count of matching program types.
   */
  async count(whereClause: Record<string, any>): Promise<number> {
    try {
      const results = await this.commonDataService.findByData(
        this.programTypeRepo,
        whereClause
      );
      return results.length;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_COUNT_FAILED, error);
    }
  }

  /**
   * Soft deletes a program type by its ID and its associated programs.
   * @param id - The ID of the program type to delete.
   * @throws NotFoundException if the program type is not found.
   */
  async softDelete(id: number): Promise<void> {
    try {
      // Check if the program type exists
      const programType = await this.programTypeRepo.findOne({
        where: { id, deletedAt: IsNull() },
      });
      if (!programType) {
        this.logger.error(programTypeConstMessages.PROGRAM_TYPE_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      this.logger.log(programTypeConstMessages.SOFT_DELETING_PROGRAM_TYPE(id));
      // Soft delete the program type
      await this.programTypeRepo.softDelete(id);
      this.logger.log(programTypeConstMessages.SOFT_DELETED_PROGRAM_TYPE(id));

      // Note: Associated programs will be handled by cascade delete or 
      // should be handled separately in the service layer if needed
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_TYPE_DELETE_FAILED, error);
    }
  }

  /**
   * Retrieves all program types without pagination or filters.
   * This method is useful for scenarios where all program types are needed without any limit or filter.
   * @return An array of program types.
   */
  async findAllNoLimit(
    isActive?: boolean,
  ): Promise<ProgramType[]> {
    this.logger.log(
      programTypeConstMessages.FIND_ALL_PROGRAM_TYPES_REQUEST_RECEIVED
    );
    try {
      const whereClause: any = { deletedAt: IsNull() };

      if (isActive !== undefined) {
        whereClause.isActive = isActive;
      }

      this.logger.log(
        programTypeConstMessages.RETRIEVING_PROGRAM_TYPES_WHERE_CLAUSE(whereClause)
      );
      const data = await this.programTypeRepo.find({
        select: ["id", "name", "description", "modeOfOperation", "onlineType", "isActive"],
        where: whereClause,
        order: { id: "ASC" },
        relations: ["programs", "programs.sessions", "programs.programQuestionMaps"],
      });

      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMON_DATA_SERVICE_GET_FAILED, error);
    }
  }
}