import { forwardRef, Inject, Injectable } from '@nestjs/common';
import { CreateProgramRegistrationDto } from './dto/create-program-registration.dto';
import { UpdateProgramRegistrationDto } from './dto/update-program-registration.dto';
import { ProgramRegistrationRepository } from './program-registration.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { programRegistrationMessages, CLEARANCE_REASONS, zeptoEmailCreadentials } from 'src/common/constants/strings-constants';
import { CreateProgramRegistrationSwapDto, MakeSwapRequestDto, UpdateSwapRequestDto } from './dto/program-registration-swap.dto';
import { ProgramRegistrationSwapRepository } from './repositories/program-registration-swap.repository';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { ProgramRepository } from 'src/program/program.repository';
import { UserRepository } from 'src/user/user.repository';
import {
  Program,
  ProgramRegistration,
  ProgramRegistrationSwap,
  RegistrationGroupMap,
  RegistrationPairMap,
  RoomAllocation,
} from 'src/common/entities';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ILike, DataSource, EntityManager, IsNull } from 'typeorm';
import { RegistrationRepository } from 'src/registration/registration.repository';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { SwapType } from 'src/common/enum/swap-type-enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { InjectDataSource } from '@nestjs/typeorm';
import { ApprovalTrackTypeEnum } from 'src/common/enum/approval-track-type.enum';
import { RegistrationApprovalRepository } from 'src/registration-approval/registration-approval.repository';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { CommunicationService } from 'src/communication/communication.service';
import { SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import {
  deductDaysFromDate,
  formatDateIST,
  generatePaymentLink,
  getWeekName,
} from 'src/common/utils/common.util';
import axios from 'axios';
import { RegistrationApprovalService } from 'src/registration-approval/registration-approval.service';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { SwapRequirementEnum } from 'src/common/enum/swap-requirement.enum';
import { AllocationClearingService } from 'src/common/services/allocation-clearing.service';

@Injectable()
export class ProgramRegistrationService {
  constructor(
    private readonly repo: ProgramRegistrationRepository,
    private readonly logger: AppLoggerService,
    private readonly swapRepo: ProgramRegistrationSwapRepository,
    private readonly programRepo: ProgramRepository,
    private readonly userRepo: UserRepository,
    @Inject(forwardRef(() => RegistrationRepository))
    private readonly registrationRepo: RegistrationRepository,
    private readonly approvalRepo: RegistrationApprovalRepository,
    @InjectDataSource()
    private readonly dataSource: DataSource,
    private readonly communicationService: CommunicationService,
    private readonly registrationApprovalService: RegistrationApprovalService,
    private readonly allocationClearingService: AllocationClearingService,
  ) {}

  async create(dto: CreateProgramRegistrationDto) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(programRegistrationMessages.CREATE_REQUEST_RECEIVED, dto);
      try {
        return await this.repo.createEntity(dto, manager);
      } catch (error) {
        this.logger.error('Error creating program registration', '', { error });
        handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
      }
    });
  }

  async findAll(limit: number, offset: number) {
    this.logger.log(programRegistrationMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset);
    } catch (error) {
      this.logger.error('Error finding all program registrations', '', { error });
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(programRegistrationMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      const registration = await this.repo.findOne(id);
      
      return registration;
    } catch (error) {
      this.logger.error('Error finding program registration by ID', '', { id, error });
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Finds program registrations by program ID and optional sub-program ID.
   * @param programId - The ID of the program to filter registrations by.
   * @param limit - The maximum number of registrations to return.
   * @param offset - The offset for pagination.
   * @param subProgramId - Optional sub-program ID to filter registrations further.
   * @returns A list of program registrations matching the criteria.
   * @throws {Error} If the retrieval fails.
   */
  async findBy(programId: number, limit: number, offset: number, subProgramId?: number, search?: string) {
  try {
    let where = {};
    if (programId) {
      where = {
        program: {
          id: programId,
        } as Program,
      };
    }
    if (subProgramId) {
      where = { ...where,
        allocatedProgram: {
          id: subProgramId,
        } as Program,
      };
    }
    if (search) {
      where = { 
        ...where,
        fullName: ILike(`%${search}%`)
      };
    }
    this.logger.log(programRegistrationMessages.FIND_BY_REQUEST_RECEIVED);
    return await this.repo.findBy(where, ['swapsRequests'], limit, offset);
  } catch (error) {
    handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Updates a program registration with the given ID.
   * @param id - The ID of the program registration to update.
   * @param dto - The data to update the program registration with.
   * @returns The updated program registration.
   * @throws {Error} If the update fails.
   */
  async update(id: number, dto: UpdateProgramRegistrationDto) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(programRegistrationMessages.UPDATE_REQUEST_RECEIVED);
      try {
        return await this.repo.updateEntity(id, dto, manager);
      } catch (error) {
        handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
      }
    });
  }

  async remove(id: number) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(programRegistrationMessages.DELETE_REQUEST_RECEIVED);
      try {
        return await this.repo.remove(id, manager);
      } catch (error) {
        handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_DELETE_FAILED, error);
      }
    });
  }

  /**
   * Requests a swap for a program registration.
   * @param data - The swap request data.
   * @param userId - The ID of the user making the request.
   * @returns The created swap request.
   */
  async requestSwap(data: CreateProgramRegistrationSwapDto, userId: number) {
    try {
      const registrationRecord = await this.repo.findOne(data.programRegistrationId);
      if (!registrationRecord || !registrationRecord.allocatedProgram) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            null,
            null,
            data.programRegistrationId.toString()
          )
        );
      }

      const existingSwapRequest = await this.swapRepo.findOneByWhere({ programRegistrationId: data.programRegistrationId, status: SwapRequestStatus.ACTIVE });
      if (existingSwapRequest && existingSwapRequest.status === SwapRequestStatus.ACTIVE) {
        this.logger.warn(programRegistrationMessages.SWAP_REQUEST_ALREADY_EXISTS, {
          programRegistrationId: data.programRegistrationId,
        });
        handleKnownErrors(
          ERROR_CODES.SWAP_REQUEST_ALREADY_EXISTS,
          new InifniBadRequestException(
            ERROR_CODES.SWAP_REQUEST_ALREADY_EXISTS,
            null,
            null,
            `${data.programRegistrationId}`
          )
        );
      }

      const targetProgramIds = data.targetPrograms?.map((program) => program.id) ?? [];
      await this.validateTargetProgramStartDates(targetProgramIds);

      return await this.swapRepo.createAndSave({
        programRegistrationId: data.programRegistrationId,
        type: data.type,
        status: SwapRequestStatus.ACTIVE,
        comment : data.comment,
        createdBy: userId,
        updatedBy: userId,
        createdAt: new Date(),
        updatedAt: new Date(),
        currentProgramId : registrationRecord.allocatedProgram.id,
        requestedPrograms: data.targetPrograms.map((program) => {
          return { id: program.id } as Program;
        }),
        swapRequirement: data.swapRequirement || 'SWAP_REQUEST',
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Updates the status of a swap request.
   * @param id - The ID of the swap request.
   * @param userId - The ID of the user updating the request.
   * @param status - The new status of the swap request.
   * @param comment - Optional comment for the swap request.
   * @returns The updated swap request.
   */
  async updateSwapRequestById(
    id: number,
    userId: number,
    data : MakeSwapRequestDto,
    swappingType: string = 'swap', 
  ) {
    const { status, comment } = data;
    this.logger.log(programRegistrationMessages.UPDATE_SWAP_REQUEST, { userId });
    try {
      const swapRequest = await this.swapRepo.findOneById(id);
      // Get the user who is accepting the swap
      const user = await this.userRepo.getUserByField('id', userId);
      
      const movingSeeker = await this.repo.findOne(swapRequest.programRegistrationId);
      // Check if the moving seeker exists and has an allocated program
      if (!movingSeeker || !movingSeeker.allocatedProgram) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            null,
            null,
            swapRequest.programRegistrationId.toString()
          )
        );
      }
      const currentSubProgram = movingSeeker.allocatedProgram;
      let movingToSubProgram;
      let outGoingSeekerSwapRequest;

      if (swapRequest && swapRequest.status === status) {
        this.logger.warn(programRegistrationMessages.SWAP_REQUEST_ALREADY_UPDATED, { id, status });
        handleKnownErrors(
          ERROR_CODES.SWAP_REQUEST_ALREADY_UPDATED,
          new InifniNotFoundException(
            ERROR_CODES.SWAP_REQUEST_ALREADY_UPDATED,
            null,
            null,
            id.toString()
          )
        );
      }
      if (status === SwapRequestStatus.CLOSED) {
        return await this.swapRepo.updateAndSave(id, { status, comment, updatedBy: userId });
      } else if (status === SwapRequestStatus.ACCEPTED) {
        if (swappingType === 'swap') {
          // Check if the outgoing seeker exists and has an allocated program
          const outGoingSeeker = await this.repo.findOne(data.outgoingSeekerRegistrationId);
          if (!outGoingSeeker || !outGoingSeeker.allocatedProgram) {
            handleKnownErrors(
              ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
              new InifniNotFoundException(ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND, null, null),
            );
          }
          movingToSubProgram = outGoingSeeker.allocatedProgram;
          // Outgoing seeker should be from the Sub program where moving seeker is moving to
          if (outGoingSeeker.allocatedProgram.id === movingSeeker.allocatedProgram.id) {
            this.logger.warn(programRegistrationMessages.INVALID_REQUEST, {
              id,
              movingToSubProgramId: data.movingToSubProgramId,
              outGoingSeekerId: outGoingSeeker.id,
            });
            handleKnownErrors(
              ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
              new InifniBadRequestException(
                ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
                null,
                null,
                `${data.outgoingSeekerRegistrationId}`,
              ),
            );
          }

          await this.allocationClearingService.clearRegistrationAllocations(movingSeeker.id, userId, CLEARANCE_REASONS.SWAP_REQUEST_ACCEPTED_INCOMING(movingSeeker.id));
          await this.allocationClearingService.clearRegistrationAllocations(outGoingSeeker.id, userId, CLEARANCE_REASONS.SWAP_REQUEST_ACCEPTED_OUTGOING(outGoingSeeker.id));

          outGoingSeekerSwapRequest = await this.swapRepo.findOneByWhere({
            programRegistrationId: outGoingSeeker.id,
            status: SwapRequestStatus.ACTIVE,
          });
          if (!outGoingSeekerSwapRequest) {
            outGoingSeekerSwapRequest = await this.requestSwap(
              {
                programRegistrationId: outGoingSeeker.id,
                targetPrograms: [{ id: movingSeeker.allocatedProgram.id }],
                type: SwapType.ReturnSwap,
                comment: '',
              },
              userId,
            );
          }
          // Update the outgoing seeker's registration with the moving seeker's current program (currentSubProgram)
          await this.repo.update(outGoingSeeker.id, {
            allocatedProgram: { id: currentSubProgram.id } as Program,
            updatedBy: user,
            auditRefId: outGoingSeeker.id,
            parentRefId: outGoingSeeker.id,
          });
          // Update approval status and approval date for outgoing seeker
          const updatedOutgoingApproval = await this.registrationRepo.updateApprovalStatus(
            outGoingSeeker.id,
            ApprovalStatusEnum.APPROVED,
            userId,
          );
          if (updatedOutgoingApproval) {
            await this.approvalRepo.createApprovalTrack(
              updatedOutgoingApproval,
              userId,
              ApprovalTrackTypeEnum.SWAP,
              id,
              currentSubProgram.id
            );
          }
          // // Increase the filled seats of the current program (currentSubProgram)
          // await this.programRepo.updateById(currentSubProgram.id, {
          //   filledSeats: currentSubProgram.filledSeats + 1,
          //   updatedBy: userId
          // });
          // Update the outgoing seeker's swap request status
          if (outGoingSeekerSwapRequest) {
            await this.swapRepo.updateAndSave(outGoingSeekerSwapRequest.id, {
              status: SwapRequestStatus.ACCEPTED,
              updatedBy: userId,
              allocatedProgramId: currentSubProgram.id,
            });
          }
        } else if (swappingType === 'move') {
          movingToSubProgram = await this.programRepo.findOneById(data.movingToSubProgramId);
        }
        if (!movingToSubProgram) {
          handleKnownErrors(
            ERROR_CODES.PROGRAM_NOTFOUND,
            new InifniNotFoundException(
              ERROR_CODES.PROGRAM_NOTFOUND,
              null,
              null,
              data.movingToSubProgramId.toString(),
            ),
          );
        }

        if (swappingType === 'move') {
          await this.allocationClearingService.clearRegistrationAllocations(movingSeeker.id, userId, CLEARANCE_REASONS.SWAP_REQUEST_ACCEPTED_MOVE(movingSeeker.id));
        }

        // Update the moving seeker's registration with the new program (movingToSubProgram)
        await this.repo.update(movingSeeker.id, {
          allocatedProgram: { id: movingToSubProgram.id } as Program,
          updatedBy: user,
          auditRefId: movingSeeker.id,
          parentRefId: movingSeeker.id,
        });

        const updatedApproval = await this.registrationRepo.updateApprovalStatus(
          swapRequest.programRegistrationId,
          ApprovalStatusEnum.APPROVED,
          userId,
        );
        if (updatedApproval) {
          await this.approvalRepo.createApprovalTrack(
            updatedApproval,
            userId,
            ApprovalTrackTypeEnum.SWAP,
            id,
            movingToSubProgram.id
          );
        }

        // // Increase the filled seats of the target program (movingToSubProgram)
        // await this.programRepo.updateById(movingToSubProgram.id, {
        //   filledSeats: movingToSubProgram.filledSeats + 1,
        //   updatedBy: userId
        // });

        // // Decrease the filled seats of the outgoing seeker's current program (outGoingSeeker.allocatedProgram)
        // await this.programRepo.updateById(outGoingSeeker.allocatedProgram.id, {
        //   filledSeats: outGoingSeeker.allocatedProgram.filledSeats - 1,
        //   updatedBy: userId
        // });

        // sending communication
        if (swappingType === 'move' || swappingType === 'swap') {
          try {
            await this.sendSwapRequestAcceptCommunication(data.movingSeekerRegistrationId, userId);

            // Reset travel details on swap acceptance
            await this.registrationRepo.resetTravelDetails(data.movingSeekerRegistrationId, userId);
          } catch (error) {
            this.logger.log('Error sending swap request acceptance communication', { error });
          }
        }
        if (swappingType === 'swap') {
          try {
            await this.sendSwapRequestAcceptCommunication(
              data.outgoingSeekerRegistrationId,
              userId,
            );

            // Reset travel details on swap acceptance
            await this.registrationRepo.resetTravelDetails(data.outgoingSeekerRegistrationId, userId);
          } catch (error) {
            this.logger.log('Error sending swap request acceptance communication', { error });
          }
        }

        this.logger.log(programRegistrationMessages.SWAP_REQUEST_ACCEPTED, {
          id,
          userId,
          status,
          comment,
        });
        // Update the swap request status of Moving Seeker
        return await this.swapRepo.updateAndSave(id, {
          status: SwapRequestStatus.ACCEPTED,
          updatedBy: userId,
          allocatedProgramId: movingToSubProgram.id,
        });
      } else if (status === SwapRequestStatus.ON_HOLD || status === SwapRequestStatus.REJECTED) {
        const registrationRecord = await this.repo.findOne(swapRequest.programRegistrationId);
        if (!registrationRecord || !registrationRecord.allocatedProgram) {
          handleKnownErrors(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            new InifniNotFoundException(
              ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
              null,
              null,
              swapRequest.programRegistrationId.toString(),
            ),
          );
        }

        await this.allocationClearingService.clearRegistrationAllocations(registrationRecord.id, userId, status === SwapRequestStatus.ON_HOLD ? CLEARANCE_REASONS.SWAP_REQUEST_ON_HOLD(registrationRecord.id) : CLEARANCE_REASONS.SWAP_REQUEST_REJECTED(registrationRecord.id));

        if (status === SwapRequestStatus.ON_HOLD) {
          // check for the existing swap request type it should be wants_swap
          const existingSwapRequestSwapType = swapRequest.type;
          if (existingSwapRequestSwapType !== SwapType.WantsSwap) {
            this.logger.warn(programRegistrationMessages.INVALID_SWAP_REQUEST_STATUS, { id, status });
            handleKnownErrors(
              ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
              new InifniBadRequestException(
                ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
                null,
                null,
                'Swap demand cannot be raised if the swap type is not Wants Swap',
              ),
            );
          }
        }

        const updateData: Partial<ProgramRegistration> = {
          allocatedProgram: null,
          allocatedSession: null,
          updatedBy: user,
          auditRefId: swapRequest.programRegistrationId,
          parentRefId: swapRequest.programRegistrationId,
        };

        if (status === SwapRequestStatus.REJECTED) {
          // If the swap is rejected, update the registration status to REJECTED
          updateData.registrationStatus = RegistrationStatusEnum.REJECTED;
        }
        await this.repo.update(swapRequest.programRegistrationId, updateData);

        // Increase the filled seats of the moving seeker's current program (movingSeeker.allocatedProgram)
        await this.programRepo.updateById(movingSeeker.allocatedProgram.id, {
          filledSeats: movingSeeker.allocatedProgram.filledSeats - 1,
          updatedBy: userId,
        });

        const updatedApproval = await this.registrationRepo.updateApprovalStatus(
          swapRequest.programRegistrationId,
          status === SwapRequestStatus.ON_HOLD
            ? ApprovalStatusEnum.ON_HOLD
            : ApprovalStatusEnum.REJECTED,
          userId,
        );
        if (updatedApproval) {
          await this.approvalRepo.createApprovalTrack(
            updatedApproval,
            userId,
            ApprovalTrackTypeEnum.SWAP,
            id,
            movingSeeker.allocatedProgram.id,
          );
        }

        const updatedSwapRequest = await this.swapRepo.updateAndSave(id, {
          status: status,
          updatedBy: userId,
          swapRequirement: status === SwapRequestStatus.ON_HOLD ? SwapRequirementEnum.SWAP_DEMAND : swapRequest.swapRequirement,
        });

        if (status === SwapRequestStatus.ON_HOLD && swapRequest.swapRequirement !== SwapRequirementEnum.SWAP_DEMAND) {
          await this.sendSwapRequestDemandCommunication(registrationRecord);
          // clear travel details
          await this.registrationRepo.resetTravelDetails(swapRequest.programRegistrationId, userId);
        }

        return updatedSwapRequest;
      } else {
        this.logger.warn(programRegistrationMessages.INVALID_SWAP_REQUEST_STATUS, { id, status });
        handleKnownErrors(
          ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
          new InifniBadRequestException(
            ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
            null,
            null,
            id.toString(),
          ),
        );
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Finds swap requests by program ID and optional sub-program ID.
   * @param programId - The ID of the program to filter swap requests by.
   * @param subProgramId - Optional sub-program ID to filter swap requests further.
   * @param limit - The maximum number of swap requests to return.
   * @param offset - The offset for pagination.
   * @param filters - Optional filters to apply to the swap requests.
   * @returns A list of swap requests matching the criteria.
   */
  async findSwapRequestsBy(
    programId: number,
    subProgramId?: number,
    limit?: number,
    offset?: number,
    filters?: { [key: string]: any },
  ): Promise<ProgramRegistration[]> {
    try {
      const where = {
         id: programId,
         swapsRequests : {
          status: SwapRequestStatus.ACTIVE,
         }
      } 
      if (subProgramId) {
        where['allocatedProgram'] = {
          id: subProgramId,
        } as Program;
      }
      if (filters && filters.gender) {
        where['gender'] = filters.gender;
      }
      if (filters && filters.requestedProgramId) {
        where.swapsRequests['requestedPrograms'] = {
          id: filters.requestedProgramId,
        } as Program;
      }
        const relations = [
        'program',
        'program.type',
        'programSession',
        'invoiceDetails',
        'paymentDetails', 
        'travelInfo',
        'travelPlans',
        'user',
        'approvals',
        'allocatedProgram',
        'allocatedSession',
        'swapsRequests',
        'swapsRequests.requestedPrograms',
        'ratings',
        'rmContactUser',
        'preferences',
        'preferences.preferredProgram',
        'preferences.preferredSession',
      ];
      const response = await this.repo.getAll(where, relations);
      return response;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  async findSwapDemandBy(
    programId: number,
    subProgramId?: number,
    limit?: number,
    offset?: number,
    filters?: { [key: string]: any },
  ): Promise<ProgramRegistration[]> {
    try {
      const where = {
         id: programId,
         swapsRequests : {
          status: SwapRequestStatus.ON_HOLD,
          swapRequirement: SwapRequirementEnum.SWAP_DEMAND,
         }
      } 
      if (subProgramId) {
        where['allocatedProgram'] = {
          id: subProgramId,
        } as Program;
      }
      if (filters && filters.gender) {
        where['gender'] = filters.gender;
      }
      if (filters && filters.requestedProgramId) {
        where.swapsRequests['requestedPrograms'] = {
          id: filters.requestedProgramId,
        } as Program;
      }
        const relations = [
        'program',
        'program.type',
        'programSession',
        'invoiceDetails',
        'paymentDetails', 
        'travelInfo',
        'travelPlans',
        'user',
        'approvals',
        'allocatedProgram',
        'allocatedSession',
        'swapsRequests',
        'swapsRequests.requestedPrograms',
        'ratings',
        'rmContactUser',
        'preferences',
        'preferences.preferredProgram',
        'preferences.preferredSession',
      ];
      const response = await this.repo.getAll(where, relations);
      return response;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Retrieves key performance indicators (KPI) for swap requests in a program.
   * @param data - The data containing program ID, filters, limit, and offset.
   * @returns An object containing swap requests
   */
  async getSwapRequestKPI(data: {
    programId: number;
    filters?: { [key: string]: any };
    limit?: number;
    offset?: number;
  }) {
    const { programId, filters, limit = 10, offset = 0 } = data;
    try {
      const swapRequests = await this.findSwapRequestsBy(
        programId,
        undefined,
        limit,
        offset,
        filters,
      );
      const maleAndFemale = await this.swapRepo.getCountsByGroupOf('gender', programId);
      const maleAndFemaleOrg = await this.swapRepo.getOrgCountsByGroupOf('gender', programId);
      const subProgramsResults = await this.swapRepo.getCountsByGroupOf('requestedPrograms', programId);
      const subProgramsOrgResults = await this.swapRepo.getOrgCountsByGroupOf('requestedPrograms', programId);
      const subPrograms = await this.registrationRepo.getSubPrograms(programId);

      const subProgramKPI = subPrograms.map((subProgram) => {
        return {
          kpiName: subProgram.name,
          count: Number(subProgramsResults.find((result) => Number(result.field) === subProgram.id)?.count || 0),
          organisationUserCount: Number(subProgramsOrgResults.find((result) => Number(result.field) === subProgram.id)?.count || 0),
          filters: {
            swapPreferredProgramId: subProgram.id,
          },
        };
      });
      const kPIs = [
        ...subProgramKPI,
        {
          kpiName: 'male',
          count: Number(maleAndFemale.find((item) => typeof item.field === 'string' && item.field.toLowerCase() === 'male')?.count || 0),
          organisationUserCount: Number(maleAndFemaleOrg.find((item) => typeof item.field === 'string' && item.field.toLowerCase() === 'male')?.count || 0),
          filters: {
            gender: 'Male',
          },
        },
        {
          kpiName: 'female',
          count: Number(maleAndFemale.find((item) => typeof item.field === 'string' && item.field.toLowerCase() === 'female')?.count || 0),
          organisationUserCount:Number(maleAndFemaleOrg.find((item) => typeof item.field === 'string' && item.field.toLowerCase() === 'female')?.count || 0),
          filters: {
            gender: 'Female',
          },
        },
      ];
      return {
        kpis: kPIs,
        data: swapRequests,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

   /**
   * Retrieves key performance indicators (KPI) for swap demands in a program.
   * @param data - The data containing program ID, filters, limit, and offset.
   * @returns An object containing swap demands with KPIs
   */
  async getSwapDemandKPI(data: {
    programId: number;
    filters?: { [key: string]: any };
    limit?: number;
    offset?: number;
  }) {
    const { programId, filters, limit = 10, offset = 0 } = data;
    try {
      const swapDemands = await this.findSwapDemandBy(
        programId,
        undefined,
        limit,
        offset,
        filters,
      );
      
      // Use swap demand specific repository methods
      const maleAndFemale = await this.swapRepo.getSwapDemandCountsByGroupOf('gender', programId);
      const maleAndFemaleOrg = await this.swapRepo.getSwapDemandOrgCountsByGroupOf('gender', programId);
      const subProgramsResults = await this.swapRepo.getSwapDemandCountsByGroupOf('requestedPrograms', programId);
      const subProgramsOrgResults = await this.swapRepo.getSwapDemandOrgCountsByGroupOf('requestedPrograms', programId);
      const subPrograms = await this.registrationRepo.getSubPrograms(programId);

      const subProgramKPI = subPrograms.map((subProgram) => {
        return {
          kpiName: subProgram.name,
          count: Number(subProgramsResults.find((result) => Number(result.field) === subProgram.id)?.count || 0),
          organisationUserCount: Number(subProgramsOrgResults.find((result) => Number(result.field) === subProgram.id)?.count || 0),
          filters: {
            swapDemandPreferredProgramId: subProgram.id
          },
        };
      });
      
      const kPIs = [
        ...subProgramKPI,
        {
          kpiName: 'male',
          count: Number(maleAndFemale.find((item) => typeof item.field === 'string' && item.field.toLowerCase() === 'male')?.count || 0),
          organisationUserCount: Number(maleAndFemaleOrg.find((item) => typeof item.field === 'string' && item.field.toLowerCase() === 'male')?.count || 0),
          filters: {
            gender: 'Male'
            },
        },
        {
          kpiName: 'female',
          count: Number(maleAndFemale.find((item) => typeof item.field === 'string' && item.field.toLowerCase() === 'female')?.count || 0),
          organisationUserCount: Number(maleAndFemaleOrg.find((item) => typeof item.field === 'string' && item.field.toLowerCase() === 'female')?.count || 0),
          filters: {
            gender: 'Female'
          },
        },
      ];
      
      return {
        kpis: kPIs,
        data: swapDemands,
      };
    } catch (error) {
      this.logger.error('Error getting swap demand KPIs', '', { error });
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }
  /**
   * Retrieves pending registration KPIs for a given program.
   * @param data - Object containing swapDemands and allKpis arrays.
   * @returns An object with pending registration KPIs.
   */
  getRegistrationPendingKPI(data: { formattedKPIs: any[] }): { kpis: any[] } | undefined {
    try {
      const { formattedKPIs } = data;
      // Defensive: formattedKPIs is an array, use the first element if present
      const kpiObj = Array.isArray(formattedKPIs) ? formattedKPIs[0] : formattedKPIs;
      // Use swap demand KPIs from swapDemands array
      const swapDemandTotals = kpiObj?.swapDemands || [];
      // For rejected counts, still use allocated.totals.rejectCount.preferredPrograms
      const allocated = kpiObj?.allocated || {};
      const rejectTotals = allocated?.totals?.rejectCount?.preferredPrograms || [];
      // Map by programId for quick lookup
      const rejectMap = new Map();
      for (const rej of rejectTotals) {
        rejectMap.set(rej.programId, rej.count || 0);
      }

      // Compose pending KPIs by summing swap demand and reject counts per subProgram
      const pendingKPIs = swapDemandTotals
        .filter((swap) => swap.kpiName.toLowerCase() !== 'male' && swap.kpiName.toLowerCase() !== 'female')
        .map((swap) => {
          const programId = swap.filters?.swapDemandPreferredProgramId;
          const pendingCount = (swap.count || 0) + (rejectMap.get(programId) || 0);
          return {
        kpiName: swap.kpiName,
        count: pendingCount,
        filters: { pendingProgramId: programId },
          };
        });

      // Calculate male/female counts from swapDemandTotals and rejectTotals
      // Try to find gender-based entries in kpiObj.swapDemands or kpiObj.allocated.totals.rejectCount
      let maleCount = 0;
      let femaleCount = 0;
      // Try to get gender counts from kpiObj if available
      if (kpiObj?.swapDemands) {
        const maleKPI = kpiObj.swapDemands.find((k) => k.kpiName && k.kpiName.toLowerCase() === 'male');
        const femaleKPI = kpiObj.swapDemands.find((k) => k.kpiName && k.kpiName.toLowerCase() === 'female');
        if (maleKPI) maleCount += maleKPI.count || 0;
        if (femaleKPI) femaleCount += femaleKPI.count || 0;
      }
      // Add reject counts if available (if gender-based reject counts exist)
      if (allocated?.totals?.rejectCount) {
        if (typeof allocated.totals.rejectCount.maleCount === 'number') {
          maleCount += allocated.totals.rejectCount.maleCount;
        }
        if (typeof allocated.totals.rejectCount.femaleCount === 'number') {
          femaleCount += allocated.totals.rejectCount.femaleCount;
        }
      }

      // Add male/female KPIs to the result
      pendingKPIs.push({
        kpiName: 'male',
        count: maleCount,
        filters: {gender: 'male'},
      });
      pendingKPIs.push({
        kpiName: 'female',
        count: femaleCount,
        filters: {gender: 'female'},
      });

      return {
        kpis: pendingKPIs,
      };
    } catch (error) {
      this.logger.error('Error getting registration pending KPIs', '', { error });
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  async getSwapRequestById(id: number) {
    try {
      const swapRequest = await this.swapRepo.findOneById(id);
      if (!swapRequest) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
            null,
            null,
            id.toString(),
          ),
        );
      }
      return swapRequest;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   *
   * @param where - Partial object to search for a swap request.
   * This can include fields like `programRegistrationId`, `status`, etc.
   * @returns
   */
  async getSwapRequestByField(where: Partial<Omit<ProgramRegistrationSwap, 'requestedPrograms'>>) {
    try {
      const swapRequest = await this.swapRepo.findOneByWhere(where);
      return swapRequest;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }
  async sendSwapRequestAcceptCommunication(registrationId: number, userId: number) {
    try {
      const registrationDetails = await this.registrationRepo.findRegistrationById(registrationId);
      if (!registrationDetails) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            null,
            null,
            registrationId.toString(),
          ),
        );
      }
      let templateKey: string | undefined;

      let approvedMergeInfo;
      let attachments;
      const ccEmails: { emailAddress: string; name: string }[] = [];
      if (registrationDetails.rmContactUser && registrationDetails.rmContactUser.email) {
        ccEmails.push({
          emailAddress: registrationDetails.rmContactUser.email,
          name: registrationDetails.rmContactUser.orgUsrName || registrationDetails.rmContactUser.fullName,
        });
      }

      if (
        (registrationDetails?.paymentDetails?.length > 0 &&
        ((registrationDetails?.paymentDetails[0]?.paymentStatus ===
          PaymentStatusEnum.ONLINE_COMPLETED) ||
          (registrationDetails?.paymentDetails[0]?.paymentStatus ===
              PaymentStatusEnum.OFFLINE_COMPLETED))) ||
        registrationDetails.isFreeSeat === true
      ) {
        templateKey = process.env.ZEPTO_BLESSED_NO_PAYMENT_HDB_EMAIL_TEMPLATE_ID;
        approvedMergeInfo = {
          venue_name: registrationDetails?.allocatedProgram?.venueNameInEmails ?? '',
          s_day: registrationDetails?.allocatedProgram?.startsAt
            ? getWeekName(registrationDetails?.allocatedProgram?.startsAt ?? new Date())
            : '-',
          e_day: getWeekName(registrationDetails?.allocatedProgram?.endsAt ?? new Date()) || '',
          reg_name: registrationDetails.fullName ?? 'Infinitheist',
          hdb_or_msd: registrationDetails?.allocatedProgram?.name ?? 'HDB',
          hdb_dates: `${formatDateIST(registrationDetails?.allocatedProgram?.startsAt?.toISOString() ?? '')} to ${formatDateIST(registrationDetails?.allocatedProgram?.endsAt?.toISOString() ?? '')}`,
          hdb_no: registrationDetails?.allocatedProgram?.name ?? 'HDB',
        };
      } else {
        templateKey = process.env.ZEPTO_BLESSED_HDB_EMAIL_TEMPLATE_ID;
        approvedMergeInfo = {
          venue_name: registrationDetails?.allocatedProgram?.venueNameInEmails ?? '',
          s_day: registrationDetails?.allocatedProgram?.startsAt
            ? getWeekName(registrationDetails?.allocatedProgram?.startsAt ?? new Date())
            : '-',
          hdb_price: registrationDetails?.allocatedProgram?.basePrice || '',
          payment_last_date: registrationDetails?.allocatedProgram?.startsAt
            ? formatDateIST(
                deductDaysFromDate(
                  registrationDetails?.allocatedProgram?.startsAt,
                  10,
                ).toISOString() || '',
              )
            : '-',
          cash_details:
            (registrationDetails?.allocatedProgram?.basePrice ?? 0) > 150
              ? ''
              : '4. Cash: If you are paying by cash, after paying the cash, please click on the following link and fill in the details. Please note that this simple update by you will ensure that there is no omission or reconciliation issue owing to human error.',
          e_day: getWeekName(registrationDetails?.allocatedProgram?.endsAt ?? new Date()) || '',
          reg_name: registrationDetails?.fullName ?? 'Infinitheist',
          hdb_or_msd: registrationDetails?.allocatedProgram?.name ?? '',
          hdb_dates: `${formatDateIST(registrationDetails?.allocatedProgram?.startsAt.toISOString() ?? '')} to ${formatDateIST(registrationDetails?.allocatedProgram?.endsAt.toISOString() ?? '')}`,
          hdb_no: registrationDetails?.allocatedProgram?.name ?? '',
          payment_online_link: generatePaymentLink(
            registrationDetails?.program?.id,
            registrationDetails?.userId,
          ),
          payment_back_transfer_link: generatePaymentLink(
            registrationDetails?.program?.id,
            registrationDetails?.userId,
          ),
          payment_cheque_link: generatePaymentLink(
            registrationDetails?.program?.id,
            registrationDetails?.userId,
          ),
        };
        const pdfBuffer = await this.registrationApprovalService.generateProFormaInvoicePDF(
          registrationDetails,
          registrationDetails.proFormaInvoiceSeqNumber,
        );
        await this.registrationRepo.updateRegistration(
          registrationDetails.id,
          {
            proFormaInvoicePdfUrl: pdfBuffer.url,
          },
          userId,
        );
        this.logger.log('Pro-forma invoice PDF URL', { url: pdfBuffer.url });
        const response = await axios.get(pdfBuffer.url, {
          responseType: 'arraybuffer',
          headers: {
            referer: process.env.BACKEND_URL,
          },
        });
        this.logger.log('PDF buffer retrieved successfully', { buffer: response.data });
        const fileData = Buffer.from(response.data, 'binary').toString('base64');

        attachments = [
          {
            name: 'pro-forma-invoice.pdf',
            content: fileData,
            mime_type: 'application/pdf',
          },
        ];
      }
      const bccEmails: { emailAddress: string; name: string }[] = [];
      if (process.env.HDB_BCC_EMAIL) {
        bccEmails.push({
          emailAddress: process.env.HDB_BCC_EMAIL,
          name: process.env.HDB_BCC_EMAIL_NAME || 'HDB Team',
        });
      }

      const emailData: SendSingleEmailDto = {
        templateKey,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name:
            registrationDetails.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        to: {
          emailAddress: registrationDetails.emailAddress,
          name: registrationDetails.fullName,
        },
        cc: ccEmails,
        bcc: bccEmails,
        mergeInfo: approvedMergeInfo,
        attachments: attachments,
        subject: '',
        trackinfo: {
          registrationId: registrationDetails.id,
        },
      };

      await this.communicationService.sendSingleEmail(emailData);
    } catch (error) {
      this.logger.log('Error sending swap request acceptance communication', { error });
    }
  }

  async validateTargetProgramStartDates(programIds: number[]): Promise<void> {
    const uniqueIds = Array.from(
      new Set(
        (programIds || []).filter(
          (id) => typeof id === 'number' && Number.isFinite(id),
        ),
      ),
    );

    if (uniqueIds.length === 0) {
      this.logger.debug('No target program IDs provided for start date validation');
      return;
    }

    try {
      const programsToValidate = await this.programRepo.findByIds(uniqueIds);

      if (!programsToValidate || programsToValidate.length === 0) {
        this.logger.warn(`No programs found for IDs: ${uniqueIds.join(', ')}`);
        return;
      }

      const currentTimeUTC = Date.now();

      const startedPrograms = programsToValidate.filter((program) => {
        if (!program?.startsAt) {
          return false;
        }

        return new Date(program.startsAt).getTime() <= currentTimeUTC;
      });

      if (startedPrograms.length > 0) {
        const programNames = startedPrograms
          .map((program) => program.name ?? `ID ${program.id}`)
          .join(', ');

        this.logger.warn(
          `Target program validation failed: Programs already started - ${programNames}`,
        );

        throw new InifniBadRequestException(
          ERROR_CODES.PREFERENCE_PROGRAM_INVALID_START_DATE,
          null,
          null,
          `${programNames}`,
        );
      }
    } catch (error) {
      if (error instanceof InifniBadRequestException) {
        throw error;
      }

      this.logger.error(
        'Unexpected error validating target program start dates',
        error?.stack,
        { programIds: uniqueIds, error },
      );
    }
  }

  /**
   * Manages swap request (edit/cancel) operations
   * @param id - The ID of the swap request to update
   * @param userId - The ID of the user performing the action
   * @param data - The update data including action type and optional fields
   * @returns The updated swap request
   */
  async manageSwapRequest(
    id: number,
    userId: number,
    data: UpdateSwapRequestDto,
  ): Promise<ProgramRegistrationSwap> {
    this.logger.log(programRegistrationMessages.MANAGE_SWAP_REQUEST, { id, userId, action: data.action });
    
    try {
      // Get the existing swap request
      const swapRequest = await this.swapRepo.findOneById(id);
      if (!swapRequest) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            null,
            null,
            id.toString()
          )
        );
      }

      // Validate that the swap request is in a state that can be modified
      if (swapRequest.status !== SwapRequestStatus.ACTIVE) {
        handleKnownErrors(
          ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
          new InifniBadRequestException(
            ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
            null,
            null,
            swapRequest.status
          )
        );
      }

      let updateData: any = {
        updatedBy: userId,
        updatedAt: new Date(),
      };

      if (data.action === 'edit') {
        // For edit action, validate required fields
        if (!data.targetPrograms || data.targetPrograms.length === 0) {
          handleKnownErrors(
            ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
            new InifniBadRequestException(
              ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
              null,
              null,
              'Target programs are required for edit action'
            )
          );
        }

        const targetProgramIds = data.targetPrograms.map((program) => program.id);

        // Validate that all target programs exist
        for (const targetProgram of data.targetPrograms) {
          const program = await this.programRepo.findOneById(targetProgram.id);
          if (!program) {
            handleKnownErrors(
              ERROR_CODES.PROGRAM_NOTFOUND,
              new InifniNotFoundException(
                ERROR_CODES.PROGRAM_NOTFOUND,
                null,
                null,
                targetProgram.id.toString()
              )
            );
          }
        }

        await this.validateTargetProgramStartDates(targetProgramIds);

        // Note: requestedPrograms will be handled explicitly in the transaction below

        // Update type if provided
        if (data.type !== undefined) {
          updateData.type = data.type;
        }

        // Update comment if provided
        if (data.comment !== undefined) {
          updateData.comment = data.comment;
        }

        this.logger.log(programRegistrationMessages.EDIT_SWAP_REQUEST, { 
          id, 
          targetPrograms: data.targetPrograms,
          type: data.type,
          comment: data.comment 
        });
        
      } else if (data.action === 'cancel') {
        // For cancel action, set status to closed
        updateData.status = SwapRequestStatus.CLOSED;
        
        // Update comment if provided
        if (data.comment !== undefined) {
          updateData.comment = data.comment;
        }

        this.logger.log(programRegistrationMessages.CANCEL_SWAP_REQUEST, { id, comment: data.comment });
      } else {
        handleKnownErrors(
          ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
          new InifniBadRequestException(
            ERROR_CODES.INVALID_SWAP_REQUEST_STATUS,
            null,
            null,
            data.action
          )
        );
      }
      
      // Update the swap request
      if (data.action === 'edit' && data.targetPrograms && data.targetPrograms.length > 0) {
        // For edit with target programs, use transaction to ensure proper M2M handling
        const targetPrograms = data.targetPrograms;
        return await this.dataSource.transaction(async (manager: EntityManager) => {
          // Update the main entity
          const updatedSwapRequest = await this.swapRepo.updateAndSave(id, updateData, manager);
          
          // Explicitly handle the many-to-many relationship using TypeORM
          // Delete existing relationships
          await manager
            .createQueryBuilder()
            .delete()
            .from('hdb_swap_requested_program')
            .where('swap_request_id = :swapRequestId', { swapRequestId: id })
            .execute();
          
          // Insert new relationships
          for (const program of targetPrograms) {
            await manager
              .createQueryBuilder()
              .insert()
              .into('hdb_swap_requested_program')
              .values({
                swap_request_id: id,
                program_id: program.id
              })
              .execute();
          }
          
          return updatedSwapRequest;
        });
      } else {
        // For cancel or edit without target programs, simple update
        return await this.swapRepo.updateAndSave(id, updateData);
      }
      
    } catch (error) {
      this.logger.error('Error managing swap request', '', { id, userId, action: data.action, error });
      throw error;
    }
  }

  /**
   * Sends swap demand communication (email and WATI) when swap request is ON_HOLD
   * @param registrationId - Registration ID
   * @param userId - User ID
     */
    async sendSwapRequestDemandCommunication(registrationDetails: any) {
      try {
        this.logger.log('Sending swap demand communication for registration ID:', registrationDetails);
        if (!registrationDetails) {
          handleKnownErrors(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            new InifniNotFoundException(
              ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
              null,
              null,
              registrationDetails.id.toString(),
            ),
          );
        }
        // Email template
        const templateKey = process.env.ZEPTO_SWAP_DEMAND_TEMPLATE_ID || '';
        if (!templateKey || templateKey.trim() === '') {
          this.logger.log('Zepto swap demand template key not configured');
          return;
        }
        const mergeInfo = {
          reg_name: registrationDetails.fullName ?? 'Infinitheist',
          hdb_msd: registrationDetails?.allocatedProgram?.name ?? '',
        };
        // Add RM email to bcc if available
        let bccEmails: { emailAddress: string; name: string }[] = [];
        if (process.env.HDB_BCC_EMAIL) {
          bccEmails.push({
            emailAddress: process.env.HDB_BCC_EMAIL,
            name: process.env.HDB_BCC_EMAIL_NAME || 'HDB Team',
          });
        }
        if (registrationDetails.rmContactUser && registrationDetails.rmContactUser.email) {
          bccEmails.push({
            emailAddress: registrationDetails.rmContactUser.email,
            name: registrationDetails.rmContactUser.orgUsrName || registrationDetails.rmContactUser.legalFullName || 'RM',
          });
        }
        const emailData: SendSingleEmailDto = {
          templateKey,
          from: {
            address: zeptoEmailCreadentials.ZEPTO_EMAIL,
            name:
              registrationDetails.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
          },
          to: {
            emailAddress: registrationDetails.emailAddress,
            name: registrationDetails.fullName,
          },
          cc: [],
          bcc: bccEmails,
          mergeInfo,
          attachments: undefined,
          subject: '',
          trackinfo: {
            registrationId: registrationDetails.id,
          },
        };
        try {
          await this.communicationService.sendSingleEmail(emailData);
        } catch (error) {
          this.logger.log('Error sending swap demand email', { error });
        }
        // WATI template
        const watiTemplateName = process.env.WATI_SWAP_DEMAND_TEMPLATE_ID || '';
        if (!watiTemplateName || watiTemplateName.trim() === '') {
          this.logger.log('WATI swap demand template name not configured');
          return;
        }
        const watiPayload = {
          templateName: watiTemplateName,
          whatsappNumber: registrationDetails.mobileNumber.replace(/^\+/, ''),
          broadcastName: watiTemplateName,
          parameters: [
            { name: 'reg_name', value: mergeInfo.reg_name },
            { name: 'hdb_msd', value: mergeInfo.hdb_msd },
          ],
          trackinfo: { registrationId: registrationDetails.id },
        };
        this.logger.log('WATI Payload swap demand:', watiPayload);
        try {
          await this.communicationService.sendTemplateMessage(watiPayload);
        } catch (error) {
          this.logger.log('Error sending swap demand WATI message', { error });
        }

        // Also send WATI to RM if required
        // if (registrationDetails.rmContactUser && registrationDetails.rmContactUser.phoneNumber) {
        //   let whatsappNumber = `${registrationDetails.rmContactUser.countryCode || ''}${registrationDetails.rmContactUser.phoneNumber || ''}`;
        //   whatsappNumber = whatsappNumber.replace(/^\+/, '');
        //   const rmWatiPayload = {
        //     templateName: watiTemplateName,
        //     whatsappNumber,
        //     broadcastName: watiTemplateName,
        //     parameters: [
        //       { name: 'reg_name', value: mergeInfo.reg_name },
        //       { name: 'hdb_msd', value: mergeInfo.hdb_msd },
        //     ],
        //     trackinfo: { registrationId: registrationDetails.id },
        //   };
        //   this.logger.log('RM WATI Payload:', rmWatiPayload);
        //  // Send WATI message to RM
        // try {
        //   await this.communicationService.sendTemplateMessage(rmWatiPayload);
        // } catch (error) {
        //   this.logger.log('Error sending RM swap demand WATI message', { error });
        // }
        // }

      } catch (error) {
        this.logger.log('Error sending swap demand communication', { error });
      }
    }

    async processOnHoldApproval(registration: any, user: number) {
      try {
        this.logger.log('Processing ON_HOLD approval for registration ID:', registration.id);
        if (!registration) {
          handleKnownErrors(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            new InifniNotFoundException(
              ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
              null,
              null,
              registration.id.toString(),
            ),
          );
        }

        //update the swap request to on_hold status and swap requirement to swap demand
        const swapRequest = await this.swapRepo.findOneByWhere({
          programRegistrationId: registration.id,
          status: SwapRequestStatus.ACTIVE,
        });
        this.logger.log(`Found swap request for ON_HOLD processing:, swapRequest`, { swapRequest });
        if (swapRequest) {
          await this.swapRepo.updateAndSave(swapRequest.id, {
            status: SwapRequestStatus.ON_HOLD,
            swapRequirement: SwapRequirementEnum.SWAP_DEMAND,
            updatedBy: user,
          });
        }

        this.logger.log('Updated swap request to ON_HOLD status and SWAP_DEMAND requirement', { swapRequest });

        await this.sendSwapRequestDemandCommunication(registration);
        // Clear travel details when swap demand is triggered
        await this.registrationRepo.resetTravelDetails(registration.id, user);

      } catch (error) {
        this.logger.log('Error processing ON_HOLD approval', { error });
      }
    }

    async processRejectedApproval(registration: any, user: number) {
      try {
        this.logger.log('Processing REJECTED approval for registration ID:', registration.id);
        if (!registration) {
          handleKnownErrors(
            ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
            new InifniNotFoundException(
              ERROR_CODES.PROGRAM_REGISTRATION_NOT_FOUND,
              null,
              null,
              registration.id.toString(),
            ),
          );
        }

        //update the swap request to rejected status
        const swapRequest = await this.swapRepo.findOneByWhere({
          programRegistrationId: registration.id,
          status: SwapRequestStatus.ACTIVE,
        });
        this.logger.log(`Found swap request for REJECTED processing:, swapRequest`, { swapRequest });
        if (swapRequest) {
          await this.swapRepo.updateAndSave(swapRequest.id, {
            status: SwapRequestStatus.REJECTED,
            updatedBy: user
        });
        }

        this.logger.log('Updated swap request to REJECTED status', { swapRequest });

      } catch (error) {
        this.logger.log('Error processing REJECTED approval', { error });
      }
    }

    async completeSwapRequestOnProgramAllocation(registration: any, userId: number): Promise<void> {
      try {
        this.logger.log(`Completing swap request on program allocation for registration ID: ${registration.id}`);
        const swapRequest = await this.swapRepo.findOneByWhere({
          programRegistrationId: registration.id,
          status: SwapRequestStatus.ACTIVE,
        });
        if (swapRequest) {
          await this.swapRepo.updateAndSave(swapRequest.id, {
            status: SwapRequestStatus.ACCEPTED,
            updatedBy: userId,
          });
          this.logger.log('Swap request marked as ACCEPTED due to program allocation', { swapRequest });
        } else {
          this.logger.log('No active swap request found for completion', { registration });
        }
      } catch (error) {
        this.logger.log('Error completing swap request on program allocation', { error });
      }
  }
}
