import { Type } from 'class-transformer';
import {
  ArrayMinSize,
  IsIn,
  IsNotEmpty,
  IsNumber,
  IsOptional,
  IsString,
  Max,
  Min,
  ValidateNested,
} from 'class-validator';
import { ApiProperty } from '@nestjs/swagger';
import { RecommendationLevel } from 'src/common/enum/recommendation-level.enum';

const RATING_KEYS = ['passion', 'growth', 'infinitheist', 'continuity', 'overall'];

export class RmRatingDto {
  @ApiProperty({
    description: 'The key for the rating',
    enum: RATING_KEYS,
    example: 'passion',
  })
  @IsNotEmpty()
  @IsString()
  @IsIn(RATING_KEYS)
  ratingKey: string;

  @ApiProperty({
    description: 'The rating value',
    minimum: 0,
    maximum: 5,
    example: 4,
  })
  @IsNotEmpty()
  @IsNumber()
  @Min(0)
  @Max(5)
  rating: number;
}

export class UpdateRmRatingDto extends RmRatingDto {
  @ApiProperty({
    description: 'Unique identifier for the rating',
    example: 1,
  })
  @IsNotEmpty()
  @IsNumber()
  id: number;
}

export class ProgramRegistrationRecommendationsDto {
  @ApiProperty({
    description: 'Unique identifier for the recommendation',
    type: Number,
    example: 1001,
  })
  @IsNumber()
  @IsOptional()
  id: number;

  @ApiProperty({
    description: 'Recommendation required or not',
    type: Boolean,
    example: true,
  })
  @IsNotEmpty()
  isRecommended: boolean;

  @ApiProperty({
    description: 'Recommendation level',
    enum: RecommendationLevel,
    example: RecommendationLevel.AFFIRMATIVELY,
  })
  @IsIn(Object.values(RecommendationLevel))
  @IsString()
  @Type(() => String)
  @IsOptional()
  recommendationKey?: RecommendationLevel;

  @ApiProperty({
    description: 'recommended by user ID',
    type: Number,
    example: 1001,
  })
  @IsNotEmpty()
  @IsNumber()
  @Type(() => Number)
  recommendedBy: number;

  @ApiProperty({
    description: 'Recommendation text',
    example: 'Highly recommend this program for anyone looking to improve their skills.',
    required: false,
  })
  @IsString()
  @IsOptional()
  recommendationText?: string;

  @ApiProperty({
    description: 'Follow-up count for the recommendation',
    type: Number,
    example: 2,
    required: false,
  })
  @IsNumber()
  @IsOptional()
  @Type(() => Number)
  followUpCount?: number;
}

export class ProgramRegistrationRMRatingDto {
  @ApiProperty({
    description: 'List of ratings',
    type: [RmRatingDto],
    minItems: 1,
    example: [
      { ratingKey: 'passion', rating: 5 },
      { ratingKey: 'growth', rating: 4 },
      { ratingKey: 'infinitheist', rating: 3 },
      { ratingKey: 'continuity', rating: 4 },
      { ratingKey: 'overall', rating: 4 },
    ],
  })
  @IsNotEmpty()
  @ValidateNested({ each: true })
  @Type(() => RmRatingDto)
  // @ArrayMinSize(RATING_KEYS.length)
  ratings: RmRatingDto[];

  @ApiProperty({
    description: 'Review text',
    example: 'Great program, learned a lot!',
  })
  @IsNotEmpty()
  @IsString()
  review: string;

  @ApiProperty({
    description: 'Recommendation text',
  })
  @IsOptional()
  @ValidateNested({ each: true })
  @Type(() => ProgramRegistrationRecommendationsDto)
  recommendation?: ProgramRegistrationRecommendationsDto;
}

export class UpdateProgramRegistrationRMRatingDto  {
  
  @ApiProperty({
    description: 'List of ratings with IDs for updates',
    type: [UpdateRmRatingDto],
    minItems: 1,
    example: [
      { id: 1, ratingKey: 'passion', rating: 5 },
      { id: 2, ratingKey: 'growth', rating: 4 },
      { id: 3, ratingKey: 'infinitheist', rating: 3 },
      { id: 4, ratingKey: 'continuity', rating: 4 },
      { id: 5, ratingKey: 'overall', rating: 4 },
    ],
  })
  @IsNotEmpty()
  @ValidateNested({ each: true })
  @Type(() => UpdateRmRatingDto)
  // @ArrayMinSize(RATING_KEYS.length)
  ratings: UpdateRmRatingDto[];

  @ApiProperty({
    description: 'Review text',
    example: 'Great program, learned a lot!',
  })
  @IsNotEmpty()
  @IsString()
  review: string;

  @ApiProperty({
    description: 'Recommendation text',
    // required: false,
  })
  @IsOptional()
  recommendation?: ProgramRegistrationRecommendationsDto;
}
