import { Injectable } from '@nestjs/common';
import { CreateProgramQuestionDto } from './dto/create-program-question.dto';
import { UpdateProgramQuestionDto } from './dto/update-program-question.dto';
import { UpdateProgramQuestionBulkDto } from './dto/update-program-question.dto'; 
import { DeleteProgramQuestionDto } from './dto/delete-program-question.dto';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { programQuestionConstMessages } from 'src/common/constants/strings-constants';
import { ProgramQuestionRepository } from './program-question.repository';
import { RegistrationLevelEnum } from 'src/common/enum/program.enums';
import { User, ProgramQuestion, Question, FormSection } from 'src/common/entities';
import { generateFormSectionKey } from 'src/common/utils/common.util';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';

@Injectable()
export class ProgramQuestionService {
  constructor(
    private readonly repository: ProgramQuestionRepository,
    private readonly logger: AppLoggerService,
  ) {}

  // ==================== CREATE OPERATIONS ====================

  /**
   * Creates new program question mappings based on provided sections and questions
   * @param dto - CreateProgramQuestionDto containing program info and sections
   * @param user - User performing the operation
   * @returns Promise<ProgramQuestion[]> - Array of created mappings
   */
  async create(dto: CreateProgramQuestionDto, user: User) {
    this.logger.log(programQuestionConstMessages.SERVICE_CREATE_CALLED, dto);
    try {
      return await this.repository.executeInTransaction(async (transactionalEntityManager) => {
        // Validate basic requirements (program exists, session validation, etc.)
        await this.validateBasicRequirements(dto, this.repository, this.logger);

        // Process sections and build questions array
        const questionsArr = await this.processSections(
          dto.sections, 
          user, 
          this.repository, 
          this.logger, 
          transactionalEntityManager
        );

        // Validate questions and check for duplicates
        await this.validateQuestionsAndCheckDuplicates(
          dto.programId,
          dto.programSessionId ?? null,
          questionsArr,
          this.repository,
          this.logger,
          transactionalEntityManager,
        );

        // Create mappings
        return await this.createProgramQuestionMappings(
          dto,
          questionsArr,
          user,
          this.repository,
          this.logger,
        );
      });
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_CREATION, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_CREATION_FAILED, error);
    }
  }

  // ==================== READ OPERATIONS ====================

  /**
   * Retrieves all program question mappings with pagination and filtering
   * @param limit - Number of records to return
   * @param offset - Number of records to skip  
   * @param programId - Optional filter by program ID
   * @param programSessionId - Optional filter by program session ID
   * @param registrationLevel - Optional filter by registration level
   * @returns Promise<object> - Paginated data and metadata
   */
  async findAll(
    limit: number,
    offset: number,
    programId: number | null = null,
    programSessionId: number | null = null,
    registrationLevel: RegistrationLevelEnum | null = null,
  ) {
    this.logger.log(programQuestionConstMessages.SERVICE_FIND_ALL_CALLED, {
      limit,
      offset,
      programId,
      programSessionId,
      registrationLevel: registrationLevel ?? '',
    });
    
    try {
      const { data, total } = await this.repository.findAllProgramQuestions(
        limit,
        offset,
        programId,
        programSessionId,
        registrationLevel,
      );

      const pagination = await this.repository.calculatePagination(total, limit, offset);
      
      return { data, pagination };
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_FIND_ALL_FAILED, error);
    }
  }

  /**
   * Retrieves a single program question mapping by ID
   * @param id - Mapping ID
   * @returns Promise<ProgramQuestion> - The found mapping
   * @throws InifniNotFoundException if mapping not found
   */
  async findOne(id: number) {
    this.logger.log(programQuestionConstMessages.SERVICE_FIND_ONE_CALLED(id));
    
    try {
      const mapping = await this.repository.findProgramQuestionById(id);
      if (!mapping) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_QUESTION_NOTFOUND,
          null,
          null,
          id.toString(),
        );
      }
      return mapping;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_FIND_BY_ID_FAILED, error);
    }
  }

  // ==================== UPDATE OPERATIONS ====================

  /**
   * Updates a single program question mapping by ID
   * @param id - Mapping ID to update
   * @param dto - UpdateProgramQuestionDto containing update data
   * @param userId - ID of user performing the operation
   * @returns Promise<ProgramQuestion> - The updated mapping
   */
  async update(id: number, dto: UpdateProgramQuestionDto, userId: number) {
    this.logger.log(programQuestionConstMessages.SERVICE_UPDATE_CALLED(id), dto);
    
    try {
      // Find existing mapping
      const mapping = await this.repository.findProgramQuestionById(id);
      if (!mapping) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_QUESTION_NOTFOUND,
          null,
          null,
          id.toString(),
        );
      }

      // Validate user
      const user = await this.repository.findUserById(userId);
      if (!user) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
      }

      // Apply updates
      await this.applyMappingUpdates(mapping, dto, user);

      return await this.repository.updateProgramQuestion(mapping);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_UPDATE, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_UPDATE_FAILED, error);
    }
  }

  /**
   * Bulk update operation that handles sections, questions, deletions, and creations
   * @param dto - UpdateProgramQuestionBulkDto containing all update operations
   * @param user - User performing the operation
   * @returns Promise<object> - Summary of all operations performed
   */
  async bulkUpdate(dto: UpdateProgramQuestionBulkDto, user: User) {
    this.logger.log('SERVICE_BULK_UPDATE_CALLED', dto);
    try {
      return await this.repository.executeInTransaction(async (transactionalEntityManager) => {
        // Validate basic requirements
        await this.validateBasicRequirements(dto, this.repository, this.logger);

        // Step 1: Process deletions first
        const deletionResults = await this.processDeletions(
          dto,
          user,
          this.repository,
          this.logger,
        );

        // Step 2: Get existing mappings for this program/session to exclude from duplicate checks
        const existingMappings = await this.repository.findExistingMappingsInScope(
          dto.programId,
          dto.programSessionId ?? null,
          [], // We'll check all questions individually
          transactionalEntityManager,
        );
        const existingMappingIds = existingMappings.map(m => m.id);

        // Step 3: Process sections (creates new sections/questions, updates existing)
        const questionsArr = await this.processSections(
          dto.sections,
          user,
          this.repository,
          this.logger,
          transactionalEntityManager,
        );

        let createdMappings: ProgramQuestion[] = [];
        if (questionsArr.length > 0) {
          // Validate questions and check for duplicates (excluding existing mappings)
          await this.validateQuestionsAndCheckDuplicates(
            dto.programId,
            dto.programSessionId ?? null,
            questionsArr,
            this.repository,
            this.logger,
            transactionalEntityManager,
            existingMappingIds,
          );

          // Create new mappings
          createdMappings = await this.createProgramQuestionMappings(
            dto,
            questionsArr,
            user,
            this.repository,
            this.logger,
          );
        }

        // Step 4: Return comprehensive results
        return {
          message: 'Bulk update operation completed successfully',
          created: {
            mappings: createdMappings.length,
            details: createdMappings.map(m => ({
              id: m.id,
              questionId: m.question.id,
              displayOrder: m.displayOrder,
            })),
          },
          deleted: deletionResults,
          summary: {
            totalOperations: createdMappings.length + deletionResults.deletedSections + deletionResults.deletedQuestions,
            createdMappings: createdMappings.length,
            deletedSections: deletionResults.deletedSections,
            deletedQuestions: deletionResults.deletedQuestions,
          },
        };
      });
    } catch (error) {
      this.logger.error('ERROR_CONTEXT_BULK_UPDATE', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_BULK_UPDATE_FAILED, error);
    }
  }

  /**
   * Applies updates to a single program question mapping
   * @param mapping - The mapping to update
   * @param dto - Update data
   * @param user - User performing the operation
   * @private
   */
  private async applyMappingUpdates(
    mapping: ProgramQuestion,
    dto: UpdateProgramQuestionDto,
    user: User,
  ): Promise<void> {
    // Update program reference
    if (dto.programId) {
      this.logger.log(programQuestionConstMessages.UPDATE_PROGRAM_REFERENCE(mapping.program.id, dto.programId));
      const program = await this.repository.findProgramById(dto.programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          dto.programId.toString(),
        );
      }
      mapping.program = program;
    }

    // Update question reference
    if (dto.questionId) {
      await this.validateAndUpdateQuestionReference(mapping, dto, user);
    }

    // Update display order
    if (dto.displayOrder !== undefined) {
      this.logger.log(programQuestionConstMessages.UPDATE_DISPLAY_ORDER(mapping.displayOrder, dto.displayOrder));
      mapping.displayOrder = dto.displayOrder;
    }

    // Update registration level and session
    if (dto.registrationLevel !== undefined) {
      await this.updateRegistrationLevelAndSession(mapping, dto);
    } else if (dto.programSessionId !== undefined) {
      await this.updateStandaloneProgramSession(mapping, dto);
    }

    mapping.updatedBy = user;
  }

  /**
   * Validates and updates question reference for a mapping
   * @param mapping - The mapping to update
   * @param dto - Update data containing new question ID
   * @param user - User performing the operation
   * @private
   */
  private async validateAndUpdateQuestionReference(
    mapping: ProgramQuestion,
    dto: UpdateProgramQuestionDto,
    user: User,
  ): Promise<void> {
    this.logger.log(
      programQuestionConstMessages.UPDATE_QUESTION_REFERENCE(
        mapping.question?.id ?? undefined,
        dto.questionId ?? -1
      )
    );
    
    // Check if the new question already exists in the same scope (excluding current mapping)
    const scopeProgramId = dto.programId || mapping.program.id;
    const scopeSessionId = dto.programSessionId !== undefined
      ? dto.programSessionId
      : mapping.programSession
        ? mapping.programSession.id
        : null;

    if (dto.questionId === undefined) {
      throw new InifniBadRequestException(
        ERROR_CODES.QUESTION_ID_REQUIRED,
        null,
        null,
        'Question ID is required for updating the question reference.',
      );
    }

    const existingMapping = await this.repository.findExistingMappingForUpdate(
      scopeProgramId,
      scopeSessionId,
      dto.questionId,
      mapping.id,
    );

    if (existingMapping) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_QUESTION_DUPLICATE_BADREQUEST,
        null,
        null,
        scopeProgramId.toString(),
        dto.questionId.toString(),
      );
    }

    const questions = await this.repository.findPublishedQuestionsByIds([dto.questionId]);
    if (!questions || questions.length === 0) {
      throw new InifniNotFoundException(
        ERROR_CODES.QUESTION_NOTFOUND_OR_UNPUBLISHED,
        null,
        null,
        dto.questionId.toString(),
      );
    }
    
    mapping.question = questions[0];
  }

  /**
   * Updates registration level and associated session
   * @param mapping - The mapping to update
   * @param dto - Update data
   * @private
   */
  private async updateRegistrationLevelAndSession(
    mapping: ProgramQuestion,
    dto: UpdateProgramQuestionDto,
  ): Promise<void> {
    this.logger.log(programQuestionConstMessages.UPDATE_REGISTRATION_LEVEL(
      mapping.registrationLevel,
      dto.registrationLevel ?? RegistrationLevelEnum.PROGRAM,
    ));
    
    if (dto.registrationLevel !== undefined) {
      mapping.registrationLevel = dto.registrationLevel;
    }

    if (dto.registrationLevel === RegistrationLevelEnum.SESSION) {
      if (!dto.programSessionId) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_SESSION_REQUIRED_FOR_SESSION_LEVEL,
          null,
          null,
          'Program session ID is required when registration level is session',
        );
      }
      
      const programSession = await this.repository.findProgramSessionById(dto.programSessionId, true);
      if (!programSession) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
          null,
          null,
          dto.programSessionId.toString(),
        );
      }
      
      // Validate that session belongs to the program
      if (programSession.program.id !== mapping.program.id) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_SESSION_MISMATCH,
          null,
          null,
          'Program session does not belong to the specified program',
        );
      }
      
      this.logger.log(programQuestionConstMessages.UPDATE_SESSION_REFERENCE(dto.programSessionId));
      mapping.programSession = programSession;
    } else {
      // If changing to program level, remove session reference
      this.logger.log(programQuestionConstMessages.UPDATE_SESSION_REFERENCE(null));
      mapping.programSession = null;
    }
  }

  /**
   * Updates program session reference independently
   * @param mapping - The mapping to update
   * @param dto - Update data
   * @private
   */
  private async updateStandaloneProgramSession(
    mapping: ProgramQuestion,
    dto: UpdateProgramQuestionDto,
  ): Promise<void> {
    if (dto.programSessionId) {
      const programSession = await this.repository.findProgramSessionById(dto.programSessionId, true);
      if (!programSession) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
          null,
          null,
          dto.programSessionId.toString(),
        );
      }
      
      // Validate that session belongs to the program
      if (programSession.program.id !== mapping.program.id) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_SESSION_MISMATCH,
          null,
          null,
          'Program session does not belong to the specified program',
        );
      }
      
      this.logger.log(programQuestionConstMessages.UPDATE_SESSION_REFERENCE(dto.programSessionId));
      mapping.programSession = programSession;
      mapping.registrationLevel = RegistrationLevelEnum.SESSION;
    } else {
      this.logger.log(programQuestionConstMessages.UPDATE_SESSION_REFERENCE(null));
      mapping.programSession = null;
      mapping.registrationLevel = RegistrationLevelEnum.PROGRAM;
    }
  }

  // ==================== DELETE OPERATIONS ====================

  /**
   * Soft deletes a single program question mapping by ID
   * @param id - Mapping ID to delete
   * @param userId - ID of user performing the operation
   * @returns Promise<ProgramQuestion> - The deleted mapping
   */
  async remove(id: number, userId: number) {
    this.logger.log(programQuestionConstMessages.SERVICE_DELETE_CALLED(id));
    
    try {
      // Find existing mapping
      const mapping = await this.repository.findProgramQuestionById(id);
      if (!mapping) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_QUESTION_NOTFOUND,
          null,
          null,
          id.toString(),
        );
      }

      // Validate user
      const user = await this.repository.findUserById(userId);
      if (!user) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
      }

      mapping.updatedBy = user;
      return await this.repository.softDeleteProgramQuestion(mapping);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_DELETE, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_DELETE_FAILED, error);
    }
  }

  /**
   * Bulk delete operation for sections and individual questions
   * @param dto - DeleteProgramQuestionDto containing deletion instructions
   * @param userId - ID of user performing the operation
   * @returns Promise<object> - Summary of deletion operations
   */
  async delete(dto: DeleteProgramQuestionDto, userId: number) {
    this.logger.log(programQuestionConstMessages.SERVICE_BULK_DELETE_CALLED, dto);
    
    try {
      return await this.repository.executeInTransaction(async (transactionalEntityManager) => {
        // Validate prerequisites
        await this.validateBulkDeleteRequest(dto, userId);

        const results: Array<{
          type: string;
          sectionId?: number;
          deletedQuestions?: number;
          message?: string;
          questionIds?: number[];
          deletedCount?: number;
        }> = [];

        // Process section deletions
        if (dto.sections && dto.sections.length > 0) {
          const sectionResults = await this.processSectionDeletions(dto, userId);
          results.push(...sectionResults);
        }

        // Process individual question deletions
        if (dto.questionIds && dto.questionIds.length > 0) {
          const questionResults = await this.processQuestionDeletions(dto, userId);
          results.push(...questionResults);
        }

        this.logger.log(programQuestionConstMessages.BULK_DELETE_COMPLETED(results.length));
        
        return {
          message: 'Bulk delete operation completed',
          results: results,
          totalOperations: results.length,
        };
      });
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_DELETE, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_BULK_DELETE_FAILED, error);
    }
  }

  /**
   * Validates requirements for bulk delete operations
   * @param dto - Delete operation data
   * @param userId - User ID performing the operation
   * @private
   */
  private async validateBulkDeleteRequest(dto: DeleteProgramQuestionDto, userId: number): Promise<void> {
    // Validate user
    const user = await this.repository.findUserById(userId);
    if (!user) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
    }

    // Validate program
    const program = await this.repository.findProgramById(dto.programId);
    if (!program) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        dto.programId.toString(),
      );
    }
  }

  /**
   * Processes section deletions (sections and their associated questions)
   * @param dto - Delete operation data
   * @param userId - User ID performing the operation
   * @private
   */
  private async processSectionDeletions(
    dto: DeleteProgramQuestionDto,
    userId: number,
  ): Promise<Array<{ type: string; sectionId: number; deletedQuestions: number; message: string }>> {
    this.logger.log(programQuestionConstMessages.PROCESSING_SECTION_DELETIONS);
    
    const results: Array<{ type: string; sectionId: number; deletedQuestions: number; message: string }> = [];
    const user = await this.repository.findUserById(userId);
    if (!user) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
    }

    if (dto.sections) {
      for (const sectionId of dto.sections) {
        this.logger.log(programQuestionConstMessages.DELETING_SECTION(sectionId));
        
        const sectionToDelete = await this.repository.findFormSectionById(sectionId);
        if (!sectionToDelete) {
          throw new InifniNotFoundException(
            ERROR_CODES.FORM_SECTION_NOTFOUND,
            null,
            null,
            sectionId.toString(),
          );
        }

        // Find and soft delete all program questions in this section
        const programQuestionsInSection = await this.repository.findProgramQuestionsInSection(
          dto.programId,
          dto.programSessionId || null,
          sectionId,
        );

        if (programQuestionsInSection.length > 0) {
          const updatedMappings = programQuestionsInSection.map(pq => {
            pq.deletedAt = new Date();
            pq.updatedBy = user;
            return pq;
          });
          await this.repository.updateProgramQuestions(updatedMappings);
        }

        // Find and soft delete all questions in this section
        const questionsInSection = await this.repository.findQuestionsInSection(sectionId);
        if (questionsInSection.length > 0) {
          const updatedQuestions = questionsInSection.map(question => {
            question.deletedAt = new Date();
            question.updatedBy = user;
            return question;
          });
          await this.repository.softDeleteQuestions(updatedQuestions);
        }

        // Soft delete the section
        sectionToDelete.deletedAt = new Date();
        sectionToDelete.updatedBy = user.id;
        await this.repository.softDeleteFormSection(sectionToDelete);

        const result = {
          type: 'section',
          sectionId: sectionId,
          deletedQuestions: programQuestionsInSection.length,
          message: `Section ${sectionId} and ${programQuestionsInSection.length} questions deleted`,
        };
        
        this.logger.log(programQuestionConstMessages.SECTION_DELETED(sectionId, programQuestionsInSection.length));
        results.push(result);
      }
    }

    return results;
  }

  /**
   * Processes individual question deletions
   * @param dto - Delete operation data
   * @param userId - User ID performing the operation
   * @private
   */
  private async processQuestionDeletions(
    dto: DeleteProgramQuestionDto,
    userId: number,
  ): Promise<Array<{ type: string; questionIds: number[]; deletedCount: number; message: string }>> {
    this.logger.log(programQuestionConstMessages.PROCESSING_QUESTION_DELETIONS);
    
    const user = await this.repository.findUserById(userId);

    if (!user) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
    }
    
    this.logger.log(programQuestionConstMessages.DELETING_QUESTIONS(dto.questionIds ?? []));
    
    const programQuestionsToDelete = await this.repository.findProgramQuestionsToDelete(
      dto.programId,
      dto.programSessionId || null,
      dto.questionIds ?? [],
    );

    if (programQuestionsToDelete.length > 0) {
      const updatedMappings = programQuestionsToDelete.map(pq => {
        pq.deletedAt = new Date();
        pq.updatedBy = user;
        return pq;
      });
      await this.repository.updateProgramQuestions(updatedMappings);
    }

    const result = {
      type: 'questions',
      questionIds: dto.questionIds ?? [],
      deletedCount: programQuestionsToDelete.length,
      message: `${programQuestionsToDelete.length} questions deleted`,
    };

    this.logger.log(programQuestionConstMessages.QUESTIONS_DELETED(programQuestionsToDelete.length));
    
    return [result];
  }

  /**
   * Validates the basic requirements for creating/updating program question mappings
   * @param dto - The DTO containing programId, registrationLevel, and programSessionId
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   * @throws InifniNotFoundException if program or program session not found
   * @throws InifniBadRequestException if validation fails
   */
  private async validateBasicRequirements(
    dto: { programId: number; registrationLevel: RegistrationLevelEnum; programSessionId?: number },
    repository: ProgramQuestionRepository,
    logger: AppLoggerService
  ): Promise<void> {
    logger.log(programQuestionConstMessages.VALIDATION_COMPLETED);
    
    // Validate program exists
    const program = await repository.findProgramById(dto.programId);
    if (!program) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        dto.programId.toString(),
      );
    }

    // Validate program session if registration level is session
    if (dto.registrationLevel === RegistrationLevelEnum.SESSION) {
      if (!dto.programSessionId) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_SESSION_REQUIRED_FOR_SESSION_LEVEL,
          null,
          null,
          'Program session ID is required when registration level is session',
        );
      }

      const programSession = await repository.findProgramSessionById(dto.programSessionId, true);
      if (!programSession) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
          null,
          null,
          dto.programSessionId.toString(),
        );
      }

      if (programSession.program.id !== dto.programId) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_SESSION_MISMATCH,
          null,
          null,
          'Program session does not belong to the specified program',
        );
      }
    }
  }

  /**
   * Processes a single section for create/update operations
   * @param section - Section data from DTO
   * @param user - Current user performing the operation
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   * @param transactionalEntityManager - Database transaction manager
   * @returns Promise<FormSection> - The processed section
   */
  private async processSection(
    section: any,
    user: User,
    repository: ProgramQuestionRepository,
    logger: AppLoggerService,
    transactionalEntityManager: any,
  ): Promise<FormSection> {
    let currentSection: FormSection;

    if (section.sectionId && section.sectionId !== 0) {
      // Process existing section
      logger.log(programQuestionConstMessages.PROCESSING_EXISTING_SECTION(section.sectionId));
      
      const existingSection = await repository.findFormSectionById(section.sectionId);
      if (!existingSection) {
        throw new InifniNotFoundException(
          ERROR_CODES.FORM_SECTION_NOTFOUND,
          null,
          null,
          section.sectionId.toString(),
        );
      }
        currentSection = existingSection;
    } else {
      // Create new section
      logger.log(programQuestionConstMessages.PROCESSING_NEW_SECTION(section.sectionName));
      
      currentSection = await repository.createFormSection({
        name: section.sectionName,
        key: generateFormSectionKey(section.sectionName),
        displayOrder: section.sectionDisplayOrder || 0,
        createdBy: user.id,
        updatedBy: user.id,
      });
    }

    return currentSection;
  }

  /**
   * Processes custom questions within a section
   * @param customQuestions - Array of custom question data
   * @param currentSection - The section these questions belong to
   * @param user - Current user performing the operation
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   * @param transactionalEntityManager - Database transaction manager
   * @returns Promise<Array> - Array of question IDs with display order and section info
   */
  private async processCustomQuestions(
    customQuestions: any[],
    currentSection: FormSection,
    user: User,
    repository: ProgramQuestionRepository,
    logger: AppLoggerService,
    transactionalEntityManager: any,
  ): Promise<{ questionId: number; displayOrder: number; sectionId: number }[]> {
    logger.log(programQuestionConstMessages.PROCESSING_CUSTOM_QUESTIONS);
    
    const questionIds: { questionId: number; displayOrder: number; sectionId: number }[] = [];

    for (const cq of customQuestions) {
      const { question, options } = cq;

      // Create custom question
      const savedQuestion = await repository.createCustomQuestion({
        label: question.label,
        type: question.type,
        config: question.config || {},
        status: question.status || 'published',
        answerLocation: question.answerLocation,
        bindingKey: question.bindingKey,
        createdBy: user,
        updatedBy: user,
        formSection: currentSection,
      });

      logger.log(programQuestionConstMessages.CREATING_CUSTOM_QUESTION(question.label));

      // Process options for the custom question
      if (Array.isArray(options)) {
        await this.processQuestionOptions(options, savedQuestion, user, repository, logger);
      }

      const displayOrder = cq.displayOrder !== undefined ? cq.displayOrder : 0;
      questionIds.push({
        questionId: savedQuestion.id,
        displayOrder,
        sectionId: currentSection.id,
      });
    }

    return questionIds;
  }

  /**
   * Processes options for a custom question
   * @param options - Array of option data
   * @param question - The question these options belong to
   * @param user - Current user performing the operation
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   */
  private async processQuestionOptions(
    options: any[],
    question: Question,
    user: User,
    repository: ProgramQuestionRepository,
    logger: AppLoggerService,
  ): Promise<void> {
    logger.log(programQuestionConstMessages.PROCESSING_QUESTION_OPTIONS);
    
    for (const opt of options) {
      const savedOption = await repository.createOption({
        name: opt.name,
        type: opt.type,
        categoryId: opt.category === null || opt.category === undefined ? null : opt.category,
        status: 'published',
        createdBy: user,
        updatedBy: user,
      });

      await repository.createQuestionOptionMapping({
        question: question,
        option: savedOption,
        createdBy: user,
        updatedBy: user
      });

      logger.log(programQuestionConstMessages.CREATING_OPTION_MAPPING(question.id, savedOption.id));
    }
  }

  /**
   * Processes existing questions within a section
   * @param questions - Array of existing question data with IDs and display order
   * @param sectionId - The section ID these questions belong to
   * @returns Array of question IDs with display order and section info
   */
  private processExistingQuestions(
    questions: any[],
    sectionId: number,
  ): { questionId: number; displayOrder: number; sectionId: number }[] {
    return questions.map((q) => ({
      questionId: q.questionId,
      displayOrder: q.displayOrder ?? 0,
      sectionId: sectionId,
    }));
  }

  /**
   * Validates that questions exist and are published, and checks for duplicates in scope
   * @param programId - Program ID
   * @param programSessionId - Program session ID (optional)
   * @param questionsArr - Array of question data to validate
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   * @param transactionalEntityManager - Database transaction manager
   * @param excludeIds - Array of mapping IDs to exclude from duplicate check (for updates)
   */
  private async validateQuestionsAndCheckDuplicates(
    programId: number,
    programSessionId: number | null,
    questionsArr: { questionId: number; displayOrder: number; sectionId: number }[],
    repository: ProgramQuestionRepository,
    logger: AppLoggerService,
    transactionalEntityManager: any,
    excludeIds: number[] = [],
  ): Promise<void> {
    const questionIds = questionsArr.map((q) => q.questionId);

    // Check for duplicate questions in the same scope
    const existingMappings = await repository.findExistingMappingsInScope(
      programId,
      programSessionId || null,
      questionIds,
      transactionalEntityManager,
    );

    // Filter out mappings that should be excluded (for updates)
    const duplicateMappings = existingMappings.filter(mapping => !excludeIds.includes(mapping.id));

    if (duplicateMappings.length > 0) {
      const duplicateQuestionIds = duplicateMappings.map((m) => m.question.id);
      logger.log(programQuestionConstMessages.DUPLICATE_MAPPING_DETECTED(programId, duplicateQuestionIds[0]));
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_QUESTION_DUPLICATE_BADREQUEST,
        null,
        null,
        programId.toString(),
        duplicateQuestionIds.toString(),
      );
    }

    // Validate that all questions exist and are published
    await this.validatePublishedQuestions(questionIds, repository, logger);
  }

  /**
   * Validates that all provided question IDs exist and are published
   * @param questionIds - Array of question IDs to validate
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   */
  private async validatePublishedQuestions(
    questionIds: number[],
    repository: ProgramQuestionRepository,
    logger: AppLoggerService,
  ): Promise<void> {
    const questionEntities = await repository.findPublishedQuestionsByIds(questionIds);

    if (questionEntities.length !== questionIds.length) {
      const foundIds = questionEntities.map((q) => q.id);
      const missingIds = questionIds.filter((id) => !foundIds.includes(id));

      const errorMessage = `Missing question IDs: ${missingIds.join(', ')}`;
      
      throw new InifniBadRequestException(
        ERROR_CODES.QUESTION_NOTFOUND_OR_UNPUBLISHED,
        null,
        null,
        errorMessage,
      );
    }
  }

  /**
   * Creates program question mappings from processed question data
   * @param dto - The main DTO containing program and session info
   * @param questionsArr - Array of processed question data
   * @param user - Current user performing the operation
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   * @returns Promise<ProgramQuestion[]> - Array of created mappings
   */
  private async createProgramQuestionMappings(
    dto: { programId: number; programSessionId?: number | null; registrationLevel: RegistrationLevelEnum },
    questionsArr: { questionId: number; displayOrder: number; sectionId: number }[],
    user: User,
    repository: ProgramQuestionRepository,
    logger: AppLoggerService,
  ): Promise<ProgramQuestion[]> {
    logger.log(programQuestionConstMessages.CREATING_MAPPINGS);
    
    const questionIds = questionsArr.map((q) => q.questionId);
    const questionEntities = await repository.findPublishedQuestionsByIds(questionIds);

    const mappings = questionIds.map((qid, idx) => {
      const info = questionsArr.find((q) => q.questionId === qid);
      const questionEntity = questionEntities.find((q) => q.id === qid);
      
      return {
        program: { id: dto.programId },
        question: questionEntity,
        programSession: dto.programSessionId ? { id: dto.programSessionId } : null,
        registrationLevel: dto.registrationLevel,
        displayOrder: info?.displayOrder ?? idx,
        programQuestionFormSection: info ? { id: info.sectionId } : null,
        createdBy: user,
        updatedBy: user,
      } as any;
    });

    return await repository.createProgramQuestionMappings(mappings);
  }

  /**
   * Processes all sections in a DTO and returns consolidated question array
   * @param sections - Array of section data from DTO
   * @param user - Current user performing the operation
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   * @param transactionalEntityManager - Database transaction manager
   * @returns Promise<Array> - Consolidated array of all question data
   */
  private async processSections(
    sections: any[],
    user: User,
    repository: ProgramQuestionRepository,
    logger: AppLoggerService,
    transactionalEntityManager: any,
  ): Promise<{ questionId: number; displayOrder: number; sectionId: number }[]> {
    logger.log(programQuestionConstMessages.PROCESSING_SECTIONS);
    
    const questionsArr: { questionId: number; displayOrder: number; sectionId: number }[] = [];

    if (!Array.isArray(sections)) {
      return questionsArr;
    }

    for (const sec of sections) {
      const currentSection = await this.processSection(sec, user, repository, logger, transactionalEntityManager);
      
      // Process custom questions
      if (Array.isArray(sec.customQuestions)) {
        const customQuestionIds = await this.processCustomQuestions(
          sec.customQuestions,
          currentSection,
          user,
          repository,
          logger,
          transactionalEntityManager,
        );
        questionsArr.push(...customQuestionIds);
      }

      // Process existing questions
      if (Array.isArray(sec.questions)) {
        const existingQuestionIds = this.processExistingQuestions(sec.questions, currentSection.id);
        questionsArr.push(...existingQuestionIds);
      }
    }

    return questionsArr;
  }

  /**
   * Handles deletion of sections and questions as specified in the DTO
   * @param dto - DTO containing deletion instructions (sections at root, questions in sections)
   * @param user - Current user performing the operation
   * @param repository - Program question repository instance
   * @param logger - Logger service instance
   * @returns Promise<object> - Summary of deletion operations
   */
  private async processDeletions(
    dto: { 
      programId: number; 
      programSessionId?: number | null; 
      deletedSections?: number[];
      sections?: Array<{ deletedQuestions?: number[] }>;
    },
    user: User,
    repository: ProgramQuestionRepository,
    logger: AppLoggerService,
  ): Promise<{ deletedSections: number; deletedQuestions: number; details: any[] }> {
    logger.log('Processing deletions from update request');
    
    const results: any[] = [];
    let totalDeletedSections = 0;
    let totalDeletedQuestions = 0;

    // Process deleted sections (from root level)
    if (dto.deletedSections && dto.deletedSections.length > 0) {
      for (const sectionId of dto.deletedSections) {
        const sectionToDelete = await repository.findFormSectionById(sectionId);
        if (sectionToDelete) {
          // Find and soft delete all program questions in this section
          const programQuestionsInSection = await repository.findProgramQuestionsInSection(
            dto.programId,
            dto.programSessionId || null,
            sectionId,
          );

          if (programQuestionsInSection.length > 0) {
            const updatedMappings = programQuestionsInSection.map(pq => {
              pq.deletedAt = new Date();
              pq.updatedBy = user;
              return pq;
            });
            await repository.updateProgramQuestions(updatedMappings);
            totalDeletedQuestions += programQuestionsInSection.length;
          }

          // Find and soft delete all questions in this section
          const questionsInSection = await repository.findQuestionsInSection(sectionId);
          if (questionsInSection.length > 0) {
            const updatedQuestions = questionsInSection.map(question => {
              question.deletedAt = new Date();
              question.updatedBy = user;
              return question;
            });
            await repository.softDeleteQuestions(updatedQuestions);
          }

          // Soft delete the section
          sectionToDelete.deletedAt = new Date();
          sectionToDelete.updatedBy = user.id;
          await repository.softDeleteFormSection(sectionToDelete);
          totalDeletedSections++;

          results.push({
            type: 'section',
            sectionId: sectionId,
            deletedQuestions: programQuestionsInSection.length,
          });
        }
      }
    }

    // Process deleted questions (from within sections)
    if (dto.sections) {
      for (const section of dto.sections) {
        if (section.deletedQuestions && section.deletedQuestions.length > 0) {
          const programQuestionsToDelete = await repository.findProgramQuestionsToDelete(
            dto.programId,
            dto.programSessionId || null,
            section.deletedQuestions,
          );

          if (programQuestionsToDelete.length > 0) {
            const updatedMappings = programQuestionsToDelete.map(pq => {
              pq.deletedAt = new Date();
              pq.updatedBy = user;
              return pq;
            });
            await repository.updateProgramQuestions(updatedMappings);
            totalDeletedQuestions += programQuestionsToDelete.length;

            results.push({
              type: 'questions',
              questionIds: section.deletedQuestions,
              deletedCount: programQuestionsToDelete.length,
            });
          }
        }
      }
    }

    return {
      deletedSections: totalDeletedSections,
      deletedQuestions: totalDeletedQuestions,
      details: results,
    };
  }
}