import { ApiProperty } from '@nestjs/swagger';
import { Type } from 'class-transformer';
import { IsInt, IsOptional, IsEnum, ValidateIf, ValidateNested, IsArray, IsString } from 'class-validator';
import { RegistrationLevelEnum } from 'src/common/enum/program.enums';
import { CustomQuestionDto, QuestionOrderDto } from './create-program-question.dto';

export class UpdateProgramQuestionDto {
  
  @IsOptional()
  @ApiProperty({ description: 'Program ID', required: false })
  @IsInt()
  programId?: number;

  @IsOptional()
  @ApiProperty({ description: 'Question ID', required: false })
  @IsInt()
  questionId?: number;

  @IsOptional()
  @ApiProperty({ 
    description: 'Registration Level - program or session', 
    enum: RegistrationLevelEnum,
    required: false 
  })
  @IsEnum(RegistrationLevelEnum)
  registrationLevel?: RegistrationLevelEnum;

  @IsOptional()
  @ApiProperty({ 
    description: 'Program Session ID - required when registration level is session',
    required: false 
  })
  @IsInt()
  @ValidateIf(o => o.registrationLevel === RegistrationLevelEnum.SESSION)
  programSessionId?: number;

  @IsOptional()
  @ApiProperty({ 
    description: 'Display order for this question',
    required: false 
  })
  @IsInt()
  displayOrder?: number;
}
export class UpdateSectionDto {
  @ApiProperty({ description: 'Existing section ID - if provided, will update existing section', required: false })
  @IsOptional()
  @IsInt()
  sectionId?: number;

  @ApiProperty({ description: 'Name of the form section' })
  @IsString()
  sectionName: string;

  @ApiProperty({ type: [QuestionOrderDto], required: false })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => QuestionOrderDto)
  questions?: QuestionOrderDto[];

  @ApiProperty({ type: [CustomQuestionDto], required: false })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => CustomQuestionDto)
  customQuestions?: CustomQuestionDto[];

  @ApiProperty({ 
    description: 'Array of Question IDs to delete from this section scope',
    type: [Number],
    required: false 
  })
  @IsOptional()
  @IsArray()
  @Type(() => Number)
  @IsInt({ each: true })
  deletedQuestions?: number[];

  @ApiProperty({ required: false })
  @IsOptional()
  @IsInt()
  sectionDisplayOrder?: number;
}

export class UpdateProgramQuestionBulkDto {
  @ApiProperty({ description: 'Program ID' })
  @IsInt()
  programId: number;

  @ApiProperty({ type: [UpdateSectionDto] })
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => UpdateSectionDto)
  sections: UpdateSectionDto[];

  @ApiProperty({ 
    description: 'Registration Level - program or session', 
    enum: RegistrationLevelEnum,
    default: RegistrationLevelEnum.PROGRAM
  })
  @IsEnum(RegistrationLevelEnum)
  registrationLevel: RegistrationLevelEnum;

  @ApiProperty({ 
    description: 'Program Session ID - required when registration level is session',
    required: false 
  })
  @IsOptional()
  @IsInt()
  @ValidateIf(o => o.registrationLevel === RegistrationLevelEnum.SESSION)
  programSessionId?: number;

  @ApiProperty({ 
    description: 'Array of Section IDs to delete entirely (with all their questions)',
    type: [Number],
    required: false 
  })
  @IsOptional()
  @IsArray()
  @Type(() => Number)
  @IsInt({ each: true })
  deletedSections?: number[];
}
