import { ApiProperty } from '@nestjs/swagger';
import {
  IsInt,
  IsEnum,
  IsOptional,
  ValidateIf,
  IsArray,
  ValidateNested,
  IsString
} from 'class-validator';
import { Type } from 'class-transformer';
import { RegistrationLevelEnum } from 'src/common/enum/program.enums';

export class QuestionOrderDto {
  @ApiProperty({ description: 'Question ID' })
  @IsInt()
  questionId: number;

  @ApiProperty({ description: 'Display order for this question', required: false })
  @IsOptional()
  @IsInt()
  displayOrder?: number;
}

export class CustomQuestionOptionDto {
  @ApiProperty({ description: 'Option name' })
  @IsString()
  name: string;

  @ApiProperty({ description: 'Option type' })
  @IsString()
  type: string;

  @ApiProperty({ description: 'Option category ID or null', required: false })
  @IsOptional()
  @IsInt()
  category?: number | null;
}

export class CustomQuestionInfoDto {
  @ApiProperty()
  @IsString()
  label: string;

  @ApiProperty()
  @IsString()
  type: string;

  @ApiProperty({ required: false })
  @IsOptional()
  @IsInt()
  section?: number | null;

  @ApiProperty({ required: false })
  @IsOptional()
  config?: Record<string, any>;

  @ApiProperty({ required: false })
  @IsOptional()
  status?: string;

  @ApiProperty({ required: false })
  @IsOptional()
  answerLocation?: string;

  @ApiProperty({ required: false })
  @IsOptional()
  bindingKey?: string;
}

export class CustomQuestionDto {
  @ApiProperty({ type: CustomQuestionInfoDto })
  @ValidateNested()
  @Type(() => CustomQuestionInfoDto)
  question: CustomQuestionInfoDto;

  @ApiProperty({ type: [CustomQuestionOptionDto], required: false })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => CustomQuestionOptionDto)
  options?: CustomQuestionOptionDto[];

  @ApiProperty({ description: 'Display order for this custom question', required: false })
  @IsOptional()
  @IsInt()
  displayOrder?: number;
}

export class SectionDto {
  @ApiProperty({ description: 'Existing section ID - if provided, will use existing section', required: false })
  @IsOptional()
  @IsInt()
  sectionId?: number;

  @ApiProperty({ description: 'Name of the form section' })
  @IsString()
  sectionName: string;

  @ApiProperty({ type: [QuestionOrderDto], required: false })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => QuestionOrderDto)
  questions?: QuestionOrderDto[];

  @ApiProperty({ type: [CustomQuestionDto], required: false })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => CustomQuestionDto)
  customQuestions?: CustomQuestionDto[];

  @ApiProperty({ required: false })
  @IsOptional()
  @IsInt()
  sectionDisplayOrder?: number;
}

export class CreateProgramQuestionDto {
  @ApiProperty({ description: 'Program ID' })
  @IsInt()
  programId: number;

  @ApiProperty({ type: [SectionDto] })
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => SectionDto)
  sections: SectionDto[];

  @ApiProperty({ 
    description: 'Registration Level - program or session', 
    enum: RegistrationLevelEnum,
    default: RegistrationLevelEnum.PROGRAM
  })
  @IsEnum(RegistrationLevelEnum)
  registrationLevel: RegistrationLevelEnum;

  @ApiProperty({ 
    description: 'Program Session ID - required when registration level is session',
    required: false 
  })
  @IsOptional()
  @IsInt()
  @ValidateIf(o => o.registrationLevel === RegistrationLevelEnum.SESSION)
  programSessionId?: number;
}