import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, EntityManager } from 'typeorm';
import { CreatePreferenceDto } from './dto/create-preference.dto';
import { UpdatePreferenceDto } from './dto/update-preference.dto';
import { PreferenceRepository } from './preference.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { preferenceMessages } from 'src/common/constants/strings-constants';
import {
  Preference,
  Program,
  ProgramRegistration,
  ProgramSession,
  User,
} from 'src/common/entities';
import { RegistrationLevelEnum } from 'src/common/enum/registration-level.enum';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { CommonDataService } from 'src/common/services/commonData.service';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';

@Injectable()
export class PreferenceService {
  constructor(
    private readonly repo: PreferenceRepository,
    private readonly logger: AppLoggerService,
    @InjectRepository(User) private readonly userRepo: Repository<User>,
    @InjectRepository(Program) private readonly programRepo: Repository<Program>,
    @InjectRepository(ProgramSession) private readonly sessionRepo: Repository<ProgramSession>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepo: Repository<ProgramRegistration>,
    private readonly commonDataService: CommonDataService,
  ) {}

  async create(dto: CreatePreferenceDto): Promise<Preference[]> {
    this.logger.log(preferenceMessages.CREATE_REQUEST_RECEIVED, dto);

    try {
      // Validate business rules
      await this.validatePreferenceCreation(dto);
      await this.validateEntityReferences(
        dto.programId,
        dto.preferences.map((p) => p.sessionId),
        dto.registrationId,
        dto.createdBy,
      );

      const program = await this.programRepo.findOne({
        where: { id: dto.programId },
        relations: ['type'],
      });

      const registrationLevel = program?.type?.registrationLevel;

      const createdPreferences: Preference[] = [];
      for (const preferenceItem of dto.preferences) {
        const preferredProgramId =
          registrationLevel === RegistrationLevelEnum.PROGRAM
            ? preferenceItem.sessionId
            : undefined;
        const preferredSessionId =
          registrationLevel !== RegistrationLevelEnum.PROGRAM
            ? preferenceItem.sessionId
            : undefined;

        const preference = await this.repo.createPreference(
          dto.programId,
          dto.registrationId,
          preferenceItem.priorityOrder ?? 0,
          dto.createdBy,
          dto.updatedBy,
          undefined,
          preferredProgramId,
          preferredSessionId,
        );
        createdPreferences.push(preference);
      }

      return createdPreferences;
    } catch (error) {
      this.logger.error('Error creating preferences', error);
      handleKnownErrors(ERROR_CODES.PREFERENCE_SAVE_FAILED, error);
    }
  }

  async findByRegistration(registrationId: number): Promise<Preference[]> {
    this.logger.log(preferenceMessages.FIND_ALL_REQUEST_RECEIVED, { registrationId });

    try {
      return await this.repo.findByRegistrationId(registrationId);
    } catch (error) {
      this.logger.error('Error finding preferences by registration', error);
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  /**
   * Enhanced update method with better transaction support and validation
   * This method can be called from within a transaction (via manager) or standalone
   */
  async update(
    registrationId: number,
    dto: UpdatePreferenceDto,
    manager?: EntityManager,
  ): Promise<Preference[]> {
    this.logger.log(preferenceMessages.UPDATE_REQUEST_RECEIVED, { registrationId, dto });

    try {
      // Enhanced validation for transaction context
      await this.validatePreferenceUpdate(registrationId, dto, manager);
      const program = await (manager
        ? manager.getRepository(Program).findOne({ where: { id: dto.programId }, relations: ['type'] })
        : this.programRepo.findOne({ where: { id: dto.programId }, relations: ['type'] }));

      const registrationLevel = program?.type?.registrationLevel;

      // Validate entity references with optional manager context
      await this.validateEntityReferences(
        dto.programId,
        dto.preferences.map((p) => p.sessionId),
        registrationId,
        dto.updatedBy,
        manager,
      );

      // Soft delete existing preferences for this registration and program
      await this.repo.softDeleteByRegistrationAndProgram(
        registrationId,
        dto.programId,
        dto.updatedBy,
        manager,
      );

      
      const updatedPreferences: Preference[] = [];

      if (dto.preferences && dto.preferences.length > 0) {
        for (const preferenceItem of dto.preferences) {
          const preferredProgramId =
            registrationLevel === RegistrationLevelEnum.PROGRAM
              ? preferenceItem.sessionId
              : undefined;
          const preferredSessionId =
            registrationLevel !== RegistrationLevelEnum.PROGRAM
              ? preferenceItem.sessionId
              : undefined;

          const preference = await this.repo.createPreference(
            dto.programId,
            registrationId,
            preferenceItem.priorityOrder ?? 0,
            dto.createdBy || dto.updatedBy,
            dto.updatedBy,
            undefined,
            preferredProgramId,
            preferredSessionId,
            manager,
          );
          updatedPreferences.push(preference);
        }
      }

      this.logger.log('Successfully updated preferences', {
        registrationId,
        programId: dto.programId,
        oldPreferencesDeleted: true,
        newPreferencesCreated: updatedPreferences.length,
        transactionMode: !!manager
      });

      return updatedPreferences;
    } catch (error) {
      this.logger.error('Error updating preferences', error);
      handleKnownErrors(ERROR_CODES.PREFERENCE_SAVE_FAILED, error);
    }
  }

  async remove(registrationId: number, userId: number): Promise<void> {
    this.logger.log(preferenceMessages.DELETE_REQUEST_RECEIVED, { registrationId });

    try {
      // Validate business rules
      await this.validatePreferenceDeletion(registrationId, userId);
      await this.validateEntityReferences(null, [], registrationId, userId);

      await this.repo.softDeleteByRegistrationId(registrationId, userId);
    } catch (error) {
      this.logger.error('Error deleting preferences', error);
      handleKnownErrors(ERROR_CODES.PREFERENCE_DELETE_FAILED, error);
    }
  }

  async findByRegistrationAndProgram(
    registrationId: number,
    programId: number,
  ): Promise<Preference[]> {
    this.logger.log('Finding preferences by registration and program', {
      registrationId,
      programId,
    });

    try {
      return await this.repo.findByRegistrationAndProgram(registrationId, programId);
    } catch (error) {
      this.logger.error('Error finding preferences by registration and program', error);
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  // Enhanced Business validation methods with transaction support

  private async validatePreferenceCreation(dto: CreatePreferenceDto): Promise<void> {
    // Check if preferences already exist for this registration and program
    const existingPreferences = await this.repo.findByRegistrationAndProgram(
      dto.registrationId,
      dto.programId,
    );

    if (existingPreferences.length > 0) {
      throw new InifniBadRequestException(
        ERROR_CODES.PREFERENCE_ALREADY_EXISTS,
        null,
        null,
        'Preferences already exist for this registration and program',
      );
    }

    // Validate priority order uniqueness
    this.validatePriorityOrderUniqueness(dto.preferences.map(p => ({ priorityOrder: p.priorityOrder })));
  }

  private async validatePreferenceUpdate(
    registrationId: number,
    dto: UpdatePreferenceDto,
    manager?: EntityManager,
  ): Promise<void> {
    // Validate priority order uniqueness
    this.validatePriorityOrderUniqueness(dto.preferences);

    // Additional validation: check if registration exists (with transaction support)
    let registration;
    if (manager) {
      registration = await manager.findOne(ProgramRegistration, {
        where: { id: registrationId }
      });
    } else {
      registration = await this.commonDataService.findOneById(
        this.registrationRepo,
        registrationId,
        true,
      );
    }

    if (!registration) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        registrationId.toString(),
      );
    }
  }

  private async validatePreferenceDeletion(registrationId: number, userId: number): Promise<void> {
    // Check if preferences exist
    const existingPreferences = await this.repo.findByRegistrationId(registrationId);

    if (existingPreferences.length === 0) {
      throw new InifniBadRequestException(
        ERROR_CODES.PREFERENCE_NOTFOUND,
        null,
        null,
        'No preferences found for the given registration ID.',
      );
    }
  }

  /**
   * Enhanced entity reference validation with transaction support
   */
  private async validateEntityReferences(
    programId: number | null,
    sessionIds: number[],
    registrationId: number,
    userId: number,
    manager?: EntityManager,
  ): Promise<void> {
    
    // Validate user existence
    let user;
    if (manager) {
      user = await manager.findOne(User, { where: { id: userId } });
    } else {
      user = await this.commonDataService.findOneById(this.userRepo, userId, false);
    }
    
    if (!user) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
    }

    let registrationLevel: RegistrationLevelEnum | undefined = "program" as RegistrationLevelEnum;
    // Validate program existence
    if (programId) {
      let program;
      if (manager) {
        program = await manager.findOne(Program, { where: { id: programId } });
      } else {
        program = await this.commonDataService.findOneById(this.programRepo, programId, true);
      }
      if (program) {
        registrationLevel = program.type?.registrationLevel;
      }
      
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }
    }

    // Validate registration existence
    let registration;
    if (manager) {
      registration = await manager.findOne(ProgramRegistration, { where: { id: registrationId } });
    } else {
      registration = await this.commonDataService.findOneById(
        this.registrationRepo,
        registrationId,
        true,
      );
    }
    
    if (!registration) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        registrationId.toString(),
      );
    }

    if (registrationLevel === RegistrationLevelEnum.PROGRAM) {
      // Validate program existence for each session
      for (const sessionId of sessionIds) {
        let program;
        if (manager) {
          program = await manager.findOne(Program, { where: { id: sessionId } });
        } else {
          program = await this.commonDataService.findOneById(this.programRepo, sessionId, true);
        }
        
        if (!program) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            null,
            sessionId.toString(),
          );
        }
      }

    } else if (registrationLevel === RegistrationLevelEnum.SESSION) {
      // Validate session existence for each session
      for (const sessionId of sessionIds) {
        let session;
        if (manager) {
          session = await manager.findOne(Program, { where: { id: sessionId } });
        } else {
          session = await this.commonDataService.findOneById(this.sessionRepo, sessionId, true);
        }
        
        if (!session) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
            null,
            null,
            sessionId.toString(),
          );
        }
      }
    }
  }

  /**
   * Helper method to validate priority order uniqueness
   */
  private validatePriorityOrderUniqueness(preferences: { priorityOrder?: number }[]): void {
    const priorityOrders = preferences
      .map((p) => p.priorityOrder)
      .filter((p) => p !== undefined);
    const uniquePriorityOrders = new Set(priorityOrders);

    if (priorityOrders.length !== uniquePriorityOrders.size) {
      throw new InifniBadRequestException(
        ERROR_CODES.PREFERENCE_PRIORITY_ORDER_NOT_UNIQUE,
        null,
        null,
        'Priority orders must be unique within preferences',
      );
    }
  }
}