import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { IsNull, Repository, EntityManager } from 'typeorm';
import { Preference } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';

@Injectable()
export class PreferenceRepository {
  constructor(
    @InjectRepository(Preference)
    private readonly repo: Repository<Preference>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Create a single preference with optional transaction manager support
   */
  async createPreference(
    programId: number,
    registrationId: number,
    priorityOrder: number,
    createdBy: number,
    updatedBy: number,
    sessionId?: number,
    preferredProgramId?: number,
    preferredSessionId?: number,
    manager?: EntityManager,
  ): Promise<Preference> {
    try {
      const repository = manager ? manager.getRepository(Preference) : this.repo;
      
      // Use only the relation id objects as per entity definition
      const entity = repository.create({
        program: { id: programId },
        session: { id: sessionId },
        preferredProgram: preferredProgramId ? { id: preferredProgramId } : undefined,
        preferredSession: preferredSessionId ? { id: preferredSessionId } : undefined,
        registration: { id: registrationId },
        priorityOrder: priorityOrder ?? 0,
        createdBy: { id: createdBy },
        updatedBy: { id: updatedBy },
      });

      this.logger.log('Preference entity before save', { 
        entity: {
          programId,
          sessionId,
          preferredProgramId,
          preferredSessionId,
          registrationId,
          priorityOrder,
          createdBy,
          updatedBy
        },
        transactionMode: !!manager
      });

      // Use the appropriate save method based on whether we have a manager
      if (manager) {
        return await repository.save(entity);
      } else {
        return await this.commonDataService.save(repository, entity);
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_SAVE_FAILED, error);
    }
  }

  /**
   * Create multiple preferences with optional transaction manager support
   */
  async createMultiplePreferences(
    preferences: Preference[],
    manager?: EntityManager,
  ): Promise<Preference[]> {
    try {
      const repository = manager ? manager.getRepository(Preference) : this.repo;
      
      this.logger.log('Creating multiple preferences', {
        count: preferences.length,
        transactionMode: !!manager
      });
      
      return await repository.save(preferences);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_SAVE_FAILED, error);
    }
  }

  /**
   * Find preferences by registration ID
   */
  async findByRegistrationId(registrationId: number): Promise<Preference[]> {
    try {
      return await this.commonDataService.findByData(
        this.repo,
        { registration: { id: registrationId }, deletedAt: IsNull() } as any,
        ['program', 'session', 'preferredProgram', 'preferredSession'],
        { priorityOrder: 'ASC', id: 'ASC' },
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  /**
   * Find preferences by registration and program
   */
  async findByRegistrationAndProgram(
    registrationId: number,
    programId: number,
  ): Promise<Preference[]> {
    try {
      return await this.commonDataService.findByData(
        this.repo,
        {
          registration: { id: registrationId },
          program: { id: programId },
          deletedAt: IsNull(),
        } as any,
        ['program', 'session', 'preferredProgram', 'preferredSession'],
        { priorityOrder: 'ASC', id: 'ASC' },
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  /**
   * Find active preferences by registration
   */
  async findActivePreferencesByRegistration(registrationId: number): Promise<Preference[]> {
    try {
      return await this.commonDataService.findByData(
        this.repo,
        { registration: { id: registrationId }, deletedAt: IsNull() } as any,
        ['program', 'session', 'preferredProgram', 'preferredSession'],
        { priorityOrder: 'ASC', id: 'ASC' },
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  /**
   * Soft delete preferences by registration ID with enhanced transaction support
   */
  async softDeleteByRegistrationId(
    registrationId: number,
    userId: number,
    manager?: EntityManager,
  ): Promise<void> {
    try {
      this.logger.log('Soft deleting preferences by registration ID', {
        registrationId,
        userId,
        transactionMode: !!manager
      });

      const preferences = await this.findActivePreferencesByRegistration(registrationId);

      if (preferences.length > 0) {
        const updatedPreferences = preferences.map((preference) => ({
          ...preference,
          deletedAt: new Date(),
          updatedBy: { id: userId } as any,
          auditRefId: preference.id,
          parentRefId: registrationId, // Use registrationId for parentRefId
        }));

        await this.createMultiplePreferences(updatedPreferences, manager);
        
        this.logger.log('Successfully soft deleted preferences', {
          registrationId,
          deletedCount: preferences.length,
          transactionMode: !!manager
        });
      } else {
        this.logger.log('No active preferences found to delete', {
          registrationId,
          transactionMode: !!manager
        });
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_DELETE_FAILED, error);
    }
  }

  /**
   * Soft delete preferences by registration and program with enhanced transaction support
   */
  async softDeleteByRegistrationAndProgram(
    registrationId: number,
    programId: number,
    userId: number,
    manager?: EntityManager,
  ): Promise<void> {
    try {
      this.logger.log('Soft deleting preferences by registration and program', {
        registrationId,
        programId,
        userId,
        transactionMode: !!manager
      });

      const preferences = await this.findByRegistrationAndProgram(registrationId, programId);

      if (preferences.length > 0) {
        const updatedPreferences = preferences.map((preference) => ({
          ...preference,
          deletedAt: new Date(),
          updatedBy: { id: userId } as any,
          auditRefId: preference.id,
          parentRefId: registrationId,
        }));

        await this.createMultiplePreferences(updatedPreferences, manager);
        
        this.logger.log('Successfully soft deleted preferences by registration and program', {
          registrationId,
          programId,
          deletedCount: preferences.length,
          transactionMode: !!manager
        });
      } else {
        this.logger.log('No preferences found to delete for registration and program', {
          registrationId,
          programId,
          transactionMode: !!manager
        });
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_DELETE_FAILED, error);
    }
  }

  /**
   * Find preference by ID
   */
  async findById(id: number): Promise<Preference | null> {
    try {
      return await this.commonDataService.findOneById(this.repo, id, true, [
        'program',
        'session',
        'preferredProgram',
        'preferredSession',
      ]);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  /**
   * Find preferences by registration with transaction manager support
   * This method is useful when you need to query preferences within a transaction
   */
  async findByRegistrationIdWithManager(
    registrationId: number,
    manager: EntityManager,
  ): Promise<Preference[]> {
    try {
      const repository = manager.getRepository(Preference);
      
      return await repository.find({
        where: { 
          registration: { id: registrationId }, 
          deletedAt: IsNull() 
        },
        relations: ['program', 'session', 'preferredProgram', 'preferredSession'],
        order: { priorityOrder: 'ASC', id: 'ASC' },
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  /**
   * Find preferences by registration and program with transaction manager support
   */
  async findByRegistrationAndProgramWithManager(
    registrationId: number,
    programId: number,
    manager: EntityManager,
  ): Promise<Preference[]> {
    try {
      const repository = manager.getRepository(Preference);
      
      return await repository.find({
        where: {
          registration: { id: registrationId },
          program: { id: programId },
          deletedAt: IsNull(),
        },
        relations: ['program', 'session', 'preferredProgram', 'preferredSession'],
        order: { priorityOrder: 'ASC', id: 'ASC' },
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  /**
   * Bulk create preferences with transaction support
   * Useful for creating multiple preferences efficiently
   */
  async bulkCreatePreferences(
    preferenceData: {
      programId: number;
      sessionId: number;
      registrationId: number;
      priorityOrder: number;
      createdBy: number;
      updatedBy: number;
      preferredProgramId?: number;
      preferredSessionId?: number;
    }[],
    manager?: EntityManager,
  ): Promise<Preference[]> {
    try {
      const repository = manager ? manager.getRepository(Preference) : this.repo;
      
      const preferences = preferenceData.map(data => repository.create({
        program: { id: data.programId },
        session: { id: data.sessionId },
        preferredProgram: data.preferredProgramId ? { id: data.preferredProgramId } : undefined,
        preferredSession: data.preferredSessionId ? { id: data.preferredSessionId } : undefined,
        registration: { id: data.registrationId },
        priorityOrder: data.priorityOrder ?? 0,
        createdBy: { id: data.createdBy },
        updatedBy: { id: data.updatedBy },
      }));

      this.logger.log('Bulk creating preferences', {
        count: preferences.length,
        transactionMode: !!manager
      });

      return await repository.save(preferences);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_SAVE_FAILED, error);
    }
  }

  /**
   * Update preference priority order with transaction support
   */
  async updatePreferencePriority(
    preferenceId: number,
    newPriorityOrder: number,
    updatedBy: number,
    manager?: EntityManager,
  ): Promise<Preference> {
    try {
      const repository = manager ? manager.getRepository(Preference) : this.repo;
      
      // Get the existing preference to extract registrationId for parentRefId
      const existingPreference = await repository.findOne({
        where: { id: preferenceId },
        relations: ['registration']
      });
      
      if (!existingPreference) {
        throw new Error(`Preference with id ${preferenceId} not found.`);
      }
      
      await repository.update(preferenceId, {
        priorityOrder: newPriorityOrder,
        updatedBy: { id: updatedBy } as any,
        auditRefId: preferenceId,
        parentRefId: existingPreference.registration.id, // Use registrationId for parentRefId
      });

      const updatedPreference = await repository.findOne({
        where: { id: preferenceId },
        relations: ['program', 'session', 'preferredProgram', 'preferredSession'],
      });

      this.logger.log('Updated preference priority order', {
        preferenceId,
        newPriorityOrder,
        updatedBy,
        transactionMode: !!manager
      });

      if (!updatedPreference) {
        throw new Error(`Preference with id ${preferenceId} not found after update.`);
      }

      return updatedPreference;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_SAVE_FAILED, error);
    }
  }

  /**
   * Count preferences by registration and program
   */
  async countByRegistrationAndProgram(
    registrationId: number,
    programId: number,
    manager?: EntityManager,
  ): Promise<number> {
    try {
      const repository = manager ? manager.getRepository(Preference) : this.repo;
      
      return await repository.count({
        where: {
          registration: { id: registrationId },
          program: { id: programId },
          deletedAt: IsNull(),
        },
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PREFERENCE_GET_FAILED, error);
    }
  }

  /**
   * Check if preferences exist for registration and program
   */
  async existsByRegistrationAndProgram(
    registrationId: number,
    programId: number,
    manager?: EntityManager,
  ): Promise<boolean> {
    try {
      const count = await this.countByRegistrationAndProgram(registrationId, programId, manager);
      return count > 0;
    } catch (error) {
      this.logger.error('Error checking if preferences exist', error);
      return false;
    }
  }
}