import { Body, Controller, Delete, Get, HttpCode, HttpStatus, Param, ParseIntPipe, Post, Put, Res, UseGuards, ValidationPipe, Req } from '@nestjs/common';
import { Response } from 'express';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiResponse, ApiSecurity, ApiTags } from '@nestjs/swagger';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { preferenceMessages, SWAGGER_API_RESPONSE, userConstMessages } from 'src/common/constants/strings-constants';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { PreferenceService } from './preference.service';
import { CreatePreferenceDto } from './dto/create-preference.dto';
import { UpdatePreferenceDto } from './dto/update-preference.dto';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('preference')
@Controller('preference')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class PreferenceController {
  constructor(
    private readonly service: PreferenceService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: preferenceMessages.CREATE_NEW })
  @ApiBody({ type: CreatePreferenceDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CreatePreferenceDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(preferenceMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      const userId = req.user?.id;
      if (!userId) {
        throw new InifniBadRequestException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          userConstMessages.USER_ID_NOT_FOUND,
        );
      }
      dto.createdBy = userId;
      dto.updatedBy = userId;
      const data = await this.service.create(dto);
      await this.responseService.success(
        res,
        preferenceMessages.PREFERENCES_CREATED,
        data,
        HttpStatus.CREATED,
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':registrationId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: preferenceMessages.GET_ALL })
  @ApiParam({ name: 'registrationId', type: Number, description: preferenceMessages.REGISTRATION_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findByRegistration(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(preferenceMessages.FIND_ALL_REQUEST_RECEIVED, { registrationId });
    try {
      const userId = req.user?.id;
      if (!userId) {
        throw new InifniBadRequestException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          userConstMessages.USER_ID_NOT_FOUND,
        );
      }
      const data = await this.service.findByRegistration(registrationId);
      await this.responseService.success(res, preferenceMessages.PREFERENCES_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':registrationId/program/:programId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get preferences by registration and program' })
  @ApiParam({ name: 'registrationId', type: Number, description: preferenceMessages.REGISTRATION_ID })
  @ApiParam({ name: 'programId', type: Number, description: 'Program ID' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findByRegistrationAndProgram(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Param('programId', ParseIntPipe) programId: number,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log('Find preferences by registration and program request received', { registrationId, programId });
    try {
      const userId = req.user?.id;
      if (!userId) {
        throw new InifniBadRequestException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          userConstMessages.USER_ID_NOT_FOUND,
        );
      }
      const data = await this.service.findByRegistrationAndProgram(registrationId, programId);
      await this.responseService.success(res, preferenceMessages.PREFERENCES_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put(':registrationId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: preferenceMessages.UPDATE_BY_REG_ID })
  @ApiParam({ name: 'registrationId', type: Number, description: preferenceMessages.REGISTRATION_ID })
  @ApiBody({ type: UpdatePreferenceDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdatePreferenceDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(preferenceMessages.UPDATE_REQUEST_RECEIVED, { registrationId, dto });
    try {
      const userId = req.user?.id;
      if (!userId) {
        throw new InifniBadRequestException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          userConstMessages.USER_ID_NOT_FOUND,
        );
      }
      dto.updatedBy = userId;
      dto.createdBy = userId;
      const data = await this.service.update(registrationId, dto);
      await this.responseService.success(res, preferenceMessages.PREFERENCES_UPDATED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Delete(':registrationId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: preferenceMessages.DELETE_BY_REG_ID })
  @ApiParam({ name: 'registrationId', type: Number, description: preferenceMessages.REGISTRATION_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const userId = req.user?.id;
    this.logger.log(preferenceMessages.DELETE_REQUEST_RECEIVED, { registrationId });
    try {
      if (!userId) {
        throw new InifniBadRequestException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          userConstMessages.USER_ID_NOT_FOUND,
        );
      }
      await this.service.remove(registrationId, userId);
      await this.responseService.success(res, preferenceMessages.PREFERENCES_DELETED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}