import { Injectable } from '@nestjs/common';
import { InitiatePaymentDto } from './dto/create-payment.dto';
import { UpdatePaymentDto } from './dto/update-payment.dto';
import { PaymentRepository } from './payment.repository';
import { InvoiceService } from 'src/invoice/invoice.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { RegistrationService } from 'src/registration/registration.service';
import { CommunicationService } from 'src/communication/communication.service';
import { zeptoEmailCreadentials } from 'src/common/constants/strings-constants';
import { areGSTsFromSameState, formatDate, formatDateIST, generatePaymentLink } from 'src/common/utils/common.util';
import { CommonDataService } from 'src/common/services/commonData.service';
import { ProgramRegistrationRepository } from 'src/program-registration/program-registration.repository';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import { TdsApplicabilityEnum } from 'src/common/enum/tds-applicability.enum';
import { UserRepository } from 'src/user/user.repository';
import * as crypto from 'crypto';
import { ProgramRegistration, RegistrationPaymentDetail, RegistrationPaymentDetailsHistory } from 'src/common/entities';
import { RazorpayPaymentStatusEnum } from 'src/common/enum/razorpay-payment-status.enum';
import { awsConfig } from 'src/common/config/config';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { DataSource } from 'typeorm';
import { PaymentEditRequestRepository } from './repositories/payment-edit-request.repository';
import { PaymentEditRequestService } from './services/payment-edit-request.service';
import { CreatePaymentEditRequestDto } from './dto/payment-edit-request.dto';
import { PaymentEditRequestStatus } from 'src/common/enum/payment-edit-request-status.enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
const Razorpay = require('razorpay');
@Injectable()
export class PaymentService {
  private razorpay: InstanceType<typeof Razorpay>;

  constructor(
    private readonly paymentRepository: PaymentRepository,
    private readonly invoiceService: InvoiceService,
    private readonly logger: AppLoggerService,
    private readonly registrationService: RegistrationService,
    private readonly communicationService: CommunicationService,
    private readonly commonDataService: CommonDataService,
    private readonly registrationRepository: ProgramRegistrationRepository,
    private readonly userRepository: UserRepository,
    private readonly dataSource: DataSource,
    private readonly paymentEditRequestService: PaymentEditRequestService,
    private readonly paymentEditRequestRepository: PaymentEditRequestRepository,
  ) {
    this.razorpay = new Razorpay({
      key_id: process.env.RAZORPAY_KEY_ID,
      key_secret: process.env.RAZORPAY_SECRET_KEY,
    });
  }

  async initiatePayment(registrationId: number, dto: InitiatePaymentDto, userId: number) {
    this.logger.log('Initiating payment for registration', {
      registrationId,
      paymentMode: dto.paymentMode,
    });

    return await this.dataSource.transaction(async (manager) => {
      try {
        // Check if payment already exists for this registration
        const existingPayment = await this.paymentRepository.findByRegistrationId(registrationId);

        if (existingPayment?.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
            existingPayment?.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) {
              return {
                message: "Payment already completed",
                isPaymentCompleted: true
              };
        }

        if (existingPayment && existingPayment.paymentStatus === PaymentStatusEnum.OFFLINE_PENDING && dto.isSeeker == true) {
          const latestEditRequest = await this.paymentEditRequestService.getLatestRequestForPayment(
            existingPayment.id,
          );

          if (latestEditRequest) {
            if (
              latestEditRequest &&
              latestEditRequest.requestStatus !== PaymentEditRequestStatus.REQUESTED
            ) {
              throw new InifniBadRequestException(
                ERROR_CODES.PAYMENT_EDIT_REQUEST_NOT_FOUND,
                null,
                null,
                `Payment edit request not found for registration ${registrationId}`,
              );
            } else {
              await this.paymentEditRequestService.completeRequest(latestEditRequest.id, userId);
            }
          }
        }

        // Get registration details and program/session information
        const registrationDetails =
          await this.paymentRepository.getRegistrationWithProgramDetails(registrationId);
        if (!registrationDetails) {
          throw new InifniBadRequestException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
            null,
            null,
            `Registration ${registrationId} not found`,
          );
        }
        // Ensure program is allocated
        if (!registrationDetails?.allocatedProgram) {
          throw new InifniBadRequestException(
            ERROR_CODES.PROGRAM_NOT_ALLOCATED,
            null,
            null,
            `${registrationId}`,
          );
        }
        // Ensure registration is approved before allowing payment
        if (!registrationDetails?.approvals ||
          (registrationDetails?.approvals?.length > 0 &&
          registrationDetails?.approvals[0]?.approvalStatus !== ApprovalStatusEnum.APPROVED)
        ) {
          throw new InifniBadRequestException(
            ERROR_CODES.REGISTRATION_APPROVAL_NEEDED,
            null,
            null,
            `${registrationId}`,
          );
        }

        if (registrationDetails.isFreeSeat) {
          throw new InifniBadRequestException(
            ERROR_CODES.PAYMENT_NOT_ALLOWED_FOR_FREE_SEAT,
            null,
            null,
           registrationId.toString(),
          );
        }

        // Calculate payment amounts from program/session
        const paymentCalculation = await this.calculatePaymentAmounts(registrationDetails, dto.gstNumber ?? null, dto.tanNumber ? dto.tdsAmount ?? 0 : 0);

        // Check seat availability and waitlist status
        let seatAvailability;
        if (!registrationDetails.program.requiresApproval) {
          seatAvailability = await this.registrationService.checkSeatAvailability(
            registrationDetails.program,
            registrationDetails.programSession,
            registrationDetails.program.type.registrationLevel,
          );
          
        } else {
          // For programs requiring approval, set default seat availability
          // This ensures waitlisting logic doesn't interfere with approval process
          seatAvailability = {
            totalSeats: registrationDetails.program.totalSeats,
            waitlistTriggerCount: registrationDetails.program.waitlistTriggerCount,
            shouldWaitlist: false,
            isWaitlistTriggered: false,
            canRegister: true,
          };
        }

        let result;
        if (dto.paymentMode === PaymentModeEnum.ONLINE) {
          result = await this.initiateOnlinePayment(
            registrationId,
            dto,
            userId,
            paymentCalculation,
            seatAvailability,
            registrationDetails,
            existingPayment,
            manager
          );
        } else {
          result = await this.initiateOfflinePayment(
            registrationId,
            dto,
            userId,
            paymentCalculation,
            seatAvailability,
            registrationDetails,
            manager,
          );
        }

        this.logger.log('Payment initiated successfully', {
          registrationId,
          paymentId: result.paymentId,
        });
        return result;
      } catch (error) {
        this.logger.error('Payment initiation failed', 'failed payment', {
          registrationId,
          error: error.message,
        });
        throw error; // Re-throw to trigger transaction rollback
      }
    });
  }

  async updatePayment(registrationId: number, dto: UpdatePaymentDto, userId: number) {
    this.logger.log('Updating payment for registration', {
      registrationId,
      newStatus: dto.paymentStatus,
    });

    try {
      // Get current payment details first
      const currentPayment = await this.paymentRepository.findPaymentDetailsByRegistrationId(registrationId);
      if (!currentPayment) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_NOTFOUND,
          null,
          null,
          `No payment found for registration ${registrationId}`
        );
      }

      // Check if payment is already completed
      if (currentPayment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED || 
          currentPayment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_ALREADY_COMPLETED,
          null,
          null,
          registrationId.toString(),
          currentPayment.paymentStatus
        );
      }
      // Ensure program is allocated
      if (!currentPayment?.registration?.allocatedProgram) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_NOT_ALLOCATED,
          null,
          null,
          `${registrationId}`,
        );
      }
      // Ensure registration is approved before allowing payment updates
      if (!currentPayment?.registration?.approvals ||
        (currentPayment?.registration?.approvals?.length > 0 &&
          currentPayment?.registration?.approvals[0]?.approvalStatus !==
            ApprovalStatusEnum.APPROVED)
      ) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_APPROVAL_NEEDED,
          null,
          null,
          `${registrationId}`,
        );
      }

      const result = await this.paymentRepository.updatePaymentStatus(registrationId, dto, userId);

      // Handle registration status updates based on payment status
      if (dto.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED || 
          dto.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) {
        await this.handlePaymentConfirmation(registrationId, userId);
      }

      this.logger.log('Payment updated successfully', {
        registrationId,
        paymentId: result.paymentId,
      });
      
      return result;
    } catch (error) {
      this.logger.error('Payment update failed', 'failed payment', {
        registrationId,
        error: error.message,
      });
      handleKnownErrors(ERROR_CODES.PAYMENT_UPDATE_FAILED, error);
    }
  }

  async getPaymentDetails(registrationId: number) {
    this.logger.log('Retrieving payment details', { registrationId });

    try {
      const paymentDetails =
        await this.paymentRepository.getPaymentDetailsWithInvoice(registrationId);
      if (!paymentDetails) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_NOTFOUND,
          null,
          null,
          `No payment found for registration ${registrationId}`,
        );
      }

      return paymentDetails;
    } catch (error) {
      this.logger.error('Failed to retrieve payment details', '', {
        registrationId,
        error: error.message,
      });
      handleKnownErrors(ERROR_CODES.PAYMENT_GET_FAILED, error);
    }
  }

  async handleWebhook(webhookData: any, headers: any, isFromPortal: boolean = false) {
    this.logger.log('Processing payment webhook', { event: webhookData.event });

    return await this.dataSource.transaction(async (manager) => {
      try {
        // Verify webhook signature (implement based on your payment gateway)
        const razorpayOrderId = webhookData.payload?.payment?.entity?.order_id;
        const paymentId = webhookData.payload?.payment?.entity?.id;
        const razorpaySignature = isFromPortal ? webhookData.payload?.payment?.entity?.signature : headers['x-razorpay-signature'];

        webhookData.date = new Date();
        if (!isFromPortal || ((webhookData.event === 'payment.captured') && isFromPortal)) {
          const isValidSignature = this.verifyPaymentSignature(razorpayOrderId, paymentId, razorpaySignature);
          if (!isValidSignature && isFromPortal) {
            throw new InifniBadRequestException(
              ERROR_CODES.INVALID_WEBHOOK_SIGNATURE,
              null,
              null,
              'Invalid payment signature'
            );
          }
        }
        try {
          await this.updatePaymentHistory(webhookData, isFromPortal, razorpayOrderId);
        } catch (error) {
          this.logger.error('Failed to update payment status with webhook', '', {
            error: error.message,
            webhookData,
          });
        }
        const payment = await this.paymentRepository.getPaymentDetailsByOrderId(razorpayOrderId);
        this.logger.log('Updating payment with webhook data', { event: webhookData.event, razorpayOrderId, paymentId });
        const result = await this.updatePaymentWithWebhook(webhookData, isFromPortal, manager);
        this.logger.log('Webhook processed successfully', { event: webhookData.event, result });

        if (razorpayOrderId) {
          if (
            payment &&
              (payment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
               payment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED)
          ) {
            this.logger.log('Payment already completed, skipping further processing', { registrationId: payment.registrationId, paymentStatus: payment.paymentStatus });
            return result;
          } else {
            // Handle additional processing if payment is successful
            if (
              (webhookData.event === 'payment.captured' || webhookData.event === 'payment.authorized' || webhookData.event === 'order.paid') &&
              result.registrationId
            ) {
              this.logger.log('Handling post-payment confirmation tasks', { registrationId: result.registrationId });
              await this.handlePaymentConfirmation(result.registrationId, null);
            } else {
              this.logger.log('No post-payment actions required for this event', { event: webhookData.event });
            }
          }
        } else {
          this.logger.log('No valid payment found');
        }

        return result;
      } catch (error) {
        this.logger.error('Webhook processing failed', '', { error: error.message });
        throw error; // Re-throw to trigger transaction rollback
      }
    });
  }
  async updatePaymentWithWebhook(webhookData: any, isFromPortal: boolean = false, manager?: any) {
    const razorpayOrderId = webhookData.payload?.payment?.entity?.order_id;
    const paymentId = webhookData.payload?.payment?.entity?.id;
    const status = webhookData.payload?.payment?.entity?.status;
    const amount = webhookData.payload?.payment?.entity?.amount;

    if (!razorpayOrderId) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_WEBHOOK_DATA,
        null,
        null,
        'Razorpay order ID not found in webhook'
      );
    }

    try {
      const payment = await this.paymentRepository.getPaymentDetailsByOrderId(razorpayOrderId);

      if (payment) {
        let gateWayDetails = {
          paymentStatus: payment.paymentStatus,
          paymentDetails: payment.gatewayOnlinePaymentAttemptStatus,
          razorPaymentStatus: payment.gatewayOnlinePaymentStatus,
          authorizedPaymentId: '',
          authorisedPaymentAmount: 0,
        };
        // const paymentStatus = this.get
        if (isFromPortal){
          const gatewayPaymentStatus = await this.getLatestPaymentStatusOfOrderId(razorpayOrderId);
          gateWayDetails = {
            razorPaymentStatus: gatewayPaymentStatus.razorPaymentStatus,
            paymentDetails: gatewayPaymentStatus.paymentDetails,
            paymentStatus: gatewayPaymentStatus.paymentStatus ?? PaymentStatusEnum.DRAFT,
            authorizedPaymentId: gatewayPaymentStatus.gateWayPaidPaymentDetails?.id,
            authorisedPaymentAmount: gatewayPaymentStatus.gateWayPaidPaymentDetails?.amount,
          };
        }
        let newPaymentStatus: PaymentStatusEnum;
        if (status === 'captured' || status === 'authorized') {
          newPaymentStatus = PaymentStatusEnum.ONLINE_COMPLETED;
          // Update registration status based on payment
          await this.paymentRepository.updateRegistrationAfterPayment(payment.registrationId, null, manager);
        } else if (status === 'failed' || status === 'dismissed') {
          newPaymentStatus = PaymentStatusEnum.FAILED; // Keep pending for retry
        } else {
          newPaymentStatus = payment.paymentStatus; // No change
        }
        if (
          (newPaymentStatus !== PaymentStatusEnum.ONLINE_COMPLETED &&
            (payment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
            payment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED))
        ) {
           return {
            message: "Payment already completed"
          };
          // handleKnownErrors(ERROR_CODES.PAYMENT_ALREADY_COMPLETED, new Error('Payment already completed'));
        }

        await this.paymentRepository.updatePayment(payment.id, {
          paymentStatus: ((payment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED) || (payment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED)) ? payment.paymentStatus : newPaymentStatus,
          paymentDate:
            status === 'captured' || status === 'authorized'
              ? payment?.paymentDate
                ? payment?.paymentDate
                : new Date()
              : undefined,
          offlineMeta: {
            ...payment.offlineMeta,
            razorpayPaymentId: paymentId,
            webhookEvent: webhookData.event,
            webhookProcessedAt: Date.now().toString,
          },

          gatewayOnlinePaymentAttemptStatus: gateWayDetails.paymentDetails,
          gatewayOnlinePaymentStatus: gateWayDetails.razorPaymentStatus,

          portalOnlinePaymentAttemptStatus: isFromPortal ? [webhookData] : payment.gatewayOnlinePaymentAttemptStatus,
          portalOnlinePaymentStatus: isFromPortal ? this.getGatewayPaymentStatusFromStatusString(status) : payment.gatewayOnlinePaymentStatus,

          webhookOnlinePaymentAttemptStatus: !isFromPortal ? [webhookData] : undefined,
          webhookOnlinePaymentStatus: !isFromPortal ? this.getGatewayPaymentStatusFromStatusString(status) : undefined,
        });
        this.logger.log('Attempting to capture Razorpay payment', {
          paymentId,
          amount,
          status,
          isFromPortal,
          gateWayDetails,
        });
        if (
          (!isFromPortal && status === 'authorized') ||
          (isFromPortal &&
            gateWayDetails.razorPaymentStatus === RazorpayPaymentStatusEnum.AUTHORIZED)
        ) {
          try {
            const authorizedPaymentId = isFromPortal
              ? (gateWayDetails.authorizedPaymentId ?? '')
              : paymentId;
            const authorizedPaymentAmount = isFromPortal
              ? (gateWayDetails.authorisedPaymentAmount ?? 0)
              : amount;
            await this.captureRazorpayPayment(
              String(authorizedPaymentId),
              Number(authorizedPaymentAmount),
            );
          } catch (error) {
            this.logger.error('Error capturing Razorpay payment', '', { error });
          }
        } else {
          this.logger.log('cant capture the amount in the razorpay', {
            paymentId,
            amount,
            isFromPortal,
            status,
            gateWayDetails,
          });
        }

        if (payment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED || (payment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED)) {
          return {
            message: "Payment already completed"
          };
          // handleKnownErrors(ERROR_CODES.PAYMENT_ALREADY_COMPLETED, new Error('Payment already completed'));
        }

        return {
          success: true,
          paymentId: payment.id,
          registrationId: payment.registrationId,
          paymentStatus: newPaymentStatus,
        };


      } else {
        handleKnownErrors(ERROR_CODES.PAYMENT_NOTFOUND, null);
      }
    } catch (error) {
      this.logger.error("Error updating payment with webhook", '', { error });
      handleKnownErrors(ERROR_CODES.WEBHOOK_PROCESSING_FAILED, error);
    }
  }

  async updatePaymentHistory(webhookData: any, isFromPortal: boolean, razorpayOrderId: string) {
    let paymentStatus = PaymentStatusEnum.DRAFT;
    if (webhookData.event === 'payment.captured' || webhookData.event === 'payment.authorized' || webhookData.event === 'order.paid') {
      paymentStatus = PaymentStatusEnum.ONLINE_COMPLETED;
    } else  {
      paymentStatus = PaymentStatusEnum.FAILED;
    }
    let razorPaymentStatus = RazorpayPaymentStatusEnum.FAILED
    if (webhookData.event === 'payment.captured' || webhookData.event === 'order.paid') {
      razorPaymentStatus = RazorpayPaymentStatusEnum.CAPTURED;
    } else if (webhookData.event === 'payment.failed') {
      razorPaymentStatus = RazorpayPaymentStatusEnum.FAILED;
    } else if (webhookData.event === 'payment.authorized') {
      razorPaymentStatus = RazorpayPaymentStatusEnum.AUTHORIZED;
    } else if (webhookData.event === 'payment.dismissed') {
      razorPaymentStatus = RazorpayPaymentStatusEnum.FAILED;
    }

    let existedPaymentHistory = await this.paymentRepository.getPaymentHistoryByOrderIdAndStatus(razorpayOrderId, paymentStatus);
    const paymentHistory = await this.paymentRepository.getPaymentHistoryByOrderId(razorpayOrderId);
    if (existedPaymentHistory && existedPaymentHistory.length > 0) {
      existedPaymentHistory.map((history) => {
        history.portalOnlinePaymentStatus = isFromPortal ? razorPaymentStatus : history.portalOnlinePaymentStatus;
        history.portalOnlinePaymentAttemptStatus = history.portalOnlinePaymentAttemptStatus ?? [];
        isFromPortal && history.portalOnlinePaymentAttemptStatus.push(webhookData);

        history.webhookOnlinePaymentStatus = !isFromPortal ? razorPaymentStatus : history.webhookOnlinePaymentStatus;
        history.webhookOnlinePaymentAttemptStatus = history.webhookOnlinePaymentAttemptStatus ?? [];
        !isFromPortal && history.webhookOnlinePaymentAttemptStatus.push(webhookData);
      });
      if (isFromPortal) {
        const gatewayPaymentStatus = await this.getLatestPaymentStatusOfOrderId(razorpayOrderId);
        existedPaymentHistory.map((history) => {
          history.gatewayOnlinePaymentStatus = gatewayPaymentStatus.razorPaymentStatus ?? history.gatewayOnlinePaymentStatus;
          history.gatewayOnlinePaymentAttemptStatus = gatewayPaymentStatus.paymentDetails ?? history.gatewayOnlinePaymentAttemptStatus;
        });
      }

      await this.paymentRepository.updatePaymentHistory(existedPaymentHistory);
    } else if (paymentHistory && paymentHistory.length > 0) {

      let gatewayOnlinePaymentStatus;
      let gatewayOnlinePaymentAttemptStatus;
      let gatewayOnlinePayment;
      if (isFromPortal) {
        const gatewayPaymentStatus = await this.getLatestPaymentStatusOfOrderId(razorpayOrderId);
        gatewayOnlinePaymentStatus = gatewayPaymentStatus.razorPaymentStatus;
        gatewayOnlinePaymentAttemptStatus = gatewayPaymentStatus.paymentDetails;
        gatewayOnlinePayment = gatewayPaymentStatus.paymentStatus;

      }
      const newHistory = {
        paymentDetailsId: paymentHistory[0].paymentDetailsId,
        razorpayId: paymentHistory[0].razorpayId,
        offlineMeta: paymentHistory[0].offlineMeta,
        paymentMode: paymentHistory[0].paymentMode,
        portalOnlinePaymentStatus: isFromPortal ? razorPaymentStatus : undefined,
        portalOnlinePaymentAttemptStatus: isFromPortal ? [webhookData] : undefined,
        webhookOnlinePaymentStatus: !isFromPortal ? razorPaymentStatus : undefined,
        webhookOnlinePaymentAttemptStatus: !isFromPortal ? [webhookData] : undefined,
        paymentStatus: gatewayOnlinePayment ?? paymentStatus,
        createdBy: paymentHistory[0].createdBy,
        updatedBy: paymentHistory[0].updatedBy,
        createdAt: new Date(),
        updatedAt: new Date(),
        gatewayOnlinePaymentStatus: gatewayOnlinePaymentStatus,
        gatewayOnlinePaymentAttemptStatus: gatewayOnlinePaymentAttemptStatus,
      };

      await this.paymentRepository.savePaymentHistory(newHistory);
    } else {
      this.logger.warn('No payment history found for order', { razorpayOrderId, webhookData });
    }
  }

  private async initiateOnlinePayment(
    registrationId: number,
    dto: InitiatePaymentDto,
    userId: number,
    paymentCalculation: any,
    seatAvailability: any,
    registrationDetails: any,
    existingPayment: any,
    manager: any,
  ) {
    // Create Razorpay order

  
    const finalAmount: number = parseFloat(paymentCalculation.finalAmount.toFixed(2)); // Ensure only 2 floating numbers
    let razorpayOrder = {
      id: existingPayment?.razorpayId,
      amount: finalAmount,
      currency: "INR",
      receipt: `receipt_${existingPayment?.razorpayId}`,
    };
    if (existingPayment) {
      if (
        (existingPayment?.paymentStatus !== PaymentStatusEnum.ONLINE_COMPLETED ||
        existingPayment?.paymentStatus !== PaymentStatusEnum.OFFLINE_COMPLETED) 
        && dto.paymentMode === PaymentModeEnum.ONLINE 
      ) {
        // const orderIdStatus = await this.getLatestPaymentStatusOfOrderId(
        //   String(existingPayment?.razorpayId),
        // );
        if (!existingPayment?.razorpayId) {
          razorpayOrder = await this.createRazorpayOrder(finalAmount, registrationId);
        } else if (
          existingPayment?.razorpayId 
          // &&
          // orderIdStatus &&
          // orderIdStatus.paymentStatus !== PaymentStatusEnum.ONLINE_COMPLETED
        ) {
          const orderIdStatus = await this.getLatestPaymentStatusOfOrderId(
            String(existingPayment?.razorpayId),
          );

          if (orderIdStatus && orderIdStatus.paymentStatus !== PaymentStatusEnum.ONLINE_COMPLETED) {
              razorpayOrder = {
              id: existingPayment?.razorpayId,
              amount: finalAmount,
              currency: 'INR',
              receipt: `receipt_${existingPayment?.razorpayId}`,
            };
            if (finalAmount !== Number(existingPayment?.subTotal)) {
              razorpayOrder = await this.createRazorpayOrder(finalAmount, registrationId);
            }
          } else {
            try {
              await this.updatePaymentStatusByGateWayDetails(
                Number(existingPayment.id),
                existingPayment,
                orderIdStatus,
              );
            } catch (error) {
              this.logger.log('Error updating payment status', { error });
            }
            return {
              message: 'Payment already completed',
              isPaymentCompleted: true,
            };
          }
        } else {
          handleKnownErrors(ERROR_CODES.PAYMENT_NOTFOUND, null);
        }
      } else {
        return {
          message: 'Payment already completed',
          isPaymentCompleted: true,
        };
      }
    } else {
      razorpayOrder = await this.createRazorpayOrder(finalAmount, registrationId);
    }
    // Determine if should be waitlisted
    const shouldWaitlist = this.shouldWaitlist(seatAvailability, registrationDetails.program);
    const program = registrationDetails.program;
    // Create payment and invoice records
    const result = await this.paymentRepository.createPaymentWithInvoice({
      registrationId,
      paymentMeta: dto.paymentMeta,
      paymentMode: PaymentModeEnum.ONLINE,
      paymentStatus: PaymentStatusEnum.ONLINE_PENDING,
      razorpayOrderId: razorpayOrder.id,
      amounts: paymentCalculation,
      billingDetails: {
        billingName: dto.invoiceName,
        billingAddress: dto.invoiceAddress,
        panNumber: dto.panNumber,
        tanNumber: dto.tanNumber,
        isGstRegistered: dto.gstNumber ? true : false,
        invoiceEmail: dto.invoiceEmail,
        gstNumber: dto.gstNumber,
        zip: dto.zip,
        tdsAmount: dto.tdsAmount,
      },
      userId,
      shouldWaitlist,
      seatAvailability,
      program,
    }, manager);
    return {
      ...result,
      razorpayOrderId: razorpayOrder.id,
      amount: razorpayOrder.amount,
      currency: razorpayOrder.currency,
      waitlisted: shouldWaitlist,
    };
  }

  private async initiateOfflinePayment(
    registrationId: number,
    dto: InitiatePaymentDto,
    userId: number,
    paymentCalculation: any,
    seatAvailability: any,
    registrationDetails: any,
    manager: any,
  ) {
    // Determine if should be waitlisted
    const shouldWaitlist = this.shouldWaitlist(seatAvailability, registrationDetails.program);
    const program = registrationDetails.program;
    // Create payment and invoice records
    const result = await this.paymentRepository.createPaymentWithInvoice({
      registrationId,
      paymentMeta: dto.paymentMeta,
      paymentMode: PaymentModeEnum.OFFLINE,
      paymentStatus: PaymentStatusEnum.OFFLINE_PENDING,
      offlinePaymentMeta: dto.offlinePaymentMeta,
      amounts: paymentCalculation,
      billingDetails: {
        billingName: dto.invoiceName,
        billingAddress: dto.invoiceAddress,
        panNumber: dto.panNumber,
        tanNumber: dto.tanNumber,
        isGstRegistered: dto.gstNumber ? true : false,
        invoiceEmail: dto.invoiceEmail,
        gstNumber: dto.gstNumber,
        zip: dto.zip,
        tdsAmount: dto.tdsAmount,
      },
      userId,
      shouldWaitlist,
      seatAvailability,
      program,
    }, manager);
    if (result) {
      const mergeInfo = {
          pay_date: formatDateIST(new Date().toISOString()),
          pay_amount: Number(paymentCalculation?.baseAmount ?? 0) + Number(paymentCalculation?.gstAmount ?? 0),
          reg_name: registrationDetails?.fullName,
          pay_method: dto.paymentMode === PaymentModeEnum.ONLINE ? "Online" : dto.offlinePaymentMeta?.paymentMethod || "Offline",
          hdb_msd: registrationDetails?.allocatedProgram?.name || '',
      }
      const emailData: SendSingleEmailDto = {
        templateKey: process.env.ZEPTO_PAYMENT_AKNOWLEDGEMENT_OFFLINE_EMAIL_TEMPLATE_ID,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name: registrationDetails.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        to: {
          emailAddress: dto.invoiceEmail,
          name: dto.invoiceName,
        },
        // bcc: bccEmails,
        mergeInfo: mergeInfo,
        attachments: [],
        subject: 'Payment Confirmation - Infinitheism',
        trackinfo: {
          registrationId: registrationId,
          createdBy: userId,
          updatedBy: userId,
        },
      };
      this.logger.log('Sending payment acknowledgement email', emailData);

      try {
        await this.communicationService.sendSingleEmail(emailData);
        this.logger.log('Payment acknowledgement email sent successfully', {
          email: dto.invoiceEmail,
          name: dto.invoiceName,
        });
      } catch (error) {
        this.logger.error('Error sending payment acknowledgement email', '', { emailData, error });
      }

      // Send WhatsApp message for payment acknowledgement
      const messageData: SendTemplateMessageDto = {
        whatsappNumber: registrationDetails.mobileNumber, // Replace with the appropriate value
        templateName: process.env.WATI_PAYMENT_AKNOWLEDGEMENT_OFFLINE_TEMPLATE_ID || '', // Replace with the correct template name
        broadcastName: process.env.WATI_PAYMENT_AKNOWLEDGEMENT_OFFLINE_TEMPLATE_ID || '', // Replace with the correct broadcast name
        parameters: [
          { name: 'reg_name', value: registrationDetails?.fullName || '' },
        ],
        trackinfo: {
          registrationId: registrationId,
        },
      };
    
      try {
        await this.communicationService.sendTemplateMessage(messageData);
        this.logger.log('WhatsApp payment acknowledgement message sent', {
          whatsappNumber: registrationDetails.mobileNumber,
        });
      } catch (error) {
        this.logger.error('Error sending WhatsApp payment acknowledgement message', '', {
          messageData,
          error,
        });
      }
    }
    return {
      ...result,
      waitlisted: shouldWaitlist,
    };
  }

  private calculatePaymentAmounts(registrationDetails: any, gstNumber: string | null, totalTdsAmount: number) {
    const program = registrationDetails.program;
    const session = registrationDetails.programSession;
    const allocatedProgram = registrationDetails.allocatedProgram;
    const allocatedProgramSession = registrationDetails.allocatedProgramSession;
    // const registrationLevel = program.type.registrationLevel;

    // Get base amount from program or session
    const baseAmount: number =
        Number(allocatedProgram.basePrice);

    if (!baseAmount) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_FEE_NOTFOUND,
        null,
        null,
        'Program fee is not defined',
      );
    }

    let totalGstRate = 0;
    let cgstRate = 0;
    let sgstRate = 0;
    let igstRate = 0;

    // Calculate GST
    if (gstNumber) {
      if (areGSTsFromSameState(
        gstNumber,
        program.gstNumber,
      )) {
        // If GST numbers are from the same state, apply CGST and SGST
        cgstRate = Number(program.cgst) ?? 0;
        sgstRate = Number(program.sgst) ?? 0;
        totalGstRate = cgstRate + sgstRate;
      } else {
        // If GST numbers are from different states, apply IGST
        igstRate = Number(program.igst) ?? 0;
        totalGstRate = igstRate;
      }
    } else {
      cgstRate = Number(program.cgst) ?? 0;
      sgstRate = Number(program.sgst) ?? 0;
      totalGstRate = cgstRate + sgstRate;
    }

    const tdsPercent = Number(program.tdsPercent) ?? 0;
    const gstAmount = (baseAmount * totalGstRate) / 100;
    const tdsAmountOnBaseAmount = (baseAmount * tdsPercent) / 100;
    const tdsAmountOnGst = (gstAmount * tdsPercent) / 100;

    if (totalTdsAmount < 0) {
      throw new InifniBadRequestException(
        ERROR_CODES.TDS_AMOUNT_INVALID,
        null,
        null,
        'TDS amount cannot be negative',
      );
    } else {
      if (program.tdsApplicability === TdsApplicabilityEnum.BASE_ONLY) {
        if (totalTdsAmount > tdsAmountOnBaseAmount) {
          throw new InifniBadRequestException(
            ERROR_CODES.TDS_AMOUNT_EXCEEDS,
            null,
            null,
            `TDS amount cannot exceed ${tdsAmountOnBaseAmount}`,
          );
        } else {
          this.logger.log('TDS amount is valid', {
            totalTdsAmount,
            tdsAmountOnBaseAmount,
          });
        }
      } else if (program.tdsApplicability === TdsApplicabilityEnum.BASE_PLUS_TAX) {
        if (totalTdsAmount > (tdsAmountOnGst + tdsAmountOnBaseAmount)) {
          throw new InifniBadRequestException(
            ERROR_CODES.TDS_AMOUNT_EXCEEDS,
            null,
            null,
            `TDS amount cannot exceed ${tdsAmountOnGst}`,
          );
        } else {
          this.logger.log('TDS amount is valid', {
            totalTdsAmount,
            tdsAmountOnGst,
          });
        }
      }
    }

    // Calculate total
    const totalAmount = Math.round(baseAmount + gstAmount - totalTdsAmount);

  

    return {
      baseAmount,
      gstAmount,
      totalAmount,
      finalAmount: totalAmount,
      cgstRate,
      sgstRate,
      igstRate,
      tdsPercent,
      totalTdsAmount,
    };
  }

  private shouldWaitlist(seatAvailability: any, program: any): boolean {
    // Logic to determine if registration should be waitlisted
    if (program.requiresApproval) {
      return false; // No waitlisting for programs requiring approval
    }
    const { filledSeats, waitlistTriggerCount, isWaitlistTriggered } = seatAvailability;

    return filledSeats >= waitlistTriggerCount || isWaitlistTriggered;
  }

  private async createRazorpayOrder(amount: number, registrationId: number) {
    // Implement Razorpay order creation
    // This is a placeholder - implement based on your Razorpay integration
    try {
      const integerAmount = Math.round(amount * 100) / 100;
      // const razorpay = new Razorpay({
      //   key_id: process.env.RAZORPAY_KEY_ID,
      //   key_secret: process.env.RAZORPAY_SECRET_KEY,
      // });
      const receiptId = `receipt_${registrationId}_${Date.now()}`;
      const options = {
        amount: integerAmount * 100, // Convert to paise
        currency: 'INR',
        receipt: receiptId,
        payment_capture: 1, // Auto-capture
      };
      const order = await this.razorpay.orders.create(options);
      return {
        id: order.id,
        amount: integerAmount * 100, // Convert to paise
        currency: "INR",
        receipt: receiptId,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.RAZORPAY_ORDER_CREATION_FAILED, error);
    }
  }

  private async verifyWebhookSignature(webhookData: any, headers: any): Promise<boolean> {
    // Implement webhook signature verification
    // This is a placeholder - implement based on your payment gateway
    return true;
  }
  verifyPaymentSignature(orderId: string, paymentId: string, signature: string): boolean {
    const generatedSignature = crypto
      .createHmac('sha256', process.env.RAZORPAY_SECRET_KEY ?? '')
      .update(orderId + "|" + paymentId)
      .digest('hex');
    return generatedSignature === signature;
  }

  private async handlePaymentConfirmation(registrationId: number, userId: number | null) {
    // Handle post-payment processing
    try {
      // Update registration status
      await this.paymentRepository.updateRegistrationAfterPayment(registrationId, userId);

      // Generate and send invoice
      await this.invoiceService.generateAndSendInvoice(registrationId, userId ? userId : undefined);

      this.logger.log('Payment confirmation processed', { registrationId });
    } catch (error) {
      this.logger.error('Payment confirmation processing failed', '', {
        registrationId,
        error: error.message,
      });
      // Don't throw here - payment was successful, just log the issue
      handleKnownErrors(ERROR_CODES.PAYMENT_CONFIRMATION_FAILED, error);
    }
  }
  async getPaymentDetailsOfOrderIdFromRazorGateWay(orderId: string) {
    this.logger.log("Retrieving payment details for order", { orderId });
    try {
      const paymentDetails = await this.razorpay.orders.fetchPayments(orderId);
      if (!paymentDetails) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_NOTFOUND,
          null,
          null,
          `No payment found for order ${orderId}`,
        );
      }
      this.logger.log(
        "Payment details retrieved successfully from Razorpay gateway",
        { orderId, paymentDetails },
      );
      return paymentDetails;
    } catch (error) {
      this.logger.error("Failed to retrieve payment details for order", "", {
        orderId,
        error: error.message,
      });
      handleKnownErrors(ERROR_CODES.PAYMENT_GET_FAILED, error);
    }
  }
  async getLatestPaymentStatusOfOrderId(
    orderId: string,
  ): Promise<{
    paymentStatus: PaymentStatusEnum | null;
    razorPaymentStatus: RazorpayPaymentStatusEnum | null;
    paymentDetails: any;
    paymentMeta: any;
    gateWayPaidPaymentDetails?: any;
  }> {
    try {
      const paymentDetails = await this.getPaymentDetailsOfOrderIdFromRazorGateWay(orderId);
      let paymentStatus: PaymentStatusEnum | null = null;
      let razorPaymentStatus: RazorpayPaymentStatusEnum | null = null;
      let gateWayPaidPaymentDetails = null;
      let paymentMeta;

      if (paymentDetails && paymentDetails.items) {
        for (const item of paymentDetails.items) {
          paymentMeta = {
            razorpayPaymentId: item.id,
            webhookEvent: item.status,
          };
          if (item.status === RazorpayPaymentStatusEnum.FAILED) {
            paymentStatus = PaymentStatusEnum.FAILED;
            razorPaymentStatus = RazorpayPaymentStatusEnum.FAILED;
            gateWayPaidPaymentDetails = item;
          } else if (
            item.status === RazorpayPaymentStatusEnum.CAPTURED
          ) {
            paymentStatus = PaymentStatusEnum.ONLINE_COMPLETED;
            razorPaymentStatus = RazorpayPaymentStatusEnum.CAPTURED;
            gateWayPaidPaymentDetails = item;
            break;
          } else if (
            item.status === RazorpayPaymentStatusEnum.AUTHORIZED
          ) {
            paymentStatus = PaymentStatusEnum.ONLINE_COMPLETED;
            razorPaymentStatus = RazorpayPaymentStatusEnum.AUTHORIZED;
            gateWayPaidPaymentDetails = item;
            break;
          }
        }
      }
      return {
        paymentStatus,
        razorPaymentStatus,
        paymentDetails,
        paymentMeta,
        gateWayPaidPaymentDetails,
      };
    } catch (error) {
      this.logger.error("Failed to retrieve latest payment status", "", {
        orderId,
        error: error.message,
      });
      handleKnownErrors(ERROR_CODES.PAYMENT_GET_FAILED, error);
    }
  }
  getGatewayPaymentStatusFromStatusString(status: string): RazorpayPaymentStatusEnum | undefined {
    switch (status) {
      case "failed":
        return RazorpayPaymentStatusEnum.FAILED;
      case "captured":
        return RazorpayPaymentStatusEnum.CAPTURED;
      case "authorized":
        return RazorpayPaymentStatusEnum.AUTHORIZED;
      default:
        return undefined;
    }
  }

  async updatePaymentStatusByGateWayDetails(
    paymentId: number,
    paymentDetails: RegistrationPaymentDetail,
    gatewayDetails: any,
  ) {
    try {
      await this.paymentRepository.updatePayment(paymentId, {
        gatewayOnlinePaymentAttemptStatus: gatewayDetails.paymentDetails,
        gatewayOnlinePaymentStatus: gatewayDetails.razorPaymentStatus,
        paymentStatus:
          paymentDetails.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
          paymentDetails.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED
            ? paymentDetails?.paymentStatus
            : gatewayDetails?.paymentStatus,
        offlineMeta: gatewayDetails?.paymentMeta,
      });
    } catch (error) {
      this.logger.error('Failed to update payment status', '', { error });
      handleKnownErrors(ERROR_CODES.PAYMENT_UPDATE_FAILED, error);
    }
  }
  async updatePaymentEditStatus(
    registrationId: number,
    status: PaymentEditRequestStatus,
    userId: number,
  ) {
    try {
      const registrationDetails = await this.registrationRepository.findOne(registrationId);
      const existingPayment = await this.paymentRepository.findPaymentDetailsByRegistrationId(registrationId);
      if (!registrationDetails) {
        this.logger.warn('No registration found', { registrationId });
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_NOT_FOUND,
          null,
          null,
          'No existing registration found',
        );
      }

      if (!existingPayment) {
        this.logger.warn('No existing payment found', { registrationId });
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_NOTFOUND,
          null,
          null,
          'No existing payment found',
        );
      }

      if (
        existingPayment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
        existingPayment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED
      ) {
        this.logger.warn('Payment is already completed', { registrationId });
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_ALREADY_COMPLETED,
          null,
          null,
          'Payment is already completed',
        );
      }

      await this.paymentEditRequestService.processRequest(status, existingPayment.id, userId);
      try {
        await this.sendBillingDetailsEditEmail(registrationDetails, status);
      } catch (error) {
        this.logger.log('Failed to send billing details edit email', { error });
        handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_EMAIL, error);
      }
    } catch (error) {
      this.logger.error('Failed to update billing details edit status', '', { error });
      handleKnownErrors(ERROR_CODES.PAYMENT_UPDATE_FAILED, error);
    }
  }

  async sendBillingDetailsEditEmail(
    registrationDetails: ProgramRegistration,
    status: PaymentEditRequestStatus,
  ) {
    try {
      let mergeInfo = {};
      if (status === PaymentEditRequestStatus.REQUESTED) {
        mergeInfo = {
          reg_name: registrationDetails.fullName,
          hdb_msd_no: registrationDetails.allocatedProgram?.name,
          hdb_msd_date: `${formatDateIST(registrationDetails?.allocatedProgram?.startsAt.toISOString() ?? '')} to ${formatDateIST(registrationDetails?.allocatedProgram?.endsAt.toISOString() ?? '')}`,
          billing_edit_link: generatePaymentLink(
            registrationDetails.program.id,
            registrationDetails.userId,
            undefined,
            true,
          ),
        };
      } else if (status === PaymentEditRequestStatus.CLOSED) {
        mergeInfo = {
          reg_name: registrationDetails.fullName,
          hdb_msd_no: registrationDetails.allocatedProgram?.name,
          hdb_msd_date: `${formatDateIST(registrationDetails?.allocatedProgram?.startsAt.toISOString() ?? '')} to ${formatDateIST(registrationDetails?.allocatedProgram?.endsAt.toISOString() ?? '')}`,
        };
      }
      const emailData: SendSingleEmailDto = {
        templateKey:
          status === PaymentEditRequestStatus.REQUESTED
            ? process.env.ZEPTO_BILLING_DETAILS_EDIT_EMAIL_TEMPLATE_ID
            : process.env.ZEPTO_BILLING_DETAILS_EDIT_REMOVED_EMAIL_TEMPLATE_ID,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name: registrationDetails.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        to: {
          emailAddress: registrationDetails.emailAddress,
          name: registrationDetails.fullName,
        },
        mergeInfo: mergeInfo,
        attachments: [],
        subject: '',
        trackinfo: {
          registrationId: registrationDetails.id,
        },
      };
      if (status === PaymentEditRequestStatus.REQUESTED) {
        await this.communicationService.sendSingleEmail(emailData);
      }
      this.logger.log('Billing details access email sent successfully', {
        registrationId: registrationDetails.id,
        status,
      });
    } catch (error) {
      this.logger.log('Failed to send billing details edit email', { error, status });
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_EMAIL, error);
    }
  }

  async captureRazorpayPayment(paymentId: string, amount: number) {
    try {
      // Razorpay expects amount in paise (multiply by 100)
      // const captureAmount = Math.round(amount * 100);
      this.logger.log('Capturing Razorpay payment', { paymentId, amount });
      const payment = await this.razorpay.payments.capture(paymentId, amount, 'INR');
      this.logger.log('Payment captured successfully in Razorpay', {
        paymentId,
        amount: amount,
      });
      return payment;
    } catch (error) {
      console.log(error);
      this.logger.error('Failed to capture payment in Razorpay', '', {
        paymentId,
        amount,
        error: error.message,
      });
      handleKnownErrors(ERROR_CODES.RAZORPAY_PAYMENT_CAPTURE_FAILED, error);
    }
  }
}
