import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, DataSource, IsNull, Not } from 'typeorm';
import { 
  RegistrationPaymentDetail, 
  RegistrationPaymentDetailsHistory,
  RegistrationInvoiceDetail,
  ProgramRegistration,
  Program,
  ProgramSession,
  RegistrationTravelInfo,
  RegistrationTravelPlan,
} from 'src/common/entities';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { UpdatePaymentDto } from './dto/update-payment.dto';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { InvoiceTypeEnum } from 'src/common/enum/invoice-type.enum';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { incrementSeatCounts } from '../common/utils/seat-count.util';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { RegistrationService } from 'src/registration/registration.service';
import { generateNextSequence } from 'src/common/utils/common.util';
import { RazorpayPaymentStatusEnum } from 'src/common/enum/razorpay-payment-status.enum';
import { RegistrationInvoiceBasicHistory } from 'src/common/entities/registration-invoice-basic-history.entity';
import { TravelStatusEnum } from 'src/common/enum/travel-status.enum';

@Injectable()
export class PaymentRepository {
  constructor(
    @InjectRepository(RegistrationPaymentDetail)
    private readonly paymentRepo: Repository<RegistrationPaymentDetail>,
    @InjectRepository(RegistrationPaymentDetailsHistory)
    private readonly paymentHistoryRepo: Repository<RegistrationPaymentDetailsHistory>,
    @InjectRepository(RegistrationInvoiceDetail)
    private readonly invoiceRepo: Repository<RegistrationInvoiceDetail>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepo: Repository<ProgramRegistration>,
    @InjectRepository(Program)
    private readonly programRepo: Repository<Program>,
    @InjectRepository(ProgramSession)
    private readonly sessionRepo: Repository<ProgramSession>,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
    private readonly registartionService: RegistrationService,
    @InjectRepository(RegistrationTravelInfo)
    private readonly travelInfo: Repository<RegistrationTravelInfo>,
    @InjectRepository(RegistrationTravelPlan)
    private readonly travelPlan: Repository<RegistrationTravelPlan>,
  ) {}

  async findByRegistrationId(registrationId: number): Promise<RegistrationPaymentDetail | null> {
    try {
      return await this.paymentRepo.findOne({
        where: { registrationId },
        relations: ['registration'],
      });
    } catch (error) {
      this.logger.error('Error finding payment by registration ID', '', { registrationId, error });
      return null;
    }
  }

  async findPaymentDetailsByRegistrationId(registrationId: number): Promise<RegistrationPaymentDetail | null> {
    try {
      return await this.paymentRepo.findOne({
        where: { registrationId },
        relations: [
          'registration',
          'registration.program',
          'registration.allocatedProgram',
          'registration.approvals',
        ],
      });
    } catch (error) {
      this.logger.error('Error finding payment details by registration ID', '', { registrationId, error });
      return null;
    }
  }

  async getRegistrationWithProgramDetails(registrationId: number) {
    try {
      return await this.registrationRepo.findOne({
        where: { id: registrationId },
        relations: [
          'program', 
          'program.type', 
          'programSession',
          'user',
          'allocatedProgram',
          'allocatedSession',
          'approvals',
        ],
      });
    } catch (error) {
      this.logger.error('Error getting registration details', '', { registrationId, error });
      throw error;
    }
  }

  /**
   * Create payment history record
   * This method captures the state of payment for audit trail
   */
  private async createPaymentHistory(
    manager: any,
    paymentId: number,
    payment: RegistrationPaymentDetail,
    userId: number | null,
    action: string = 'UPDATE'
  ) {
    try {
      const historyRecord = manager.create(RegistrationPaymentDetailsHistory, {
        paymentDetailsId: paymentId,
        paymentMode: payment.paymentMode,
        razorpayId: payment.razorpayId,
        offlineMeta: payment.offlineMeta,
        paymentStatus: payment.paymentStatus,
        createdBy: userId,
        updatedBy: userId,
      });

      await manager.save(RegistrationPaymentDetailsHistory, historyRecord);

      this.logger.log('Payment history record created', {
        paymentId,
        paymentStatus: payment.paymentStatus,
        action,
        historyId: historyRecord.id
      });

      return historyRecord;
    } catch (error) {
      this.logger.error('Failed to create payment history record', error);
      handleKnownErrors(ERROR_CODES.PAYMENT_INITIATION_FAILED, error);
      // Don't throw here - history is important but shouldn't break the main flow
    }
  }

  /**
   * Get payment history for a specific payment
   */
  async getPaymentHistory(paymentId: number): Promise<RegistrationPaymentDetailsHistory[]> {
    try {
      return await this.paymentHistoryRepo.find({
        where: { paymentDetailsId: paymentId },
        order: { createdAt: 'DESC' },
        relations: ['createdByUser', 'updatedByUser'],
      });
    } catch (error) {
      this.logger.error('Error getting payment history', '', { paymentId, error });
      return [];
    }
  }

  /**
   * Get payment history for a registration
   */
  async getPaymentHistoryByRegistration(registrationId: number): Promise<RegistrationPaymentDetailsHistory[]> {
    try {
      const payment = await this.findByRegistrationId(registrationId);
      if (!payment) {
        return [];
      }
      return await this.getPaymentHistory(payment.id);
    } catch (error) {
      this.logger.error('Error getting payment history by registration', '', { registrationId, error });
      return [];
    }
  }

  async createPaymentWithInvoice(paymentData: any, manager?: any) {
    if (manager) {
      // Use provided manager (transaction already started at service level)
      return this.createPaymentWithInvoiceInternal(paymentData, manager);
    }

    // Backward compatibility - create own transaction
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      await queryRunner.startTransaction();
      transactionStarted = true;
      
      const result = await this.createPaymentWithInvoiceInternal(paymentData, queryRunner.manager);
      
      await queryRunner.commitTransaction();
      transactionCommitted = true;
      
      return result;
    } catch (error) {
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Payment creation transaction failed', error);
      handleKnownErrors(ERROR_CODES.PAYMENT_CREATION_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }

  private async createPaymentWithInvoiceInternal(paymentData: any, manager: any) {
    this.logger.log('Creating payment and invoice', { 
      registrationId: paymentData.registrationId,
      paymentMode: paymentData.paymentMode 
    });

    // Store question answers using registration service logic
    if (paymentData.paymentMeta && paymentData.paymentMeta.length) {
      await this.registartionService.storeQuestionAnswers(
        manager,
        paymentData.registrationId,
        paymentData.paymentMeta,
        paymentData.userId,
      );
    }
    
    // Find existing payment
    const existedPayment = await manager.findOne(RegistrationPaymentDetail, {
      where: { registrationId: paymentData.registrationId }
    });

    // Create payment record
    const payment = manager.create(RegistrationPaymentDetail, {
      id: existedPayment ? existedPayment.id : undefined,
      registrationId: paymentData.registrationId,
      programSessionId: paymentData.programSessionId || null,
      paymentMode: paymentData.paymentMode,
      paymentStatus: paymentData.paymentStatus,
      originalAmount: paymentData.amounts.baseAmount,
      gstAmount: paymentData.amounts.gstAmount,
      tds: paymentData.billingDetails.tanNumber ? paymentData.billingDetails.tdsAmount || 0 : 0,
      taxAmount: paymentData.amounts.gstAmount,
      subTotal: paymentData.amounts.totalAmount,
      razorpayId: paymentData.razorpayOrderId || null,
      offlineMeta: paymentData.offlinePaymentMeta || null,
      createdBy: { id: paymentData.userId } as any,
      updatedBy: { id: paymentData.userId } as any,
    });

    const savedPayment = await manager.save(RegistrationPaymentDetail, payment);
    
    // Create payment history record for this payment
    const existedPaymentHistory = await this.getPaymentHistory(savedPayment.id);
    // if (!existedPaymentHistory || existedPaymentHistory.length === 0) {
    await this.createPaymentHistory(
      manager,
      savedPayment.id,
      savedPayment,
      paymentData.userId,
      'CREATE'
    );
    // }

    // Check if invoice already exists for this registration
    const existedInvoice = await manager.findOne(RegistrationInvoiceDetail, {
      where: { registrationId: paymentData.registrationId }
    });

    // Create invoice record
    const invoice = manager.create(RegistrationInvoiceDetail, {
      id: existedInvoice?.id ? existedInvoice.id : undefined,
      registrationId: paymentData.registrationId,
      invoiceType: paymentData.paymentMode === PaymentModeEnum.ONLINE 
        ? InvoiceTypeEnum.ONLINE 
        : InvoiceTypeEnum.OFFLINE,
      invoiceName: paymentData.billingDetails.billingName,
      invoiceEmail: paymentData.billingDetails.invoiceEmail,
      invoiceAddress: paymentData.billingDetails.billingAddress,
      panNumber: paymentData.billingDetails.panNumber || null,
      tdsApplicable: paymentData.billingDetails.tanNumber ? true : false,
      tanNumber: paymentData.billingDetails.tanNumber || null,
      tdsAmount: paymentData.billingDetails.tanNumber ? paymentData.billingDetails.tdsAmount || 0 : 0,
      isGstRegistered: paymentData.billingDetails.gstNumber ? true : false,
      taxAmount: paymentData.amounts.gstAmount,
      invoiceStatus: InvoiceStatusEnum.DRAFT,
      gstNumber: paymentData.billingDetails.gstNumber || null,
      zip: paymentData.billingDetails.zip || null,
      razorpayId: paymentData.razorpayOrderId || null,
      createdBy: { id: paymentData.userId } as any,
      updatedBy: { id: paymentData.userId } as any,
    });

    await manager.save(RegistrationInvoiceDetail, invoice);

    await manager.save(RegistrationInvoiceBasicHistory, {
      invoiceId: invoice.id,
      invoiceType:
        paymentData.paymentMode === PaymentModeEnum.ONLINE
          ? InvoiceTypeEnum.ONLINE
          : InvoiceTypeEnum.OFFLINE,
      invoiceName: paymentData.billingDetails.billingName,
      invoiceEmail: paymentData.billingDetails.invoiceEmail,
      invoiceAddress: paymentData.billingDetails.billingAddress,
      panNumber: paymentData.billingDetails.panNumber || null,
      tanNumber: paymentData.billingDetails.tanNumber || null,
      tdsAmount: paymentData.billingDetails.tanNumber
        ? paymentData.billingDetails.tdsAmount || 0
        : 0,
      gstNumber: paymentData.billingDetails.gstNumber || null,
      zip: paymentData.billingDetails.zip || null,
      createdBy: { id: paymentData.userId } as any,
      createdAt: new Date(),
    });

    // Update registration status based on waitlist and payment requirements
    let newRegistrationStatus = RegistrationStatusEnum.PENDING;
    
    if (paymentData.shouldWaitlist) {
      newRegistrationStatus = RegistrationStatusEnum.WAITLISTED;
      
      // Generate waitlist sequence number
      const waitlistQuery: any = { waitingListSeqNumber: Not(IsNull()) };
      if (paymentData.registrationLevel === 'program') {
        waitlistQuery.program = { id: paymentData.program.programId };
      } else {
        waitlistQuery.programSession = { id: paymentData.programSessionId };
      }
      
      const lastWaitlistRegistration = await manager.findOne(ProgramRegistration, {
        where: waitlistQuery,
        order: { waitingListSeqNumber: 'DESC' },
      });
      
      const nextWaitlistSeqNumber = lastWaitlistRegistration?.waitingListSeqNumber
        ? lastWaitlistRegistration.waitingListSeqNumber + 1
        : 1;

      await manager.update(ProgramRegistration, paymentData.registrationId, {
        registrationStatus: newRegistrationStatus,
        waitingListSeqNumber: nextWaitlistSeqNumber,
        auditRefId: paymentData.registrationId,
        parentRefId: paymentData.registrationId,
      });
    } else {
      if (paymentData.paymentMode === PaymentModeEnum.OFFLINE) {
        await this.updateRegistrationAfterPayment(paymentData.registrationId, paymentData.userId, manager);
      }
    }

    this.logger.log('Payment and invoice created successfully', {
      paymentId: savedPayment.id,
      invoiceId: invoice.id,
      registrationId: paymentData.registrationId
    });

    return {
      paymentId: savedPayment.id,
      invoiceId: invoice.id,
      registrationId: paymentData.registrationId,
      paymentStatus: savedPayment.paymentStatus,
      invoiceStatus: invoice.invoiceStatus,
      remainingSeats: paymentData.seatAvailability.waitlistTriggerCount - paymentData.seatAvailability.filledSeats,
      participantRemainingSeats: paymentData.seatAvailability.totalSeats - paymentData.seatAvailability.filledSeats,
    };
  }

  async updatePaymentStatus(registrationId: number, dto: UpdatePaymentDto, userId: number, manager?: any) {
    if (manager) {
      // Use provided manager (transaction already started at service level)
      return this.updatePaymentStatusInternal(registrationId, dto, userId, manager);
    }

    // Backward compatibility - create own transaction
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      await queryRunner.startTransaction();
      transactionStarted = true;
      
      const result = await this.updatePaymentStatusInternal(registrationId, dto, userId, queryRunner.manager);
      
      await queryRunner.commitTransaction();
      transactionCommitted = true;
      
      return result;
    } catch (error) {
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Payment update transaction failed', error);
      throw error;
    } finally {
      await queryRunner.release();
    }
  }

  private async updatePaymentStatusInternal(registrationId: number, dto: UpdatePaymentDto, userId: number, manager: any) {
    // Find existing payment
    const payment = await manager.findOne(RegistrationPaymentDetail, {
      where: { registrationId },
      relations: ['registration','registration.program', 'registration.programSession'],
    });

    if (!payment) {
      throw new InifniBadRequestException(
        ERROR_CODES.PAYMENT_NOTFOUND,
        null,
        null,
        `No payment found for registration ${registrationId}`
      );
    }

    // Store the current state before update for history
    const currentPaymentState = { ...payment };

    // Update payment details
    const updateData: any = {
      paymentStatus: dto.paymentStatus,
      updatedBy: { id: userId } as any,
    };

    // if (dto.paymentDate) {
    //   updateData.paymentDate = new Date(dto.paymentDate);
    // }

    if (dto.paymentMeta) {
      updateData.offlineMeta = { ...payment.offlineMeta, ...dto.paymentMeta };
    }

    // if(dto.markAsReceivedDate){
    //   updateData.markAsReceivedDate = dto.markAsReceivedDate ?? null;
    // }
    if (
      currentPaymentState.paymentStatus != PaymentStatusEnum.ONLINE_COMPLETED &&
      currentPaymentState.paymentStatus != PaymentStatusEnum.OFFLINE_COMPLETED &&
      (dto.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
       dto.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED)
    ) {
      if (payment.paymentMode === PaymentModeEnum.ONLINE) {
        updateData.paymentDate = dto.markAsReceivedDate ?? null;
      } else {
        updateData.markAsReceivedDate = dto.markAsReceivedDate ?? null;
      }
    }

    // Always set audit fields for auditable entities
    updateData.auditRefId = payment.id;
    updateData.parentRefId = payment.registrationId;  // Use registrationId for parentRefId

    await manager.update(RegistrationPaymentDetail, payment.id, updateData);

    // Get updated payment for history
    const updatedPayment = await manager.findOne(RegistrationPaymentDetail, {
      where: { id: payment.id },
    });

    // Create payment history record for this update
    if (updatedPayment) {
      await this.createPaymentHistory(
        manager,
        payment.id,
        updatedPayment,
        userId,
        'UPDATE'
      );
    }


    this.logger.log('Payment status updated successfully', {
      paymentId: payment.id,
      registrationId,
      oldStatus: currentPaymentState.paymentStatus,
      newStatus: dto.paymentStatus
    });

    return {
      paymentId: payment.id,
      registrationId,
      paymentStatus: dto.paymentStatus,
      registrationConfirmed: dto.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
                            dto.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED,
    };
  }

  async getPaymentDetailsWithInvoice(registrationId: number) {
    try {
      const payment = await this.paymentRepo.findOne({
        where: { registrationId },
        relations: ['registration', 'registration.program', 'registration.programSession'],
      });

      const invoice = await this.invoiceRepo.findOne({
        where: { registrationId },
      });

      // Get payment history if payment exists
      let paymentHistory: RegistrationPaymentDetailsHistory[] = [];
      if (payment) {
        paymentHistory = await this.getPaymentHistory(payment.id);
      }

      return {
        payment,
        invoice,
        paymentHistory,
        registrationId,
      };
    } catch (error) {
      this.logger.error('Error getting payment details with invoice', '', { registrationId, error });
      throw error;
    }
  }

  async processWebhookUpdate(webhookData: any, isFromPortal: boolean = false) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      // Start transaction
      await queryRunner.startTransaction();
      transactionStarted = true;

      // Extract payment details from webhook
      const razorpayOrderId = webhookData.payload?.payment?.entity?.order_id;
      const paymentId = webhookData.payload?.payment?.entity?.id;
      const status = webhookData.payload?.payment?.entity?.status;

      if (!razorpayOrderId) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_WEBHOOK_DATA,
          null,
          null,
          'Razorpay order ID not found in webhook'
        );
      }

      // Find payment by Razorpay order ID
      const payment = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
        where: { razorpayId: razorpayOrderId },
        relations: ['registration', 'registration.program', 'registration.programSession'],
      });

      if (!payment) {
        this.logger.warn('Payment not found for webhook', { razorpayOrderId });
        return { success: false, message: 'Payment not found' };
      }
      // Update payment status based on webhook event
      let newPaymentStatus: PaymentStatusEnum;
      if (status === 'captured' || status === 'authorized') {
        newPaymentStatus = PaymentStatusEnum.ONLINE_COMPLETED;
        // Update registration status based on payment
        await this.updateRegistrationAfterPayment(payment.registrationId, null);
      } else if (status === 'failed' || status === 'dismissed') {
        newPaymentStatus = PaymentStatusEnum.FAILED; // Keep pending for retry
      } else {
        newPaymentStatus = payment.paymentStatus; // No change
      }

      // Check if payment is already completed
      if (payment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED || 
        payment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) {
        if (newPaymentStatus === PaymentStatusEnum.ONLINE_COMPLETED && payment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED) {
          this.logger.warn('Payment already completed for webhook', { paymentId, status });
          await queryRunner.manager.update(RegistrationPaymentDetail, payment.id, {
            portalOnlinePaymentStatus: isFromPortal ? RazorpayPaymentStatusEnum.CAPTURED : payment.portalOnlinePaymentStatus,
            portalOnlinePaymentAttemptStatus: isFromPortal ? [webhookData] : payment.portalOnlinePaymentAttemptStatus,

            gatewayOnlinePaymentStatus: isFromPortal ? RazorpayPaymentStatusEnum.CAPTURED : payment.gatewayOnlinePaymentStatus,
            gatewayOnlinePaymentAttemptStatus: isFromPortal ? [webhookData] : payment.gatewayOnlinePaymentAttemptStatus,

            webhookOnlinePaymentStatus: !isFromPortal ? RazorpayPaymentStatusEnum.CAPTURED : payment.webhookOnlinePaymentStatus,
            webhookOnlinePaymentAttemptStatus: !isFromPortal ? [webhookData] : payment.webhookOnlinePaymentAttemptStatus,
            auditRefId: payment.id,
            parentRefId: payment.registrationId,
          });
        }
        this.logger.warn('Payment already completed for webhook', { paymentId, status });
        return { success: false, message: 'Payment already completed'  };
      }

      // Store current state for history
      const currentPaymentState = { ...payment };

      // Update payment
      await queryRunner.manager.update(RegistrationPaymentDetail, payment.id, {
        paymentStatus: newPaymentStatus,
        paymentDate: status === 'captured' || status === 'authorized' ? new Date() : undefined,
        offlineMeta: {
          ...payment.offlineMeta,
          razorpayPaymentId: paymentId,
          webhookEvent: webhookData.event,
          webhookProcessedAt: Date.now().toString,
        },
        portalOnlinePaymentStatus: isFromPortal ? this.getGatewayPaymentStatusFromStatusString(status) : payment.portalOnlinePaymentStatus,
        portalOnlinePaymentAttemptStatus: isFromPortal ? [webhookData] : payment.portalOnlinePaymentAttemptStatus,
        webhookOnlinePaymentStatus: !isFromPortal ? this.getGatewayPaymentStatusFromStatusString(status) : payment.webhookOnlinePaymentStatus,
        webhookOnlinePaymentAttemptStatus: !isFromPortal ? [webhookData] : payment.webhookOnlinePaymentAttemptStatus,
        auditRefId: payment.id,
        parentRefId: payment.registrationId,
      });

      await queryRunner.commitTransaction();
      transactionCommitted = true;

      this.logger.log('Webhook processed successfully', {
        paymentId: payment.id,
        registrationId: payment.registrationId,
        razorpayOrderId,
        oldStatus: currentPaymentState.paymentStatus,
        newStatus: newPaymentStatus
      });

      return {
        success: true,
        paymentId: payment.id,
        registrationId: payment.registrationId,
        paymentStatus: newPaymentStatus,
      };

    } catch (error) {
      // Simple rollback like payment module
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Webhook processing transaction failed', error);
      throw error;
    } finally {
      await queryRunner.release();
    }
  }

  async updateRegistrationAfterPayment(registrationId: number, userId: number | null, manager?: any) {
    if (manager) {
      // Use provided manager (transaction already started at service level)
      return this.updateRegistrationAfterPaymentInternal(manager, registrationId, userId);
    }

    // Backward compatibility - create own transaction
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      await queryRunner.startTransaction();
      transactionStarted = true;
      
      const result = await this.updateRegistrationAfterPaymentInternal(queryRunner.manager, registrationId, userId);
      
      await queryRunner.commitTransaction();
      transactionCommitted = true;
      
      return result;
    } catch (error) {
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Registration update after payment failed', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }

  private async updateRegistrationAfterPaymentInternal(manager: any, registrationId: number, userId: number | null) {
    // Get registration details
    const registration = await manager.findOne(ProgramRegistration, {
      where: { id: registrationId },
      relations: ['program', 'programSession', 'program.type'],
    });

    if (!registration) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        `Registration ${registrationId} not found`
      );
    }
    
    if (!registration.program.requiresApproval) {
      // Check if registration is waitlisted
      if (registration.registrationStatus === RegistrationStatusEnum.WAITLISTED) {
        // Check if seat is now available
        const seatAvailability = await this.registartionService.checkSeatAvailability(
          registration.program,
          registration.programSession,
          registration.program.type.registrationLevel
        );

        if (!seatAvailability.shouldWaitlist) {
          // Move from waitlist to confirmed
          await manager.update(ProgramRegistration, registrationId, {
            registrationStatus: RegistrationStatusEnum.COMPLETED,
            waitingListSeqNumber: undefined,
            auditRefId: registrationId,
            parentRefId: registrationId,
          });

          // Update seat counts
          await incrementSeatCounts(manager, registration.program.id, registration.programSession?.id);
        }
        // If still should waitlist, keep status as is
      } else {
        // Direct confirmation
        await manager.update(ProgramRegistration, registrationId, {
          registrationStatus: RegistrationStatusEnum.COMPLETED,
          auditRefId: registrationId,
          parentRefId: registrationId,
        });

        // Update seat counts
        await incrementSeatCounts(manager, registration.program.id, registration.programSession?.id);
      }
    } else {
      const travelInfo = await this.travelInfo.findOne({
        where: {
          registrationId: registrationId,
        },
      });
      const travelPlan = await this.travelPlan.findOne({
        where: {
          registrationId: registrationId,
        },
      });
      const paymentInfo = await this.paymentRepo.findOne({
        where: {
          registrationId: registrationId,
        },
      });
      // If travel info and plan are completed, update registration to COMPLETED
      if (
        travelInfo?.travelInfoStatus === TravelStatusEnum.COMPLETED &&
        travelPlan?.travelPlanStatus === TravelStatusEnum.COMPLETED &&
        (paymentInfo?.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
          paymentInfo?.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) &&
        registration?.registrationStatus !== RegistrationStatusEnum.COMPLETED &&
        registration?.registrationStatus !== RegistrationStatusEnum.CANCELLED &&
        registration?.registrationStatus !== RegistrationStatusEnum.REJECTED &&
        registration?.registrationStatus !== RegistrationStatusEnum.ON_HOLD
      ) {
        await manager.update(ProgramRegistration, registrationId, {
          registrationStatus: RegistrationStatusEnum.COMPLETED,
          auditRefId: registrationId,
          parentRefId: registrationId,
        });
        this.logger.log(
          'Registration status updated to COMPLETED after payment and travel info/plan completion',
          { registrationId, registrationStatus: RegistrationStatusEnum.COMPLETED },
        );
      } else {
        this.logger.log('Registration status not updated as travel info/plan is incomplete', {
          registrationId: registrationId,
          travelInfoStatus: travelInfo?.travelInfoStatus,
          travelPlanStatus: travelPlan?.travelPlanStatus,
          registrationStatus: registration?.registrationStatus,
        });
      }
    }

    this.logger.log('Registration updated after payment', { registrationId });
  }

  async updateSeatCountsAfterPayment(registrationId: number) {
    // This method is called from service layer after payment confirmation
    // Seat count updates are handled in updateRegistrationAfterPayment
    await this.updateRegistrationAfterPayment(registrationId, null);
    this.logger.log('Seat count update triggered for registration', { registrationId });
  }
  async getPaymentHistoryByOrderId(razorpayOrderId: string): Promise<RegistrationPaymentDetailsHistory[]> {
    try {
      const paymentHostory = await this.paymentHistoryRepo.find({
        where: { razorpayId: razorpayOrderId },
      });

      if (!paymentHostory || paymentHostory.length === 0) {
        return [];
      }

      return paymentHostory;
    } catch (error) {
      this.logger.error('Error getting payment history by order ID', '', { razorpayOrderId, error });
      return [];
    }
  }

async getPaymentHistoryByOrderIdAndStatus(
    razorpayOrderId: string,
    paymentStatus: PaymentStatusEnum
  ): Promise<RegistrationPaymentDetailsHistory[]> {
    try {
      const paymentHistory = await this.paymentHistoryRepo.find({
        where: {
          razorpayId: razorpayOrderId,
          paymentStatus: paymentStatus,
        },
        order: { id: 'DESC' },
      });

      if (!paymentHistory || paymentHistory.length === 0) {
        return [];
      }
      return paymentHistory;
    } catch (error) {
      this.logger.error('Error getting payment history by order ID and status', '', { razorpayOrderId, paymentStatus, error });
      return [];
    }
  }
  async savePaymentHistory(
    paymentHistory: any
  ): Promise<RegistrationPaymentDetailsHistory> {
    try {
      return await this.paymentHistoryRepo.save(paymentHistory);
    } catch (error) {
      this.logger.error('Error saving payment history', '', { paymentHistory, error });
      handleKnownErrors(ERROR_CODES.PAYMENT_HISTORY_SAVE_FAILED, error);
    }
  }
  async updatePaymentHistory(paymentHistory: RegistrationPaymentDetailsHistory[]) {
    // Update payment history records
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();

    try {
      await queryRunner.startTransaction();

      if (!paymentHistory || !Array.isArray(paymentHistory) || paymentHistory.length === 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_HISTORY_NOT_FOUND,
          null,
          null,
          'No payment history records provided for update'
        );
      }

      for (const history of paymentHistory) {
        await queryRunner.manager.update(RegistrationPaymentDetailsHistory, 
           history.id, 
          {
            portalOnlinePaymentStatus: history.portalOnlinePaymentStatus,
            portalOnlinePaymentAttemptStatus: history.portalOnlinePaymentAttemptStatus,
            gatewayOnlinePaymentStatus: history.gatewayOnlinePaymentStatus,
            gatewayOnlinePaymentAttemptStatus: history.gatewayOnlinePaymentAttemptStatus,
            webhookOnlinePaymentAttemptStatus: history.webhookOnlinePaymentAttemptStatus,
            webhookOnlinePaymentStatus: history.webhookOnlinePaymentStatus,
            paymentStatus: history.paymentStatus,
            updatedAt: new Date(),
          }
        );
        // Save the updated history record
        // await queryRunner.manager.save(RegistrationPaymentDetailsHistory, history);
      }

      await queryRunner.commitTransaction();

      this.logger.log('Payment history updated successfully', { count: paymentHistory.length });

    } catch (error) {
      try {
        await queryRunner.rollbackTransaction();
      } catch (rollbackError) {
        this.logger.error('Failed to rollback transaction', rollbackError);
      }
      this.logger.error('Payment history update failed', error);
      handleKnownErrors(ERROR_CODES.PAYMENT_HISTORY_UPDATE_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }
  async getPaymentDetailsByOrderId(razorpayOrderId: string): Promise<RegistrationPaymentDetail | null> {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();

    try {
      const payment = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
        where: { razorpayId: razorpayOrderId },
        relations: ['registration', 'registration.program', 'registration.programSession'],
      });

      return payment;
    } catch (error) {
      this.logger.error('Error getting payment details by order ID', '', { razorpayOrderId, error });
      return null;
    } finally {
      await queryRunner.release();
    }
  }

  getGatewayPaymentStatusFromStatusString(status: string): RazorpayPaymentStatusEnum | undefined {
    switch (status) {
      case "failed":
        return RazorpayPaymentStatusEnum.FAILED;
      case "captured":
        return RazorpayPaymentStatusEnum.CAPTURED;
      case "authorized":
        return RazorpayPaymentStatusEnum.AUTHORIZED;
      default:
        return undefined;
    }
  }
  async updateOnlinePaymentStatus(paymentId: number, updateData: Partial<RegistrationPaymentDetail>, manager?: any): Promise<void> {
   // get the existing payment record
    const existingPayment = await manager.findOne(RegistrationPaymentDetail, {
      where: { id: paymentId },
    });
    if (!existingPayment) {
      this.logger.warn('Payment not found for update', { paymentId });
      handleKnownErrors(ERROR_CODES.PAYMENT_NOTFOUND, `Payment ${paymentId} not found`);
    }
    if (manager) {
      // Use provided manager (transaction already started at service level)
      try {
        // Ensure audit fields are set for auditable entities
        const auditableUpdateData = {
          ...updateData,
          auditRefId: paymentId,
          parentRefId: existingPayment.registrationId,
        };
        await manager.update(RegistrationPaymentDetail, paymentId, auditableUpdateData);
        this.logger.log('Online payment status updated with manager', { paymentId });
      } catch (error) {
        this.logger.error('Error updating online payment status with manager', '', { paymentId, updateData, error });
        handleKnownErrors(ERROR_CODES.PAYMENT_UPDATE_FAILED, error);
      }
      return;
    }

    // Backward compatibility - use QueryRunner for operations without existing transaction
    const queryRunner = this.dataSource.createQueryRunner();
    
    try {
      await queryRunner.connect();
      // Ensure audit fields are set for auditable entities  
      const auditableUpdateData = {
        ...updateData,
        auditRefId: paymentId,
        parentRefId: existingPayment.registrationId,
      };
      await queryRunner.manager.update(RegistrationPaymentDetail, paymentId, auditableUpdateData);
      this.logger.log('Online payment status updated with QueryRunner', { paymentId });
    } catch (error) {
      this.logger.error('Error updating online payment status', '', { paymentId, updateData, error });
      handleKnownErrors(ERROR_CODES.PAYMENT_UPDATE_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }
  async updatePaymentEditStatus(registrationId: number, enableEdit: boolean) {
    try {
      await this.paymentRepo.update(
        { registrationId },
        {
          paymentStatus: PaymentStatusEnum.OFFLINE_COMPLETED,
          // isEditable: enableEdit,
        },
      );
    } catch (error) {
      this.logger.error('Failed to update payment edit status', '', { error });
      handleKnownErrors(ERROR_CODES.PAYMENT_UPDATE_FAILED, error);
    }
  }
  async updatePayment(paymentId: number, updateData: Partial<RegistrationPaymentDetail>) {
    try {
      await this.paymentRepo.update(paymentId, updateData);
    } catch (error) {
      this.logger.error('Failed to update payment', '', { error });
      handleKnownErrors(ERROR_CODES.PAYMENT_UPDATE_FAILED, error);
    }
  }
}
