import {
  Body,
  Controller,
  HttpCode,
  HttpStatus,
  Post,
  Get,
  Put,
  Res,
  UseGuards,
  ValidationPipe,
  Req,
  Param,
  ParseIntPipe,
} from '@nestjs/common';
import { Response } from 'express';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiResponse, ApiSecurity, ApiTags, ApiParam } from '@nestjs/swagger';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { PaymentService } from './payment.service';
import { UpdatePaymentDto } from './dto/update-payment.dto';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import {InitiatePaymentDto} from './dto/create-payment.dto'
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
import { PaymentEditRequest } from 'src/common/entities/payment-edit-request.entity';
import { PaymentEditRequestStatus } from 'src/common/enum/payment-edit-request-status.enum';
import { PaymentEditRequestDto } from './dto/payment-edit-request.dto';

@ApiTags('payment')
@Controller('payment')
export class PaymentController {
  constructor(
    private readonly paymentService: PaymentService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Post('initiate/:registrationId')
  @HttpCode(HttpStatus.CREATED)
  @UseGuards(CombinedAuthGuard, RolesGuard)
  @Roles('admin', 'viewer', 'finance_manager','mahatria', 'shoba', 'relational_manager', 'operational_manger', 'rm_support')
  @ApiBearerAuth('Authorization')
  @ApiSecurity('userIdAuth')
  @ApiOperation({ summary: 'Initiate payment for a registration' })
  @ApiParam({ name: 'registrationId', description: 'Registration ID', type: Number })
  @ApiBody({ type: InitiatePaymentDto })
  @ApiResponse({ 
    status: HttpStatus.CREATED, 
    description: SWAGGER_API_RESPONSE.CREATED,
    schema: {
      properties: {
        paymentId: { type: 'number' },
        registrationId: { type: 'number' },
        paymentStatus: { type: 'string' },
        invoiceId: { type: 'number' },
        razorpayOrderId: { type: 'string' },
        amount: { type: 'number' },
        waitlisted: { type: 'boolean' },
        remainingSeats: { type: 'number' }
      }
    }
  })
  async initiatePayment(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: InitiatePaymentDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
    }
    const userId = user.id;
    
    this.logger.log('Payment initiation request received', { registrationId, dto });
    
    try {
      const data = await this.paymentService.initiatePayment(registrationId, dto, userId);
      
      let message = 'Payment initiated successfully';
      if (data.waitlisted) {
        message = 'Registration waitlisted - payment initiated';
      } else if (data.paymentStatus === 'online_pending') {
        message = 'Online payment initiated - complete payment to confirm registration';
      } else if (data.paymentStatus === 'offline_pending') {
        message = 'Offline payment initiated - payment verification pending';
      }
      
      await this.responseService.success(res, message, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put('update/:registrationId')
  @HttpCode(HttpStatus.OK)
  @UseGuards(CombinedAuthGuard, RolesGuard)
  @Roles('admin', 'viewer', 'finance_manager', 'shoba')
  @ApiBearerAuth('Authorization')
  @ApiSecurity('userIdAuth')
  @ApiOperation({ summary: 'Update payment status for a registration' })
  @ApiParam({ name: 'registrationId', description: 'Registration ID', type: Number })
  @ApiBody({ type: UpdatePaymentDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  async updatePayment(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdatePaymentDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    const user = req.user;
    if (!user || !user.id) {
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
    }
    const userId = user.id;
    
    this.logger.log('Payment update request received', { registrationId, dto });
    
    try {
      const data = await this.paymentService.updatePayment(registrationId, dto, userId);
      
      let message = 'Payment updated successfully';
      if (data.registrationConfirmed) {
        message = 'Payment confirmed - registration completed';
      } 
      else if (data.paymentStatus === PaymentStatusEnum.WAITLISTED) {
        message = 'Payment updated - registration waitlisted';
      }
      
      await this.responseService.success(res, message, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':registrationId')
  @HttpCode(HttpStatus.OK)
  @UseGuards(CombinedAuthGuard, RolesGuard)
  @Roles('admin', 'viewer')
  @ApiBearerAuth('Authorization')
  @ApiSecurity('userIdAuth')
  @ApiOperation({ summary: 'Get payment details for a registration' })
  @ApiParam({ name: 'registrationId', description: 'Registration ID', type: Number })
  async getPaymentDetails(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Res() res: Response,
  ) {
    try {
      const data = await this.paymentService.getPaymentDetails(registrationId);
      await this.responseService.success(res, 'Payment details retrieved successfully', data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Post('webhook')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Handle payment webhook from Razorpay' })
  @ApiBody({ description: 'Webhook data from Razorpay', type: Object })
  async handleWebhook(
    @Body() webhookData: any,
    @Req() req: Request,
    @Res() res: Response,
  ) {
    this.logger.log('Payment webhook received', webhookData);
    
    try {
      try {
        await this.paymentService.handleWebhook(webhookData, req.headers);
      } catch (error) {
        this.logger.error('Failed to process webhook', '', {
          error: error.message,
          webhookData,
        });
      }
      
      await this.responseService.success(res, 'Webhook processed successfully');
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Post('portal-webhook')
  @HttpCode(HttpStatus.OK)
  @UseGuards(CombinedAuthGuard, RolesGuard)
  @Roles('admin', 'viewer', 'finance_manager', 'shoba', 'relational_manager', 'operational_manger', 'rm_support')
  @ApiBearerAuth('Authorization')
  @ApiSecurity('userIdAuth')
  @ApiOperation({ summary: 'Handle portal payment webhook from Razorpay' })
  @ApiBody({ description: 'Portal webhook data from Razorpay', type: Object })
  async handlePortalWebhook(
    @Body() webhookData: any,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log('Portal payment webhook received', webhookData);
    
    try {
       try {
        await this.paymentService.handleWebhook(webhookData, req.headers, true);
      } catch (error) {
        this.logger.error('Failed to process webhook', '', {
          error: error.message,
          webhookData,
        });
      }
      await this.responseService.success(res, 'Webhook processed successfully');
    } catch (error) {
      handleControllerError(res, error);
    }
  }
  @Put(':registrationId/change-payment-request')
  @HttpCode(HttpStatus.OK)
  @UseGuards(CombinedAuthGuard, RolesGuard)
  @Roles('admin', 'finance_manager','relational_manager', 'rm_support')
  @ApiBearerAuth('Authorization')
  @ApiSecurity('userIdAuth')
  @ApiOperation({ summary: 'Update payment status for a registration' })
  @ApiParam({ name: 'registrationId', description: 'Registration ID', type: Number })
  // @ApiParam({ name: 'enableEdit', description: 'New payment status', type: String, example: PaymentEditRequestStatus.REQUESTED })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  async handlePaymentEdit(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    // @Param('enableEdit') enableEdit: PaymentEditRequestStatus,
    @Body() body: PaymentEditRequestDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log('Payment edit request received', { registrationId, body });

    try {
      const user = req.user;
      if (!user || !user.id) {
        throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, 'User not found in request');
      }
      const userId = user.id;

      await this.paymentService.updatePaymentEditStatus(registrationId, body.requestStatus, userId);

      await this.responseService.success(res, 'Payment edit status updated successfully');
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}