import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { ILike, IsNull, Repository } from 'typeorm';
import { OptionCategory, User } from 'src/common/entities';
import { CreateOptionCategoryDto } from './dto/create-option-category.dto';
import { UpdateOptionCategoryDto } from './dto/update-option-category.dto';
import { AppLoggerService } from 'src/common/services/logger.service';
import { optionCategoryMessages, userConstMessages } from 'src/common/constants/strings-constants';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import  InifniBadRequestException  from 'src/common/exceptions/infini-badrequest-exception';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { CommonDataService } from 'src/common/services/commonData.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';

@Injectable()
export class OptionCategoryRepository {
  constructor(
    @InjectRepository(OptionCategory)
    private readonly optionCategoryRepo: Repository<OptionCategory>,
    private readonly logger: AppLoggerService,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    private readonly commonDataService: CommonDataService,
  ) {}

    /**
   * Creates a new option category.
   * @param dto - Data transfer object containing category details.
   * @returns The created option category.
   */
  async create(dto: CreateOptionCategoryDto): Promise<OptionCategory> {
    const { name, createdBy, updatedBy } = dto;

    const [creator, updater] = await Promise.all([
      this.userRepo.findOne({ where: { id: createdBy } }),
      this.userRepo.findOne({ where: { id: updatedBy } }),
    ]);

    if (!creator) {
      this.logger.error(userConstMessages.USER_NOT_FOUND_ID(createdBy));
      throw new InifniNotFoundException(ERROR_CODES.OPTION_CATEGORY_UPDATOR_NOTFOUND,        null,
        null,
        createdBy.toString());
    }
    if (!updater) {
      this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
      throw new InifniNotFoundException(ERROR_CODES.OPTION_CATEGORY_UPDATOR_NOTFOUND,  null, null, updatedBy.toString());
    }

    const existingCategory = await this.optionCategoryRepo.findOne({ where: { name , deletedAt: IsNull()} });
    if (existingCategory) {
      this.logger.error(optionCategoryMessages.DUPLICATE_CATEGORY);
      throw new InifniBadRequestException(ERROR_CODES.OPTION_CATEGORY_DUPLICATE_BADREQUEST, null, null, name);
    }

    const category = new OptionCategory({ name, createdBy: creator, updatedBy: updater });
    try {
      return await this.commonDataService.save(this.optionCategoryRepo, category);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_SAVE_FAILED, error);
    }
  }

   /**
   * Retrieves all option categories with pagination and optional search functionality.
   * @param limit - Number of records per page.
   * @param offset - Offset for pagination.
   * @param searchText - Optional search text to filter categories by name.
   * @returns Paginated list of option categories.
   */
  async findAll(limit: number, offset: number, searchText: string) {
    try {
     const searchConditions = searchText ? { name: searchText } : {};

      const selectFields = ['id', 'name'];

      
      // Fetch data using CommonDataService
      const data = await this.commonDataService.get(
        this.optionCategoryRepo,
        selectFields as Array<keyof OptionCategory>,
        {deletedAt: IsNull()},
        limit,
        offset,
        { name: 'ASC' }, // Sort by name in ascending order
        searchConditions,
      );
  
      // Get total count of records
      const whereClause:any= {deletedAt: IsNull()};
      if (searchText) {
        whereClause.name =  ILike(`%${searchText}%`);
      }
      const total = await this.optionCategoryRepo.count({ where: whereClause });
      
      // Calculate total pages for pagination
      const totalPages = Math.ceil(total / limit);

      return {
        data,
        pagination: {
          totalPages,
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: limit,
          totalRecords: total,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_GET_FAILED, error);
    }
  }


  /**
   * Retrieves a single option category by its ID.
   * @param id - ID of the category to retrieve.
   * @returns The option category if found.
   * @throws NotFoundException if the category does not exist.
   */
  async findOneById(id: number): Promise<OptionCategory | null> {
    try {
      return await this.commonDataService.findOneById(this.optionCategoryRepo, id, true);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_FIND_BY_ID_FAILED, error);
    }
  }


    /**
   * Updates an existing option category.
   * @param id - ID of the category to update.
   * @param dto - Data transfer object containing updated category details.
   * @returns The updated option category.
   */
  async update(id: number, dto: UpdateOptionCategoryDto): Promise<OptionCategory> {
    const { name, updatedBy } = dto;

    const category = await this.commonDataService.findOneById(this.optionCategoryRepo, id, true);
    if (!category) {
      this.logger.warn(optionCategoryMessages.CATEGORY_NOT_FOUND_ID(id));
      throw new InifniNotFoundException(ERROR_CODES.OPTION_CATEGORY_NOTFOUND, null, null, id.toString());
    }

    const existingCategory = await this.optionCategoryRepo.findOne({ where: { name } });
    if (existingCategory && existingCategory.id !== id) {
      this.logger.error(optionCategoryMessages.DUPLICATE_CATEGORY);
      throw new InifniBadRequestException(ERROR_CODES.OPTION_CATEGORY_DUPLICATE_BADREQUEST, null, null, name || "");
    }

    Object.assign(category, dto);
    try {
      return await this.commonDataService.save(this.optionCategoryRepo, category);
    } catch (error) {
      this.logger.error(optionCategoryMessages.FAILED_UPDATING_CATEGORY(id), error);
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_SAVE_FAILED, error);
    }
  }

    /**
   * Deletes an option category by its ID.
   * @param id - ID of the category to delete.
   * @param user - User performing the deletion.
   * @returns The deleted option category.
   * @throws NotFoundException if the category does not exist.
   */
  async remove(id: number, user: User): Promise<void> {
    try {
      // Check if the category exists and is not already soft-deleted
      const category = await this.optionCategoryRepo.findOne({
        where: { id, deletedAt: IsNull() },
      });
  
      if (!category) {
        this.logger.warn(optionCategoryMessages.CATEGORY_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(ERROR_CODES.OPTION_CATEGORY_NOTFOUND, null, null, id.toString());
      }
  
      this.logger.log(optionCategoryMessages.SOFT_DELETING_CATEGORY(id));
       // update the `updatedBy` field
      category.updatedBy = user;
      await this.optionCategoryRepo.save(category);
  
      // Perform the soft delete
      await this.optionCategoryRepo.softDelete(id);
  
      this.logger.log(optionCategoryMessages.SOFT_DELETED_CATEGORY(id));
  
     
    } catch (error) {
      this.logger.error(optionCategoryMessages.FAILED_TO_DELETE_CATEGORY, error);
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_DELETE_FAILED, error);
    }
  }
}