import {
  Body,
  Controller,
  Get,
  HttpCode,
  HttpStatus,
  Post,
  Query,
  Req,
  Res,
  UseGuards,
  ValidationPipe,
} from '@nestjs/common';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiQuery, ApiResponse, ApiSecurity, ApiTags } from '@nestjs/swagger';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { Response } from 'express';
import { MessagesService } from './messages.service';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { CreateMessageDto } from './dto/create-message.dto';
import { messageConstMessages, SWAGGER_API_RESPONSE, paginationConstants } from 'src/common/constants/strings-constants';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('messages')
@Controller('messages')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'mahatria', 'rm', 'finance_manager', 'relational_manager', 'shoba', 'operational_manger', 'rm_support')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class MessagesController {
  constructor(
    private readonly service: MessagesService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: messageConstMessages.FIND_ALL_MESSAGES })
  @ApiQuery({ name: 'program', type: Number, required: true, description: 'Program ID' })
  @ApiQuery({ name: 'seeker', type: Number, required: false })
  @ApiQuery({ name: 'messenger', type: Number, required: false })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'sort', type: String, required: false, description: paginationConstants.SORT })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async findMessages(
    @Query('program') programId: number,
    @Query('seeker') seekerId: number,
    @Query('messenger') messengerId: number,
    @Query('limit') limit = 20,
    @Query('offset') offset = 0,
    @Query('sort') sort = 'DESC',
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(messageConstMessages.FIND_ALL_MESSAGES, { programId, seekerId, messengerId, limit, offset, sort });
    try {
      const userId = req.user?.id;
      const data = await this.service.findAll(
        Number(programId),
        userId,
        seekerId ? Number(seekerId) : undefined,
        messengerId ? Number(messengerId) : undefined,
        Number(limit),
        Number(offset),
        sort && sort.toUpperCase() === 'ASC' ? 'ASC' : 'DESC',
      );
      await this.responseService.success(res, messageConstMessages.MESSAGES_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('senders')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: messageConstMessages.FIND_ALL_SENDERS })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: paginationConstants.SEARCH_TEXT })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async findSenders(
    @Query('searchText') searchText = '',
    @Res() res: Response,
  ) {
    this.logger.log(messageConstMessages.FIND_ALL_SENDERS, { searchText });
    try {
      const data = await this.service.findSenders(searchText);
      await this.responseService.success(res, messageConstMessages.SENDERS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: messageConstMessages.CREATE_MESSAGE })
  @ApiBody({ type: CreateMessageDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  async create(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CreateMessageDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(messageConstMessages.CREATE_MESSAGE, dto);
    try {
      const userId = req.user?.id;
      const data = await this.service.create(dto, userId);
      await this.responseService.success(res, messageConstMessages.MESSAGE_CREATED, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
