import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { LookupData } from 'src/common/entities/lookup-data.entity';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { lookupDataConstMessages } from 'src/common/constants/strings-constants';
import { CommonStatus } from 'src/common/enum/common-status.enum';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';

@Injectable()
export class LookupDataRepository {
  constructor(
    @InjectRepository(LookupData)
    private readonly repo: Repository<LookupData>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Finds active lookup data by category.
   * @param {string} category - The lookup category to filter by.
   * @returns {Promise<object>} A promise resolving to lookup data with pagination.
   */
  async findActiveByCategory(category: string) {
    this.logger.log(lookupDataConstMessages.FINDING_BY_CATEGORY(category));
    try {
      const lookups = await this.commonDataService.findByData(
        this.repo,
        {
          lookupCategory: category,
          lookupStatus: CommonStatus.ACTIVE,
        },
        undefined,
        { lookupOrder: 'ASC' },
      );
      const data = lookups.map((item) => ({ id: item.id, key: item.lookupKey, label: item.lookupLabel }));
      const total = data.length;
      return {
        data,
        pagination: {
          totalPages: 1,
          pageNumber: 1,
          pageSize: total + 1,
          totalRecords: total,
          numberOfRecords: total,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }

  /**
   * Finds all active lookup categories.
   * @returns {Promise<object>} A promise resolving to active categories with pagination.
   */
  async findActiveCategories() {
    this.logger.log(lookupDataConstMessages.FINDING_CATEGORIES);
    try {
      const cats = await this.repo
        .createQueryBuilder('lookup')
        .select(['DISTINCT lookup.id AS id', 'lookup.lookupCategory AS key', 'lookup.lookupCategoryName AS label'])
        .where('lookup.lookupStatus = :status', { status: CommonStatus.ACTIVE })
        .orderBy('lookup.lookupCategoryName', 'ASC')
        .getRawMany();
      const data = cats.map((item) => ({
        id: item.id,
        key: item.key,
        label: item.label,
      }));

      const total = data.length;
      return {
        data,
        pagination: {
          totalPages: 1,
          pageNumber: 1,
          pageSize: total + 1,
          totalRecords: total,
          numberOfRecords: total,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }

  /**
   * Finds all active lookup data grouped by category.
   * @returns {Promise<Record<string, object[]>>} A promise resolving to grouped lookup data.
   */
  async findAllActiveGrouped() {
    this.logger.log(lookupDataConstMessages.FINDING_ALL);
    try {
      const lookups = await this.commonDataService.findByData(
        this.repo,
        {
          lookupStatus: CommonStatus.ACTIVE,
        },
        undefined,
        { lookupOrder: 'ASC' },
      );

      return lookups.reduce(
        (acc, item) => {
          if (!acc[item.lookupCategory]) {
            acc[item.lookupCategory] = [];
          }
          acc[item.lookupCategory].push({
            id: item.id,
            key: item.lookupKey,
            value: item.lookupLabel,
          });
          return acc;
        },
        {} as Record<string, { id: number; key: string; value: string }[]>,
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }
  /**
   * Finds active lookup data by category and label.
   * @param {string} category - The lookup category to filter by.
   * @param {string} label - The lookup label to filter by.
   * @returns {Promise<object | undefined>} A promise resolving to the lookup data or undefined.
   */
  async findActiveByCategoryAndLabel(category: string, label: string) {
    this.logger.log(`Finding lookup data by category: ${category} and label: ${label}`);
    try {
      const lookup = await this.repo.findOne({
        where: {
          lookupCategory: category,
          lookupLabel: label,
          lookupStatus: CommonStatus.ACTIVE,
        },
        order: { lookupOrder: 'ASC' },
      });
      if (!lookup) return undefined;
      return {
        id: lookup.id,
        key: lookup.lookupKey,
        label: lookup.lookupLabel,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }
}