import { Controller, Get, HttpCode, HttpStatus, Query, Res, UseGuards } from '@nestjs/common';
import { Response } from 'express';
import { LookupDataService } from './lookup-data.service';
import { ApiBearerAuth, ApiOperation, ApiQuery, ApiResponse, ApiSecurity, ApiTags } from '@nestjs/swagger';
import { lookupDataConstMessages, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('lookup-data')
@Controller('lookup-data')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'viewer', 'finance_manager', 'relational_manager', 'shoba', 'rm', 'operational_manger', 'rm_support','mahatria')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class LookupDataController {
  constructor(
    private readonly service: LookupDataService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: lookupDataConstMessages.GET_LOOKUP_BY_CATEGORY })
  @ApiQuery({ name: 'category', type: String, required: true, description: lookupDataConstMessages.CATEGORY_QUERY })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async getByCategory(@Query('category') category: string, @Res() res: Response) {
    this.logger.log(lookupDataConstMessages.FINDING_BY_CATEGORY(category));
    try {
      const data = await this.service.findActiveByCategory(category);
      await this.responseService.success(res, lookupDataConstMessages.LOOKUP_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('all')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: lookupDataConstMessages.GET_ALL })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async getAll(@Res() res: Response) {
    this.logger.log(lookupDataConstMessages.GET_ALL);
    try {
      const data = await this.service.findAllWithRM();
      await this.responseService.success(res, lookupDataConstMessages.LOOKUP_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('banks')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: lookupDataConstMessages.GET_BANKS })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async getBankNames(@Res() res: Response) {
    this.logger.log(lookupDataConstMessages.GET_BANKS);
    try {
      const data = await this.service.findActiveByCategory('BANK_NAME');
      await this.responseService.success(res, lookupDataConstMessages.LOOKUP_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('cities')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: lookupDataConstMessages.GET_CITIES })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async getCityNames(@Res() res: Response) {
    this.logger.log(lookupDataConstMessages.GET_CITIES);
    try {
      const data = await this.service.findActiveByCategory('CITY_NAME');
      await this.responseService.success(res, lookupDataConstMessages.LOOKUP_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('categories')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: lookupDataConstMessages.GET_CATEGORIES })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  async getCategories(@Res() res: Response) {
    this.logger.log(lookupDataConstMessages.GET_CATEGORIES);
    try {
      const data = await this.service.findCategories();
      await this.responseService.success(res, lookupDataConstMessages.CATEGORIES_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
