import { IsEmail, IsString, IsOptional, IsArray, ValidateNested, IsObject, IsNumber } from 'class-validator';
import { Type } from 'class-transformer';
import { User } from 'src/common/entities';
import { CommunicationTypeEnum } from 'src/common/enum/communication-type.enum';
import { CommunicationTemplatesKeysEnum } from 'src/common/enum/communication-template-keys.enum';
import { ApiProperty } from '@nestjs/swagger';

export class EmailRecipient {
  @IsEmail()
  emailAddress: string;

  @IsOptional()
  @IsString()
  name?: string;

  @IsOptional()
  @IsObject()
  mergeInfo?: Record<string, any>;
}

export class EmailSender {
  @IsEmail()
  address: string;
  @IsOptional()
  @IsString()
  name?: string;
}

export class EmailAttachments {
  content: any;

  @IsString()
  mime_type: string;

  @IsString()
  name: string;
}

export class TrackInfo {

  @IsNumber()
  registrationId: number; 

  @IsNumber()
  @IsOptional()
  createdBy?: number;

  @IsNumber()
  @IsOptional()
  updatedBy?: number;
}

export class SendSingleEmailDto {
  @ValidateNested()
  @Type(() => EmailRecipient)
  to: EmailRecipient;

  @ValidateNested()
  @Type(() => EmailSender)
  from: EmailSender;

  @IsString()
  subject: string;

  @IsOptional()
  @IsString()
  htmlbody?: string;

  @IsOptional()
  @IsString()
  textbody?: string;

  @IsOptional()
  @IsString()
  templateKey?: string;

  @IsOptional()
  @IsObject()
  mergeInfo?: Record<string, any>;

  @IsOptional()
  attachments?: EmailAttachments[];

  @ValidateNested()
  @Type(() => EmailRecipient)
  @IsOptional()
  trackinfo: TrackInfo;

  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => EmailRecipient)
  cc?: EmailRecipient[];

  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => EmailRecipient)
  bcc?: EmailRecipient[];

}

export class SendBulkEmailDto {
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => EmailRecipient)
  to: EmailRecipient[];

  @ValidateNested()
  @Type(() => EmailSender)
  from: EmailSender;

  @IsString()
  subject: string;

  @IsOptional()
  @IsString()
  htmlbody?: string;

  @IsOptional()
  @IsString()
  textbody?: string;

  @IsOptional()
  @IsString()
  templateKey?: string;

  @IsOptional()
  @IsObject()
  mergeInfo?: Record<string, any>;

  @IsOptional()
  attachments?: EmailAttachments[];
}

export class EmailWebhookDto {
  @IsString()
  event: 'open' | 'click' | 'bounce' | 'delivered' | 'sent' | 'spam' | 'unsubscribed';

  @IsEmail()
  email: string;

  @IsString()
  subject: string;

  @IsString()
  timestamp: string;

  @IsOptional()
  @IsString()
  message_id?: string;

  @IsOptional()
  @IsString()
  bounce_type?: string;

  @IsOptional()
  @IsString()
  bounce_reason?: string;

  @IsOptional()
  @IsString()
  click_url?: string;

  @IsOptional()
  @IsObject()
  additionalInfo?: Record<string, any>;
}

export class BulkCommunicationDto {
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => SendSingleEmailDto)
  communicationTo: string;

  programId: number;

  @ApiProperty({
    description: 'ID of the program to which the email communication is allocated',
    example: 1,
  })
  @IsNumber()
  allocatedProgramId: number;

  @ApiProperty({
    description: 'Key of the communication template used for the email',
    example: CommunicationTemplatesKeysEnum.FIRST_TIMER,
  })
  @IsString()
  templateKey: CommunicationTemplatesKeysEnum;

  @ApiProperty({
    description: 'Type of the communication template',
    example: CommunicationTypeEnum.EMAIL,
  })
  @IsString()
  templateType: CommunicationTypeEnum;
}

