import { Injectable, Logger } from '@nestjs/common';
import { SendSingleEmailDto, SendBulkEmailDto, BulkCommunicationDto } from './dto/email-communication.dto';
import {
  SendTemplateMessageDto,
  SendBulkTemplateMessageDto,
} from './dto/whatsapp-communication.dto';
import { SendMailClient } from 'zeptomail';
import axios from 'axios';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { delay, chunkArray } from 'src/common/utils/common.util';
import { CommunicationRepository } from './communication.repository';
import { CommunicationTypeEnum } from 'src/common/enum/communication-type.enum';
import { CommunicationStatusEnum } from 'src/common/enum/communication-status.enum';
import { CreateCommunicationTrackDto } from './dto/communication-track.dto';
import { CommunicationTemplatesKeysEnum } from 'src/common/enum/communication-template-keys.enum';
import { CommunicationTemplatesService } from './service/communication-templates.service';
import { constants } from 'buffer';
import { zeptoEmailCreadentials } from 'src/common/constants/strings-constants';

@Injectable()
export class CommunicationService {
  private readonly logger = new Logger(CommunicationService.name);
  private readonly emailApiKey = process.env.ZEPTO_EMAIL_TOKEN;
  private readonly emailApiUrl = process.env.ZEPTO_EMAIL_URL;
  private readonly emailWebhookSecret = process.env.ZEPTO_EMAIL_WEBHOOK_SECRET;
  private readonly whatsappApiUrl = process.env.WATI_BASE_URL;
  private readonly whatsappAccessToken = process.env.WATI_TOKEN;
  private readonly whatsappWebhookSecret = process.env.WATI_WEBHOOK_SECRET;

  private readonly client: SendMailClient;
  private readonly bulkClient: SendMailClient;

  constructor(
    private readonly trackRepo: CommunicationRepository,
    private readonly templatesService: CommunicationTemplatesService,
    // private readonly httpService: HttpService,
  ) {
    this.logger.log('Initializing CommunicationService', {
      url: process.env.ZEPTO_EMAIL_URL,
      token: process.env.ZEPTO_EMAIL_TOKEN,
    });
    this.client = new SendMailClient({
      url: process.env.ZEPTO_EMAIL_URL,
      token: process.env.ZEPTO_EMAIL_TOKEN,
    });
    this.bulkClient = new SendMailClient({
      url: process.env.ZEPTO_EMAIL_URL + '/batch',
      token: process.env.ZEPTO_EMAIL_TOKEN,
    });
    // Validate required configurations
    if (!this.emailApiKey) {
      this.logger.warn('ZeptoMail API key is not configured');
    }
    if (!this.whatsappApiUrl || !this.whatsappAccessToken) {
      this.logger.warn('WATI API URL or access token is not configured');
    }
  }

  // ==================== EMAIL METHODS ====================

  /**
   * Send a single email using ZeptoMail API
   */
  async sendSingleEmail(emailData: SendSingleEmailDto): Promise<any> {
    if (process.env.ENABLE_EMAIL !== 'true') {
      this.logger.log('Email sending is disabled in the environment configuration');
      return;
    }

    try {
      this.logger.log(
        `Sending single email to: ${emailData.templateKey} , ${emailData.to.emailAddress}, ${emailData.from.address}`,
      );

      const mailPayload: any = {
        template_key: emailData.templateKey,
        from: {
          address: emailData.from.address,
          name: emailData.from.name || 'Infinitheism',
        },
        to: [
          {
            email_address: {
              address: emailData.to.emailAddress,
              name: emailData.to.name || '',
            },
          },
        ],
        merge_info: JSON.stringify(emailData.mergeInfo),
        subject: 'Test Email',
        attachments: emailData.attachments || [],
      };

      // Add cc if provided
      if (emailData.cc && emailData.cc.length > 0) {
        mailPayload.cc = emailData.cc.map((ccRecipient) => ({
          email_address: {
            address: ccRecipient.emailAddress,
            name: ccRecipient.name || '',
          },
        }));
      }

      // Add bcc if provided
      if (emailData.bcc && emailData.bcc.length > 0) {
        mailPayload.bcc = emailData.bcc.map((bccRecipient) => ({
          email_address: {
            address: bccRecipient.emailAddress,
            name: bccRecipient.name || '',
          },
        }));
      }

      this.logger.log(`Preparing to send email with payload: ${JSON.stringify(mailPayload)}`);
      const response = await this.client.sendMailWithTemplate(mailPayload);
      //this.logger.log(`Email sent successfully to: ${emailData.to.emailAddress}`, { response });

      if (emailData.trackinfo && emailData.trackinfo.registrationId) {
        const trackData: CreateCommunicationTrackDto = {
          registrationId: emailData.trackinfo.registrationId,
          typ: CommunicationTypeEnum.EMAIL,
          stats: CommunicationStatusEnum.SUCCESS,
          failedReason: response || '',
        };

        if (emailData.trackinfo.createdBy) {
          trackData.createdBy = emailData.trackinfo.createdBy;
        }

        if (emailData.trackinfo.updatedBy) {
          trackData.updatedBy = emailData.trackinfo.updatedBy;
        }
        await this.updateCommunicationTrack(trackData);
      }
      return response.data;
    } catch (error) {
      this.logger.error(`Failed to send email: ${error}`, error.stack);
      this.logger.error(JSON.stringify(error), 'error while sending mail');
      if (emailData.trackinfo) {
        const trackData: CreateCommunicationTrackDto = {
          registrationId: emailData.trackinfo.registrationId,
          typ: CommunicationTypeEnum.EMAIL,
          stats: CommunicationStatusEnum.FAILED,
          failedReason: error.message || 'Unknown error',
        };

        if (emailData.trackinfo.createdBy) {
          trackData.createdBy = emailData.trackinfo.createdBy;
        }

        if (emailData.trackinfo.updatedBy) {
          trackData.updatedBy = emailData.trackinfo.updatedBy;
        }
        await this.updateCommunicationTrack(trackData);
        // await this.trackRepo.createEntity(trackData)
      }
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_EMAIL, error);
    }
  }

  /**
   * Send bulk emails using ZeptoMail API
   */
  async sendBulkEmail(emailData: SendBulkEmailDto): Promise<any> {
    // if (!this.emailApiKey) {
    //   throw new BadRequestException('Email service is not configured');
    // }
    if (process.env.ENABLE_EMAIL !== 'true') {
      this.logger.warn('Email sending is disabled in the environment configuration');
      return;
    }
    try {
      this.logger.log(`Sending bulk email to ${emailData.to} recipients`);

      const toRecipients = emailData.to.map((recipient) => ({
        email_address: {
          address: recipient.emailAddress,
          name: recipient.name || '',
        },
        merge_info: JSON.stringify(recipient.mergeInfo || {}),
      }));
      this.logger.log(`Preparing to send bulk email to ${JSON.stringify(toRecipients)} recipients`);

      const response = await this.bulkClient.sendMailWithTemplate({
        template_key: emailData.templateKey,
        from: {
          address: emailData.from.address,
          name: emailData.from.name || 'Infinitheism',
        },
        to: toRecipients,
        // merge_info: JSON.stringify(emailData.mergeInfo),
        subject: emailData.subject,
        attachments: emailData.attachments || [],
      });

      this.logger.log(`Bulk email sent successfully to ${emailData.to.length} recipients`);
      return response.data;
    } catch (error) {
      this.logger.log(`Failed to send bulk email: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_BULK_EMAIL, 'Failed to send bulk email');
    }
  }

  async updateCommunicationTrack(trackInfo: CreateCommunicationTrackDto): Promise<void> {
    try {
      this.logger.log(
        `Updating communication track info for registration ID: ${trackInfo.registrationId}`,
      );

      await this.trackRepo.createEntity(trackInfo);
      this.logger.log(
        `communication track info updated successfully for registration ID: ${trackInfo.registrationId}`,
      );
    } catch (error) {
      this.logger.error(`Failed to update communication track info: ${error.message}`, error.stack);
    }
  }

  /**
   * Validate ZeptoMail webhook signature
   */
  validateEmailWebhookSignature(payload: string, signature: string): boolean {
    try {
      if (!this.emailWebhookSecret) {
        this.logger.warn('Email webhook secret not configured, skipping validation');
        return true; // Allow if no secret configured
      }

      const crypto = require('crypto');
      const expectedSignature = crypto
        .createHmac('sha256', this.emailWebhookSecret)
        .update(payload)
        .digest('hex');

      const providedSignature = signature.replace('sha256=', '');

      return crypto.timingSafeEqual(
        Buffer.from(providedSignature, 'hex'),
        Buffer.from(expectedSignature, 'hex'),
      );
    } catch (error) {
      this.logger.error(`Email webhook signature validation failed: ${error.message}`);
      return false;
    }
  }

  // ==================== WHATSAPP METHODS ====================

  /**
   * Send WhatsApp template message using WATI API
   */
  async sendTemplateMessage(messageData: SendTemplateMessageDto): Promise<any> {
    if (process.env.ENABLE_WHATSAPP !== 'true') {
      this.logger.warn('WhatsApp sending is disabled in the environment configuration');
      return;
    }
    try {
      this.logger.log(`Sending WhatsApp template message to: ${messageData.whatsappNumber}`);
      const url = `${process.env.WATI_BASE_URL}/sendTemplateMessage?whatsappNumber=${messageData.whatsappNumber}`;
      const payload = {
        template_name: messageData.templateName,
        broadcast_name: messageData.broadcastName,
        parameters:
          messageData.parameters?.map((param) => ({
            name: param.name,
            value: param.value,
          })) || [],
      };
      const headers = {
        Authorization: `Bearer ${process.env.WATI_TOKEN}`,
        'Content-Type': 'application/json',
      };

      const response = await axios.post(url, payload, { headers });

      this.logger.log(
        `WhatsApp template message sent successfully to: ${messageData.whatsappNumber} with response: ${JSON.stringify(response.data)}`,
      );
      if (messageData.trackinfo) {
        const trackData: CreateCommunicationTrackDto = {
          registrationId: messageData.trackinfo.registrationId,
          typ: CommunicationTypeEnum.WHATSAPP,
          stats: CommunicationStatusEnum.SUCCESS,
          failedReason: JSON.stringify(response.data) || '',
        };

        if (messageData.trackinfo.createdBy) {
          trackData.createdBy = messageData.trackinfo.createdBy;
        }

        if (messageData.trackinfo.updatedBy) {
          trackData.updatedBy = messageData.trackinfo.updatedBy;
        }
        this.updateCommunicationTrack(trackData);
      }
      return response.data;
    } catch (error) {
      this.logger.error(`Failed to send WhatsApp template message: ${error.message}`, error.stack);
      if (messageData.trackinfo) {
        const trackData: CreateCommunicationTrackDto = {
          registrationId: messageData.trackinfo.registrationId,
          typ: CommunicationTypeEnum.WHATSAPP,
          stats: CommunicationStatusEnum.FAILED,
          failedReason: error.message || 'Unknown error',
        };

        if (messageData.trackinfo.createdBy) {
          trackData.createdBy = messageData.trackinfo.createdBy;
        }

        if (messageData.trackinfo.updatedBy) {
          trackData.updatedBy = messageData.trackinfo.updatedBy;
        }
        this.updateCommunicationTrack(trackData);
      }
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_WHATSAPP_MESSAGE, error);
    }
  }

  /**
   * Send bulk WhatsApp template messages using WATI API
   */
  async sendBulkTemplateMessage(messageData: SendBulkTemplateMessageDto): Promise<any> {

    if (process.env.ENABLE_WHATSAPP !== 'true') {
      this.logger.warn('WhatsApp sending is disabled in the environment configuration');
      return;
    }
    try {
      this.logger.log(
        `Sending bulk WhatsApp template messages to ${messageData.recipients.length} recipients`,
      );

      const results: { status: string; [key: string]: any }[] = [];
      const failedRecipients: { whatsappNumber: string; name?: string; error: any }[] = [];

      // Process recipients in batches to avoid rate limiting
      const batchSize = 10; // Adjust based on WATI rate limits
      const batches = chunkArray(messageData.recipients, batchSize);
      const watiBaseUrl = process.env.WATI_BASE_URL;
      const watiApiKey = process.env.WATI_TOKEN;
      const url = `${watiBaseUrl}/sendTemplateMessages`;
      for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
        const batch = batches[batchIndex];
        this.logger.log(
          `Processing batch ${batchIndex + 1}/${batches.length} with ${batch.length} recipients`,
        );

        const batchPromises = batch.map(async (recipient) => {
          try {
            const payload = {
              template_name: messageData.templateName,
              broadcast_name: messageData.broadcastName,
              receivers: 
              [
                recipient,
                ...(messageData.globalParameters || []),
              ]
            };

            const headers = {
              Authorization: `Bearer ${watiApiKey}`,
              'Content-Type': 'application/json',
            };
            this.logger.log(
              `Sending template message to ${recipient.whatsappNumber} with payload: ${JSON.stringify(payload)}`,
            );


            const response = await axios.post(url, payload, { headers });
            this.logger.log(
              `WhatsApp template message sent successfully to: ${recipient.whatsappNumber} and ${JSON.stringify(response.data)}`,
            );

            return {
              whatsappNumber: recipient.whatsappNumber,
              name: recipient,
              status: 'success',
              response: response.data,
            };
          } catch (error) {
            this.logger.error(
              `Failed to send template message to ${recipient.whatsappNumber}: ${error.message}`,
            );
            failedRecipients.push({
              whatsappNumber: recipient.whatsappNumber,
              // name: recipient,
              error: error.response?.data?.message || error.message,
            });
            return {
              whatsappNumber: recipient.whatsappNumber,
              name: recipient,
              status: 'failed',
              error: error.response?.data?.message || error.message,
            };
          }
        });

        const batchResults = await Promise.allSettled(batchPromises);
        results.push(
          ...batchResults
            .map((result) => (result.status === 'fulfilled' ? result.value : null))
            .filter((item) => item !== null),
        );

        // Add delay between batches to respect rate limits
        if (batchIndex < batches.length - 1) {
          await delay(1000); // 1 second delay between batches
        }
      }

      const successCount = results.filter((r) => r && r.status === 'success').length;
      const failedCount = failedRecipients.length;

      this.logger.log(
        `Bulk WhatsApp template messages completed: ${successCount} successful, ${failedCount} failed`,
      );

      return {
        total: messageData.recipients.length,
        successful: successCount,
        failed: failedCount,
        results,
        failedRecipients,
      };
    } catch (error) {
      this.logger.error(
        `Failed to send bulk WhatsApp template messages: ${error.message}`,
        error.stack,
      );
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_BULK_WHATSAPP_MESSAGE, error);
    }
  }
  

  /**
   * Validate WATI webhook token
   */
  validateWhatsAppWebhookToken(providedToken: string): boolean {
    if (!this.whatsappWebhookSecret) {
      this.logger.warn('WhatsApp webhook secret not configured, skipping validation');
      return true; // Allow if no secret configured
    }

    return providedToken === this.whatsappWebhookSecret;
  }

  /** ==== Service methods for communication ==== **/

  async sendBulkCommunication(bulkData: BulkCommunicationDto): Promise<any> {
    try {
      this.logger.log(`Sending bulk communication for program ID: ${bulkData.programId}`);
      const { allocatedProgramId, templateKey, templateType, communicationTo } = bulkData;

      // Validate required fields
      if (!allocatedProgramId || !templateKey || !templateType || !communicationTo) {
        throw new Error('Missing required fields for bulk communication');
      }
    } catch (error) {
      this.logger.error(`Failed to send bulk communication: ${error.message}`, error.stack);
      // handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_BULK_COMMUNICATION, error);
    }
  }
  async bulkCommunicationData(data: any, templateType: CommunicationTypeEnum, templateKey: CommunicationTemplatesKeysEnum) {
    try {
      if (templateType === CommunicationTypeEnum.EMAIL) {
        const templateId = await this.templatesService.getTemplateByKeyAndType(templateKey, templateType);
        if (templateKey === CommunicationTemplatesKeysEnum.FIRST_TIMER) {
          await this.sendFirstTimerEmailCommunication(data, templateId.templateId);
        } else if (templateKey === CommunicationTemplatesKeysEnum.CHECKLIST) {
          await this.sendChecklistEmailCommunication(data, templateId.templateId);
        } else if (templateKey === CommunicationTemplatesKeysEnum.PREPARATORY) {
          await this.sendPreparatoryEmailCommunication(data, templateId.templateId);
        } else {
          handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_NOTFOUND, new Error(`No specific handler for template key: ${templateKey}`));
        }
      }
    } catch (error) {
      this.logger.error(`Failed to process bulk communication data: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_FIND_BY_KEY_AND_TYPE_FAILED, error);
    }
    
  }
  async sendFirstTimerEmailCommunication(data: any, templateId): Promise<any> {
    try {
      const emailData: SendBulkEmailDto = {
        to: data.map((item) => ({
          emailAddress: item.emailAddress,
          name: item.fullName,
          mergeInfo: {
            reg_name: item.fullName
          }
        })),
        templateKey: templateId,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name: data[0]?.program?.emailSenderName ?? zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        subject: 'First Timer Communication',
      }
      await this.sendBulkEmail(emailData);
    } catch (error) {
      this.logger.error(`Failed to send first timer communication: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_FIRST_TIMER_COMMUNICATION, error);
    }
  }
  async sendChecklistEmailCommunication(data: any, templateId): Promise<any> {
    try {
      const emailData: SendBulkEmailDto = {
        to: data.map((item) => ({
          emailAddress: item.emailAddress,
          name: item.fullName,
          mergeInfo: {
            reg_name: item.fullName
          }
        })),
        templateKey: templateId,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name: data[0]?.program?.emailSenderName ?? zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        subject: 'Checklist Communication',
      }
      await this.sendBulkEmail(emailData);
    } catch (error) {
      this.logger.error(`Failed to send checklist communication: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_CHECKLIST_COMMUNICATION, error);
    }
  }
  async sendPreparatoryEmailCommunication(data: any, templateId): Promise<any> {
    try {
      const emailData: SendBulkEmailDto = {
        to: data.map((item) => ({
          emailAddress: item.emailAddress,
          name: item.fullName,
          mergeInfo: {
            reg_name: item.fullName,
            meet_id:'',
            meet_time: '',
            meet_pass: '',
            hdb_msd: '',
            meet_link: '',
            meet_date: '',
          }
        })),
        templateKey: templateId,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name: data[0]?.program?.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        subject: 'Preparatory Communication',
      }
      await this.sendBulkEmail(emailData);
    } catch (error) {
      this.logger.error(`Failed to send preparatory communication: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_PREPARATORY_COMMUNICATION, error);
    }
  }
}
