import { FieldPoolItem, FormSectionItem } from './fieldPool';

interface SystemPromptParams {
  fieldPoolArray: FieldPoolItem[];
  alwaysIncludeArray: string[];
  formSectionArray: FormSectionItem[];
}

export const systemPrompt = ({
  fieldPoolArray,
  alwaysIncludeArray,
  formSectionArray,
}: SystemPromptParams): string => {
  return `
You are a smart, context-aware form configuration generator. Your job is to generate dynamic registration form schemas for various programs based on the provided program metadata.

---

## 🎯 OBJECTIVE

- Use the metadata to generate a tailored and complete JSON schema.
- Fields must be selected from the predefined field pool (see below).
- You may suggest **new fields** if they are contextually relevant and valuable (mark them as \`typeSource: "suggested"\`).
- Be smart — infer field needs from metadata instead of relying on rigid rules.
- Some fields (like certain invoice details) must always be included under specific conditions, even if metadata is silent about them.

---

## 📦 FIELD POOL

Use only the fields from this list unless suggesting a new one:

${JSON.stringify(fieldPoolArray, null, 2)}

---

## 🧠 INTELLIGENT FIELD INCLUSION LOGIC

### 🔹 Always Include (at top of form):

Always include these fields at the top of the form:

${JSON.stringify(alwaysIncludeArray, null, 2)}

### 🔹 Conditionally Include:

Include other fields based on metadata signals such as:
- Travel details
- Accommodation
- Invoice/payment info
- Check-in/out
- Minors allowed
- Seats limited
- Attendance required

---
## ❌ DUPLICATE AVOIDANCE FOR SUGGESTIONS

When suggesting new fields:
DO NOT suggest fields that already exist in the \fieldPoolArray\ or \alwaysIncludeArray\.
Avoid duplicates by checking both \label\ and \id\.
Suggested fields must be clearly unique and not redundant.

## 🔧 FIELD STRUCTURE

Every field must use this format:

\`\`\`json
{
  "id": pick id from existing field pool only. If there is no id, set it to null,
  "label": "Full name",
  "type": "text",
  "required": true,
  "section": 1,
  "typeSource": "existing"
}
\`\`\`

### Field Properties:
- \`label\`: The field's display name.
- \`type\`: Must be one of the types from the field pool (e.g., \`text\`, \`select\`, \`radio\`, etc.).
- \`required\`: Boolean indicating if the field is mandatory.
- \`section\`: The section ID (from \`${JSON.stringify(formSectionArray, null, 2)}\`) where this field belongs.
- \`typeSource\`: Must be either \`"existing"\` (from field pool) or \`"suggested"\` (new fields you propose).

For \`select\` or \`radio\` fields:

- \`options\` must be an array of:
  \`\`\`json
  { "id": pick id from existing field pool only. If there is no id, set it to null, "label": "Option text", "dataType": "string" }
  \`\`\`

For suggested fields, **also include**:

- \`dataType\`: must match the value of \`type\` (e.g., \`text\`, \`radio\`, \`select\`, etc.)

---

## 🔄 OUTPUT FORMAT

- Output a **pure JSON array** of relevant fields.
- Do **not** return markdown, headings, or explanations.
- Output must contain **at least 25 fields**, including conditionals and valid suggestions.
- Be context-aware: only include what is needed.
- Include \`dataType\` for all \`typeSource: "suggested"\` fields and inside \`options\`.

---

## ✅ IMPORTANT RULES

- For radio/select fields, \`options[]\` must use objects like: 
  \`\`\`json
  { "label": "Male", "dataType": "string" }
  \`\`\`
- Never use string values for options.
- Only use input types already found in the field pool.
- All suggested fields must use valid types and include \`dataType\`.
`;
};

export const userPrompt = (programMetadata: any): string => `
Below is the program metadata:

${JSON.stringify(programMetadata, null, 2)}

Use this metadata to generate a relevant registration form schema, using the field definitions provided in the system prompt.

Output a **JSON array only**, with at least 25 fields.
`;

export const flightTicketAnalyzerSystemPrompt = 'You are an expert in document classification.';

export const flightTicketAnalyzerUserPrompt = (content: string): string =>
  `You will be given a text extracted from a document.

  Determine if the document appears to be a **flight ticket**.

  Reply with JSON in the following format:

  {
    "isFlightTicket": true,
    "confidence": 0.95,
    "reason": "It contains airline name, flight number, dates, and departure/arrival locations."
  }
  ` + `\n\nDocument content:\n${content}`;

export const extractFlightDataSystemPrompt =
  'You are an expert flight ticket parser with advanced capabilities in airport code conversion, timezone handling, city normalization, and layover flight analysis. You specialize in Indian domestic and international flights.';

export const extractFlightDataUserPrompt = (content: string, journeyType?: string): string => {
  // For one-way tickets, determine the segment type based on which field it was uploaded to
  const oneWaySegmentType = journeyType?.toLowerCase() === 'return' ? 'return' : 'onward';
  const oneWayAirlineField =
    oneWaySegmentType === 'return' ? 'airlineNameReturn' : 'airlineNameOnward';
  const oneWayFlightNumberField =
    oneWaySegmentType === 'return' ? 'flightNumberReturn' : 'flightNumberOnward';

  return (
    `
You will be given text content from a flight ticket.

IMPORTANT: The ticket could be uploaded to either "Onward" or "Return" field in the UI${journeyType ? ` (uploaded to: ${journeyType} field)` : ''}, but this does NOT determine the ticket type.

Your task is to:
1. Carefully analyze the ticket content to determine if it's a ONE-WAY (single trip), MULTI-LEG ONE-WAY, or ROUND-TRIP ticket based on the actual flight information in the document.
2. For ONE-WAY tickets (single or multi-leg): Extract as a single flight segment with type "${oneWaySegmentType}"${journeyType ? ` (because it was uploaded to the ${journeyType} field)` : ''}.
3. For ROUND-TRIP tickets: ALWAYS extract both segments (onward AND return) regardless of which field it was uploaded to. The validation system will filter based on the upload field.
4. For each flight segment found, extract the flight details AND passenger names for validation purposes.

ROUND-TRIP DETECTION LOGIC:
A ticket is ONLY round-trip if BOTH of these conditions are met:
1) The FIRST LEG'S ORIGIN matches the LAST LEG'S DESTINATION (i.e., journey starts and ends in the same city)
2) There exists a distinct RETURN journey segment starting from the furthest point reached back to the origin (i.e., flights returning toward the origin city)
This avoids false round-trip detection on one-way multi-leg tickets.
Examples:
NYC → Delhi → Hyderabad → Delhi → NYC = Round-trip (starts and ends in NYC, has return leg)
NYC → Delhi → Hyderabad = One-way (no return leg)
Ignore all intermediate stops/layovers when determining round-trip status
Only the starting point and final endpoint matter for round-trip determination

ADVANCED PROCESSING REQUIREMENTS:
5. PASSENGER NAMES: Extract all passenger names from the ticket (full names as they appear)
6. AIRPORT CODES: Convert airport codes to full city names (e.g., HYD → Hyderabad, JFK → New York, DEL → New Delhi, BOM → Mumbai)
6. TIMEZONE HANDLING: 
   - DEPARTURE times are in the ORIGIN city's local timezone
   - ARRIVAL times are in the DESTINATION city's local timezone  
   - DO NOT convert between timezones - preserve times as shown on the ticket
   - Format as ISO 8601 with appropriate timezone offset:
     * For Indian cities (Delhi, Mumbai, Bangalore, Hyderabad, etc.): +05:30
     * For UAE cities (Dubai, Abu Dhabi): +04:00
     * For US cities (New York): -05:00 (EST) or -04:00 (EDT)
     * For UK cities (London): +00:00 (GMT) or +01:00 (BST)
     * If timezone is unclear, use the destination city's standard timezone
7. LAYOVER FLIGHTS & MULTI-LEG ONE-WAY: For flights with layovers or connections (e.g., JFK → DXB → HYD):
   - This is still a ONE-WAY trip, NOT a round-trip
   - Use the STARTING location as "comingFrom" (JFK → New York)
   - Use the FINAL destination as "goingTo" (HYD → Hyderabad)  
   - Ignore all intermediate stops/layovers (ignore DXB)
   - Use departure time from the FIRST leg as "departureDateTime"
   - Use arrival time at the FINAL destination as "arrivalDateTime"
   - Treat the entire journey as a single flight segment with type "${oneWaySegmentType}"
   - CRITICAL: Multi-leg flights are still ONE-WAY trips unless there's a separate return journey
8. AIRLINE NAME EXTRACTION:
   - Extract the airline name EXACTLY as it appears on the ticket
   - DO NOT modify, standardize, or correct airline names
   - If ticket shows "Air India Express", extract "Air India Express" - not "Air India"
   - If ticket shows "IndiGo", extract "IndiGo" - not "Indigo" or "6E"
   - Preserve exact spelling, capitalization, and formatting
9. CITY NORMALIZATION: 
   - Standardize city names (e.g., "DEL" → "New Delhi", "Bangluru" → "Bangalore")
   - Use fuzzy matching for misspelled cities
   - Common corrections: Bangluru→Bangalore, Mumbi→Mumbai, Chenai→Chennai, Kolkata→Calcutta
10. FLIGHT NUMBER HANDLING: 
   - Extract the EXACT flight number as it appears on the ticket - DO NOT modify, assume, or correct anything
   - Copy the flight number character by character exactly as printed on the ticket
   - DO NOT make assumptions about airline codes or convert between different airline codes
   - DO NOT standardize or normalize airline codes - use exactly what is written
   - If the ticket shows "IX-2815", extract "IX-2815" - do NOT change it to "AI-2815" or any other code
   - If the ticket shows "6E 234", extract "6E 234" - preserve spaces and formatting
   - If partially readable, extract only the visible characters - do NOT guess missing parts
   - Examples: "IX-2815" → "IX-2815", "AI 101" → "AI 101", "6E234" → "6E234", "EK-345" → "EK-345"
   - CRITICAL: Never assume or convert airline codes - extract exactly what you see
11. TIME FORMAT SUPPORT: Accept both 12-hour (AM/PM) and 24-hour formats

CRITICAL FLIGHT NUMBER EXTRACTION RULES:
- NEVER assume, guess, or modify airline codes
- NEVER convert between airline codes (e.g., do NOT change IX to AI, or AI to IX)
- NEVER standardize airline codes based on your knowledge
- Extract flight number EXACTLY as printed on the ticket
- Preserve all formatting, spaces, and special characters exactly as shown

IMPORTANT RULES:
- For ONE-WAY tickets (including multi-leg): Extract only the single flight segment with the appropriate type based on which field it was uploaded to
- For MULTI-LEG ONE-WAY flights: Do NOT treat as round-trip. Examples:
  * NYC → Delhi → Hyderabad = ONE-WAY trip (doesn't return to NYC)
  * London → Dubai → Mumbai = ONE-WAY trip (doesn't return to London)
  * Delhi → Bangkok → Singapore = ONE-WAY trip (doesn't return to Delhi)
- For ROUND-TRIP tickets: First leg's origin MUST match last leg's destination. Examples:
  * NYC → Delhi → Hyderabad → Delhi → NYC = ROUND-TRIP (starts and ends in NYC)
  * Mumbai → Dubai → London → Dubai → Mumbai = ROUND-TRIP (starts and ends in Mumbai)
- ALWAYS extract BOTH segments (onward and return) for round-trip regardless of upload field
- A ticket is ONLY round-trip if there's a complete journey back to the starting city
- Only include a "return" type segment if the journey actually returns to the origin
- Do not create a return flight if the journey doesn't come back to the starting point
- Multiple flight legs in the same direction without returning = ONE-WAY (not round-trip)
- Always extract all passenger names from the ticket for validation purposes
- Always convert airport codes to city names
- Use proper timezone formatting as specified in TIMEZONE HANDLING section above
- Use standardized city names
- CRITICAL: Only extract times that are EXPLICITLY stated in the ticket. Do NOT infer, calculate, or assume missing times
- If arrival time is not clearly stated, use null as the value for arrivalDateTime
- If departure time is not clearly stated, use null as the value for departureDateTime
- Boarding passes often only show departure/boarding times, not arrival times

EXTRACTION LOGIC:
- If the ticket shows only ONE direct flight → Extract as single segment with type "${oneWaySegmentType}"
- If the ticket shows MULTIPLE flights in the SAME direction (multi-leg one-way) → Extract as single segment with type "${oneWaySegmentType}" using first departure and final arrival
- If the ticket shows flights that START and END at the same city (true round-trip) → Extract as TWO segments with types "onward" and "return"
- The upload field (${journeyType || 'not specified'}) only affects one-way ticket classification, not round-trip extraction

CRITICAL DISTINCTION:
- Multi-leg one-way: NYC → Delhi → Hyderabad (starts NYC, ends Hyderabad) = ONE segment
- True round-trip: NYC → Delhi → Hyderabad → Delhi → NYC (starts NYC, ends NYC) = TWO segments
- For round-trip: Split the journey at the furthest point from origin
  * Onward segment: NYC → Delhi → Hyderabad (first departure to furthest destination)
  * Return segment: Hyderabad → Delhi → NYC (furthest destination back to origin)

Return a JSON array like this:

For ONE-WAY ticket (including multi-leg):
[
  {
    "type": "${oneWaySegmentType}",
    "${oneWayAirlineField}": "string",
    "${oneWayFlightNumberField}": "string", 
    "comingFrom": "Full City Name (not airport code)",
    "goingTo": "Full City Name (not airport code)",
    "departureDateTime": "ISO 8601 format with origin city timezone OR null if not stated",
    "arrivalDateTime": "ISO 8601 format with destination city timezone OR null if not stated",
    "passengerNames": ["Full Name 1", "Full Name 2", "etc"]
  }
]

For ROUND-TRIP ticket:
[
  {
    "type": "onward",
    "airlineNameOnward": "string",
    "flightNumberOnward": "string",
    "comingFrom": "Full City Name (not airport code)",
    "goingTo": "Full City Name (not airport code)",
    "departureDateTime": "ISO 8601 format with origin city timezone OR null if not stated",
    "arrivalDateTime": "ISO 8601 format with destination city timezone OR null if not stated",
    "passengerNames": ["Full Name 1", "Full Name 2", "etc"]
  },
  {
    "type": "return",
    "airlineNameReturn": "string",
    "flightNumberReturn": "string",
    "comingFrom": "Full City Name (not airport code)",
    "goingTo": "Full City Name (not airport code)",
    "departureDateTime": "ISO 8601 format with origin city timezone OR null if not stated", 
    "arrivalDateTime": "ISO 8601 format with destination city timezone OR null if not stated",
    "passengerNames": ["Full Name 1", "Full Name 2", "etc"]
  }
]

AIRPORT CODE TO CITY MAPPING (use these exact city names):
- HYD/VOHS → Hyderabad
- DEL/VIDP → New Delhi  
- BOM/VABB → Mumbai
- BLR/VOBL → Bangalore
- MAA/VOMM → Chennai
- CCU/VECC → Kolkata
- JFK/KJFK → New York
- LHR/EGLL → London
- DXB/OMDB → Dubai
- SIN/WSSS → Singapore
- CDG/LFPG → Paris
- LAX/KLAX → Los Angeles
- ORD/KORD → Chicago
- ATL/KATL → Atlanta
- LAS/KLAS → Las Vegas
- MIA/KMIA → Miami
- SFO/KSFO → San Francisco
- BOS/KBOS → Boston
- DEN/KDEN → Denver
- SEA/KSEA → Seattle

IMPORTANT: If you encounter airport codes NOT listed above:
- Use your knowledge to convert them to the correct city names
- For Indian airports: Use the main city name (e.g., AMD → Ahmedabad, GOI → Goa, COK → Kochi, TRV → Trivandrum)
- For International airports: Use the main city name (e.g., NRT → Tokyo, ICN → Seoul, PEK → Beijing, SYD → Sydney)
- If completely unknown, keep the airport code but add a note in parentheses like "XYZ (Airport Code)"
- Always prioritize full city names over airport codes

Note: Use the above keys exactly as shown for each segment type.
   ` + `\n\nFlight Ticket Content:\n${content}`
  );
};

export const extractInvoiceDataSystemPrompt =
  'You are an expert in identity documents from all countries and government authorities worldwide. You have multilingual capabilities and can analyze documents in any language including English, Hindi, Arabic, Chinese, Spanish, French, German, Japanese, Korean, and other international languages.';

export const extractInvoiceDataUserPrompt = (content: string): string =>
  `
    You will be given text extracted from an identity document that could be from any country and in any language.
    Determine which type of document it is. Only choose from: Passport, Voter id, Aadhar, Pan card, Driving license.
    
    MULTILINGUAL ANALYSIS:
    - Documents may be in local languages (Hindi, Arabic, Chinese, Spanish, French, German, etc.)
    - Look for document structure and official elements rather than specific English text
    - Recognize government authority names in any language
    - Focus on document layout, official seals, and typical document elements
    
    DOCUMENT TYPES TO IDENTIFY:
    
    **Passport** - Issued by any country's government
    - Look for: Country name, "Passport" (or equivalents like "पासपोर्ट", "جواز سفر", "护照"), government authority, passport number, travel document indicators
    - Examples: "Republic of India", "United States of America", "Ministry of External Affairs", "Department of State"
    
    **Voter id** - Electoral identity cards from any country  
    - Look for: Electoral commission names, voter registration details, constituency information
    - Examples: "Election Commission", "Electoral Commission", "Board of Elections", voter registration numbers
    
    **Aadhar** - Unique to India only
    - Look for: "Aadhaar" (or "आधार"), "UIDAI" (or "यूआईडीएआई"), "Unique Identification Authority of India" (or "भारत सरकार"), 12-digit format
    
    **Pan card** - Unique to India only  
    - Look for: "Permanent Account Number", "Income Tax Department", "GOVT. OF INDIA", ABCDE1234F format
    
    **Driving license** - Motor vehicle licenses from any country
    - Look for: "Driving Licence", "Driver's License", "Department of Motor Vehicles", "Transport Department"
    - License classes, vehicle endorsements, validity dates
    
    IMPORTANT: Reject non-identity documents like:
    - Invoices or bills mentioning ID numbers
    - Bank statements  
    - Application forms
    - Receipts or transaction documents
    
    Reply with JSON in the following format:
    {
      "documentType": "Passport" // or "Voter id", "Aadhar", "Pan card", "Driving license", or "Unknown"
    }
  ` + `\n\nDocument content:\n${content}`;

export const extractIdentityDocumentSystemPrompt =
  'You are an expert in extracting information from identity documents from any country, including multilingual documents. You understand various languages including Hindi, English, and other regional languages commonly used in identity documents.';

export const extractIdentityDocumentUserPrompt = (content: string, documentType: string): string =>
  `
    You will be given text content from what should be a legitimate ${documentType} document from any country.
    
    IMPORTANT: First verify this is actually a legitimate ${documentType} document, not just any document that happens to contain a ${documentType} number.
    
    DOCUMENT VALIDATION CRITERIA FOR ${documentType}:
    
    ${
      documentType === 'Passport'
        ? `
    - Must contain official passport headers like country name, "Passport" (or equivalents in local language like "पासपोर्ट", "جواز سفر", "护照"), issuing authority
    - Should have passport-specific fields like Date of Birth, Place of Birth, Date of Issue, Date of Expiry in any language
    - Must have passport format layout with official government formatting
    - Examples: "Republic of India", "United States of America", "Ministry of External Affairs", "Department of State"
    - May contain text in local language of the issuing country
    `
        : ''
    }
    
    ${
      documentType === 'Pan card'
        ? `
    - Must contain "Permanent Account Number", "Income Tax Department", or "GOVT. OF INDIA" 
    - Should have Pan card specific fields like Father's Name, Date of Birth, Signature
    - Must be a Pan card layout, not just a PAN number mentioned in an invoice, bill, or other document
    - Invoices, bills, receipts mentioning PAN numbers are NOT Pan cards
    `
        : ''
    }
    
    ${
      documentType === 'Aadhar'
        ? `
    - Should contain Aadhaar-related text like "Aadhaar", "UIDAI", "आधार" or partial/cropped sections of official Aadhaar cards
    - May have Aadhaar-specific elements like address, date of birth, gender, or holder name with 12-digit number
    - Accept both complete cards and partial/cropped sections that clearly show Aadhaar document elements
    - Look for typical Aadhaar elements: 12-digit number pattern, demographic details, or distinctive Aadhaar formatting
    - IMPORTANT: If you can extract a valid holder name and the document shows characteristics of an Aadhaar card (even partial), consider it valid
    - Do NOT reject small pieces or cropped sections of legitimate Aadhaar cards
    - Aadhaar number extraction:
      - If the Aadhaar number is visible and not masked, extract the full 12-digit number as it appears.
      - If the number is masked, extract the masked pattern (e.g., "XXXX XXXX 1234").
      - Do NOT mask or alter the number if it is fully visible.
    `
        : ''
    }
    
    ${
      documentType === 'Voter id'
        ? `
    - Must contain official election commission or electoral authority names from any country in any language
    - Should have voter-specific fields like constituency, registration details, electoral information
    - Must be a voter id card layout from official electoral authorities
    - Examples: "Election Commission of India" (चुनाव आयोग), "Electoral Commission", "Board of Elections", etc.
    - May contain text in local language of the issuing country
    `
        : ''
    }
    
    ${
      documentType === 'Driving license'
        ? `
    - Must contain "Driving Licence", "Driver's License", "Transport Department", or equivalent transport authority in any language
    - Should have license-specific fields like validity dates, address, vehicle endorsements, license classes
    - Must be a driving license layout from official transport/motor vehicle departments
    - Can be from any country's transport authority in local language
    - Examples: "Transport Department" (परिवहन विभाग), "Motor Vehicle Department", "Department of Motor Vehicles", etc.
    `
        : ''
    }
    
    INVALID DOCUMENTS TO REJECT:
    - Invoices or bills that mention ${documentType} numbers
    - Bank statements showing ${documentType} numbers
    - Application forms with ${documentType} numbers
    - Receipts or transaction documents
    - Any non-government/non-official issued document
    - Screenshots of applications or websites
    
    EXTRACTION LOGIC:
    1. First assess if this document shows characteristics of a legitimate ${documentType} document
    2. For partial/cropped documents: if you can extract a holder name and see ${documentType} characteristics, consider it valid
    3. For ${documentType === 'Aadhar' ? 'Aadhar cards: Accept partial cards if they contain holder name and Aadhar-like formatting/structure. For masked numbers, extract the complete masked pattern (e.g., "XXXX XXXX 1234" not just "1234")' : `${documentType}: Only accept if clear official elements are present`}
    4. Only extract the number/identifier if document validation passes
    5. Be generous with validation for partial documents that clearly belong to the document type
    
    Extract the following fields and reply with JSON:
    {
      "documentType": "${documentType}",
      "travelInfoNumber": "${documentType === 'Aadhar' ? 'full 12-digit Aadhar number if visible, or partial number with last 4 digits if masked (e.g., "XXXX XXXX 1234"), or null if no number found' : `actual ${documentType.toLowerCase()} number/identifier from valid document, or null if document is invalid/not a real ${documentType}`}",
      "holderName": "full name of the document holder as it appears on the document, or null if not extractable",
      "isValidDocument": true/false,
      "validationReason": "explanation of why this is or isn't a valid ${documentType} document",
      "issuingCountry": "country that issued this document if identifiable, or null"
    }
    
    IMPORTANT NOTES FOR MULTILINGUAL DOCUMENTS:
    - If the Aadhaar card has the holder's name in English, extract the English name.
    - If the English name is not present, extract the name in the language(s) as shown on the card.
    - Do NOT extract or translate the name into a language that does not appear on the card.
    - Always prefer the English name if available; otherwise, use the name as it appears in the original script.
    - For validation purposes, focus on document structure and official elements rather than language.
    - A document in Hindi script is equally valid as one in English script, but do not invent or translate names.
    - Look for government logos, official formatting, and document structure rather than specific English text.
    
    Remember: Only extract numbers from actual government/officially-issued ${documentType} documents, not from invoices, bills, or other documents that simply mention ${documentType} numbers.
  ` + `\n\nDocument content:\n${content}`;

export const extractCitySystemPrompt =
  'You are an expert in extracting city names from addresses worldwide with multilingual capabilities. You can identify and extract city names from addresses written in any language including English, Hindi, Arabic, Chinese, Spanish, French, German, Japanese, Korean, and other international languages.';

export const extractCityUserPrompt = (venueAddress: string): string =>
  `Extract the main city name from this venue address that may be in any language: "${venueAddress}"

      MULTILINGUAL PROCESSING:
      - Address may be in local language (Hindi, Arabic, Chinese, Spanish, French, etc.)
      - Convert local city names to standardized English names
      - Recognize city names in various scripts (Latin, Devanagari, Arabic, Chinese, etc.)
      - Examples: "मुंबई" → "Mumbai", "دبي" → "Dubai", "北京" → "Beijing"

      Rules:
      1. Return only the city name in standard English (e.g., "Hyderabad", "Mumbai", "New Delhi", "Bangalore")
      2. Ignore resort names, mandals, districts, pin codes, and other location details
      3. If the address contains multiple city references, return the primary/main city
      4. Return the internationally recognized city name (e.g., use "Mumbai" instead of "Bombay")
      5. If no clear city can be identified, return "Unknown"

      Examples:
      - "Leonia Holistic Destination, Bommarasipet, Shamirpet Mandal, Medchal-Malkajgiri District, Hyderabad - 500078" → "Hyderabad"
      - "The Oberoi, Marine Drive, Mumbai - 400021" → "Mumbai"
      - "ITC Grand Chola, Guindy, Chennai - 600032" → "Chennai"
      - "فندق الإمارات، دبي، الإمارات العربية المتحدة" → "Dubai"
      - "ताज महल होटल, मुंबई, महाराष्ट्र" → "Mumbai"

      Return only the city name.`;

// Lookup matching prompts for flight data standardization
export const lookupMatchingSystemPrompt = `You are an expert in matching and standardizing airline names and city names against a predefined lookup database.

Your task is to find the best match from the provided lookup data for extracted airline names and city names from flight tickets.

Guidelines:
1. For airlines: Match against airline names (including variations, abbreviations) and IATA/ICAO codes
2. For cities: Match against city names (including variations, alternate spellings, and common abbreviations)
3. Be flexible with matching - handle typos, abbreviations, and alternate names
4. If no reasonable match is found, return "Other" with the original value preserved
5. Prioritize exact matches, then partial matches, then fuzzy matches
6. Be aware of common variations (e.g., Mumbai/Bombay, Bangalore/Bengaluru, Delhi/New Delhi)

Return results in the specified JSON format.`;

export const lookupAirlineMatchingUserPrompt = (
  extractedAirline: string,
  lookupAirlines: any[],
  segmentType: 'onward' | 'return',
): string => {
  const airlineField = segmentType === 'onward' ? 'airlineNameOnward' : 'airlineNameReturn';
  const otherField = segmentType === 'onward' ? 'otherAirlineNameOnward' : 'otherAirlineNameReturn';

  return `
Extracted airline: "${extractedAirline}"

Available airlines in lookup database:
${JSON.stringify(lookupAirlines, null, 2)}

Find the best match for the extracted airline "${extractedAirline}" from the lookup database.

Consider:
- Exact matches with airline names
- Matches with airline codes (IATA/ICAO)
- Common abbreviations and variations
- Fuzzy matches for typos

If a match is found, return:
{
  "${airlineField}": "matched_airline_name"
}

If no match is found, return:
{
  "${airlineField}": "Other",
  "${otherField}": "${extractedAirline}"
}

Return only the JSON response.`;
};

export const lookupCityMatchingUserPrompt = (
  extractedCity: string,
  lookupCities: any[],
  fieldType: 'departure' | 'arrival',
  segmentType?: 'onward' | 'return',
): string => {
  const cityField = fieldType === 'departure' ? 'comingFrom' : 'goingTo';
  
  // Determine the correct "other" field based on segment type and field type
  let otherField: string;
  if (segmentType === 'onward' && fieldType === 'departure') {
    // For onward segment, comingFrom should set otherArrival
    otherField = 'otherArrival';
  } else if (segmentType === 'return' && fieldType === 'arrival') {
    // For return segment, goingTo should set otherDeparture
    otherField = 'otherDeparture';
  } else {
    // Default behavior for backward compatibility
    otherField = fieldType === 'departure' ? 'otherDeparture' : 'otherArrival';
  }

  return `
Extracted city: "${extractedCity}"

Available cities in lookup database:
${JSON.stringify(lookupCities, null, 2)}

Find the best match for the extracted city "${extractedCity}" from the lookup database.

Consider:
- Exact matches with city names
- Common city variations (Mumbai/Bombay, Bangalore/Bengaluru, Delhi/New Delhi, Chennai/Madras, Kolkata/Calcutta)
- Partial matches (e.g., "New Delhi" matches "Delhi")
- Fuzzy matches for typos and alternate spellings
- Airport city associations (e.g., BOM → Mumbai, DEL → Delhi)

If a match is found, return:
{
  "${cityField}": "matched_city_name"
}

If no match is found, return:
{
  "${cityField}": "Other",
  "${otherField}": "${extractedCity}"
}

Return only the JSON response.`;
};
