import axios from 'axios';
import { AIRLINE_CODE_TO_NAME, KPI_CATEGORIES, KPI_FILTERS, ONES, SEEKER_FE_REG_PATH, SIGNED_URL_EXCLUDED_DOMAINS, TENS } from '../constants/constants';
import { RegistrationStatusEnum } from '../enum/registration-status.enum';
import { PaymentModeEnum } from '../enum/payment-mode.enum';
const puppeteer = require('puppeteer-core');
import chromium from "@sparticuz/chromium-min";
import { PersonTypeEnum } from '../enum/person-type.enum';
import { ParentalFormStatusEnum } from '../enum/parental-form-status.enum';
import { ApprovalStatusEnum } from '../enum/approval-status.enum';
import { TravelStatusEnum } from '../enum/travel-status.enum';
import { 
  GOODIES_DISPLAY_VALUES, 
  SIZE_ORDER, 
  SIZE_VALUES, 
  RATRIA_PILLAR_ORDER, 
  RATRIA_PILLAR_VALUES, 
  EMPTY_VALUE_IDENTIFIERS,
  STATUS_MESSAGES
} from '../constants/constants';
import { SwapRequestStatus } from '../enum/swap-request-status-enum';
import { SwapType } from '../enum/swap-type-enum';

const qs = require('qs');

/**
 * Split array into chunks of specified size
 */
export function chunkArray<T>(array: T[], chunkSize: number): T[][] {
  const chunks: T[][] = [];
  for (let i = 0; i < array.length; i += chunkSize) {
    chunks.push(array.slice(i, i + chunkSize));
  }
  return chunks;
}

/**
 * Add delay between operations
 */
export function delay(ms: number): Promise<void> {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

/**
 * Get the week name based on the date
 */
export function getWeekName(date: Date): string {
  const weekNames = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
  return weekNames[date.getDay()];
}

/**
 * Convert date format from ISO string to DD-Month-YYYY
 */
export function formatDate(dateString: string): string {
  const date = new Date(dateString);
  const day = String(date.getDate()).padStart(2, '0');
  const monthNames = [
    'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
    'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'
  ];
  const month = monthNames[date.getMonth()];
  const year = date.getFullYear();
  return `${day}-${month}-${year}`;
}

/**
 * Determines whether two GST (Goods and Services Tax) numbers belong to the same state.
 *
 * GST numbers are assumed to have their state codes as the first two characters.
 *
 * @param gst1 - The first GST number as a string.
 * @param gst2 - The second GST number as a string.
 * @returns `true` if both GST numbers have the same state code, otherwise `false`.
 */
export function areGSTsFromSameState(gst1: string, gst2: string): boolean {
  if (!gst1 || !gst2) return false;

  const stateCode1 = gst1.slice(0, 2);
  const stateCode2 = gst2.slice(0, 2);

  return stateCode1 === stateCode2;
}

export function convertRupeesToWords(amount: number): string {
  const units = [
    '', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine',
    'Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen', 'Seventeen', 'Eighteen', 'Nineteen'
  ];
  const tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];
  const scales = ['', 'Thousand', 'Lakh', 'Crore'];

  if (amount === 0) return 'Zero Rupees';

  const getWordsForThreeDigits = (num: number): string => {
    let result = '';
    if (num > 99) {
      result += `${units[Math.floor(num / 100)]} Hundred `;
      num %= 100;
    }
    if (num > 19) {
      result += `${tens[Math.floor(num / 10)]} `;
      num %= 10;
    }
    if (num > 0) {
      result += `${units[num]} `;
    }
    return result.trim();
  };

  let rupees = Math.floor(amount); // Extract Rupees
  const paise = Math.round((amount - rupees) * 100); // Extract Paise

  let words = '';
  let scaleIndex = 0;

  while (rupees > 0) {
    const chunk = rupees % 1000;
    if (chunk > 0) {
      const chunkWords = getWordsForThreeDigits(chunk);
      words = `${chunkWords} ${scales[scaleIndex]} ${words}`.trim();
    }
    rupees = Math.floor(rupees / 1000);
    scaleIndex++;
  }

  const rupeesInWords = `${words.trim()} Rupees`;
  const paiseInWords = paise > 0 ? `${getWordsForThreeDigits(paise)} Paise` : '';

  return `${rupeesInWords}${paiseInWords ? ` and ${paiseInWords}` : ''}`;
}

/**
 * Convert a number to words in Indian numbering system.
 * @param num - The number to convert.
 * @returns The number in words.
 */
function convertNumberToWords(num: number): string {
  if (num === 0) return 'Zero';

  const units = [
    { value: 1000000000000, str: 'Lakh Crore' },
    { value: 10000000, str: 'Crore' },
    { value: 100000, str: 'Lakh' },
    { value: 1000, str: 'Thousand' },
    { value: 100, str: 'Hundred' }
  ];

  let result = '';

  for (const unit of units) {
    const quotient = Math.floor(num / unit.value);
    if (quotient > 0) {
      result += `${convertNumberToWords(quotient)} ${unit.str} `;
      num %= unit.value;
    }
  }

  if (num > 0) {
    if (num < 20) {
      result += ONES[num];
    } else {
      result += `${TENS[Math.floor(num / 10)]} ${ONES[num % 10]}`.trim();
    }
  }

  return result.trim();
}

/**
 * Convert an amount in rupees and paise to words in Indian numbering system.
 *
 * @param amount - The amount in rupees (can be a number or string).
 * @returns The amount in words.
 */
export function amountInWordsIndian(amount: number): string {
  if (isNaN(amount) || amount < 0) {
    throw Error('Invalid amount');
  }
  const [rupees, paise] = Number(amount).toFixed(2).split('.').map(Number);

  const rupeeWords = rupees > 0
    ? `Rupees ${convertNumberToWords(rupees)}`
    : '';

  const paiseWords = paise > 0
    ? `${convertNumberToWords(paise)} Paise`
    : '';

  let result = '';
  if (rupeeWords && paiseWords) {
    result = `${rupeeWords} and ${paiseWords}`;
  } else if (rupeeWords) {
    result = rupeeWords;
  } else if (paiseWords) {
    result = paiseWords;
  } else {
    result = 'Rupees Zero';
  }

  return result.charAt(0).toUpperCase() + result.slice(1);
}


/**
 * Generate the next sequence number based on a prefix and the previous sequence number.
 *
 * @param prefix - The prefix for the sequence.
 * @param previousSequence - The previous sequence number as a string.
 * @returns The next sequence number as a string.
 */
export function generateNextSequence(prefix: string, previousSequence: string): string {
  const prefixLength = prefix.length;

  if (!previousSequence.startsWith(prefix)) {
    throw new Error('Previous sequence does not match the provided prefix.');
  }

  const sequenceNumberPart = previousSequence.slice(prefixLength);
  const nextSequenceNumber = (parseInt(sequenceNumberPart, 10) + 1).toString().padStart(sequenceNumberPart.length, '0');

  return `${prefix}${nextSequenceNumber}`;
}

export const oAuth = async () => {
  try {
    const data = qs.stringify({
      refresh_token: process.env.ZOHO_REFERESH_TOKEN,
      client_id: process.env.ZOHO_CLIENT_ID,
      client_secret: process.env.ZOHO_CLIENT_SECRET,
      redirect_uri: process.env.ZOHO_REDIRECT_URL,
      grant_type: process.env.ZOHO_GRANT_TYPE,
    });

    const config = {
      method: 'post',
      url: process.env.ZOHO_TOKEN_URL,
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
      },
      data: data,
    };

    try {
      const response = await axios(config);
      return response.data;
    } catch (error) {
      console.error(error);
    }
  } catch (error) {
    console.log('------', error);
    return error;
  }
};

export function toSnakeCase(str: string): string {
  return str
    .trim()                         // Remove leading/trailing whitespace
    .replace(/\s+/g, '_')          // Replace one or more spaces with underscore
    .replace(/[^a-zA-Z0-9_]/g, '_') // Replace non-alphanumeric chars with underscore
    .replace(/_+/g, '_')           // Replace multiple consecutive underscores with single
    .replace(/^_+|_+$/g, '')       // Remove leading and trailing underscores
    .toLowerCase();                // Convert to lowercase
}

export function generateFormSectionKey(name: string): string {
  const prefix = name.replace(/\s+/g, '').substring(0, 6).toUpperCase();
  const random = Math.floor(Math.random() * 10000)
    .toString()
    .padStart(4, '0');
  return `FS_${prefix}_${random}`;
}
/**
 * Deduct a specified number of days from a given date and return the resulting date.
 *
 * @param date - The initial date.
 * @param days - The number of days to deduct.
 * @returns The date after deducting the specified number of days.
 */
export function deductDaysFromDate(date: Date, days: number): Date {
  const resultDate = new Date(date);
  resultDate.setDate(resultDate.getDate() - days);
  return resultDate;
}

/**
 * Format a date or datetime string into a specific format.
 *
 * @param dateInput - The date or datetime string to format.
 * @returns The formatted date string.
 * example: any date to "14-08-2025 14:48:00"
 */
export function formatDateTime(dateInput: Date | string): string {
  const date = new Date(dateInput);
  const pad = (n: number) => n.toString().padStart(2, '0');
  const day = pad(date.getDate());
  const month = pad(date.getMonth() + 1);
  const year = date.getFullYear();
  const hours = pad(date.getHours());
  const minutes = pad(date.getMinutes());
  const seconds = pad(date.getSeconds());
  return `${day}-${month}-${year} ${hours}:${minutes}:${seconds}`;
}

/**
 * Returns the number of days between two dates.
 *
 * @param startDate - The start date.
 * @param endDate - The end date.
 * @returns The count of days between the two dates.
 */
export function getDaysCountBetweenDates(startDate: Date | string, endDate: Date | string): number {
  const start = new Date(startDate);
  const end = new Date(endDate);
  const diffTime = end.getTime() - start.getTime();
  return Math.floor(diffTime / (1000 * 60 * 60 * 24));
}

/**
 * Format time in IST format (HH:MM)
 * @param date - The date to format
 * @returns Time string in IST format
 */
export function formatTimeIST(date: any): string {
  try {
    const dateObj = new Date(date);
    if (isNaN(dateObj.getTime())) return '';
    
    // Convert to IST timezone
    const istTime = dateObj.toLocaleTimeString('en-IN', { 
      timeZone: 'Asia/Kolkata',
      hour: '2-digit', 
      minute: '2-digit',
      hour12: false
    });
    
    return istTime;
  } catch (error) {
    return '';
  }
}

/**
 * Format date and time in IST format (DD-MMM-YYYY HH:MM:SS)
 * @param date - The date to format
 * @returns Date and time string in DD-MMM-YYYY HH:MM:SS format with IST timezone
 */
export function formatDateTimeIST(date: any): string {
  try {
    const dateObj = new Date(date);
    if (isNaN(dateObj.getTime())) return '';

    const formatter = new Intl.DateTimeFormat('en-IN', {
      timeZone: 'Asia/Kolkata',
      day: '2-digit',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
      hour12: false
    });

    const parts = formatter.formatToParts(dateObj);
    const partsMap = parts.reduce((acc, part) => {
      acc[part.type] = part.value;
      return acc;
    }, {} as Record<string, string>);

    const month = partsMap.month.replace('Sept', 'Sep');
    
    // Ensure proper padding for time components
    const hour = partsMap.hour.padStart(2, '0');
    const minute = partsMap.minute.padStart(2, '0');
    const second = partsMap.second.padStart(2, '0');

    return `${partsMap.day}-${month}-${partsMap.year} ${hour}:${minute}:${second}`;
  } catch (error) {
    return '';
  }
}


  /**
   * Format arrival/departure time to show only Date + HH:MM
   * @param date - Date to format
   * @returns Formatted date string as DD/MM/YYYY HH:MM
   */
  export function formatDateTimeISTWithoutSec(date: Date | string): string {
    try {
      const dateObj = new Date(date);
      if (isNaN(dateObj.getTime())) return '';

      const formatter = new Intl.DateTimeFormat('en-IN', {
        timeZone: 'Asia/Kolkata',
        day: '2-digit',
        month: 'short',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
        hour12: false
      });

      const parts = formatter.formatToParts(dateObj);
      const partsMap = parts.reduce((acc, part) => {
        acc[part.type] = part.value;
        return acc;
      }, {} as Record<string, string>);

      const month = partsMap.month.replace('Sept', 'Sep');
      
      // Ensure proper padding for time components
      const hour = partsMap.hour.padStart(2, '0');
      const minute = partsMap.minute.padStart(2, '0');

      return `${partsMap.day}-${month}-${partsMap.year} ${hour}:${minute}`;
    } catch (error) {
      return '';
    }
  }

/**
 * Calculate age from date of birth
 * @param dateOfBirth - The date of birth
 * @returns Age in years
 */
export function calculateAge(dateOfBirth: Date | string | null | undefined): number {
  if (!dateOfBirth) return 0;
  
  let birthDate: Date;
  
  // Handle different input types
  if (dateOfBirth instanceof Date) {
    birthDate = dateOfBirth;
  } else if (typeof dateOfBirth === 'string') {
    birthDate = new Date(dateOfBirth);
  } else {
    return 0;
  }
  
  // Check if the date is valid
  if (isNaN(birthDate.getTime())) {
    return 0;
  }
  
  const today = new Date();
  let age = today.getFullYear() - birthDate.getFullYear();
  const monthDiff = today.getMonth() - birthDate.getMonth();
  if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
    age--;
  }
  return age;
}

/**
 * Format boolean values for display
 * @param value - The boolean value to format
 * @returns 'Yes' for true, 'No' for false, empty string for null/undefined
 */
export function formatBooleanValue(value: any): string {
  if (value === true) return 'Yes';
  if (value === false) return 'No';
  return '';
}

export function generatePaymentLink(
  programId: number,
  userId: number | undefined = undefined,
  paymentMode?: PaymentModeEnum,
  editPayment?: boolean,
  withBaseUrl: boolean = true,
): string {
  const baseUrl = process.env.SEEKER_FE_BASE_URL || '';
  let url = withBaseUrl
    ? `${baseUrl}${SEEKER_FE_REG_PATH}${programId}`
    : `${SEEKER_FE_REG_PATH}${programId}`;
  if (paymentMode) {
    url += `&paymentMode=${encodeURIComponent(paymentMode)}`;
  }
  if (userId) {
    url += `&userId=${userId}`;
  }
  if (editPayment) {
    url += `&editpayment=true`;
  }
  return url;
}

/**
 * Convert a date to IST (Indian Standard Time) and return as Date object.
 * @param dateInput - The date to convert (Date or string)
 * @returns Date object in IST timezone
 */

export function formatDateIST(date: any): string {
  try {
    const dateObj = new Date(date);
    if (isNaN(dateObj.getTime())) return '';

    // Convert to Asia/Kolkata timezone and get date parts
    const options: Intl.DateTimeFormatOptions = {
      timeZone: 'Asia/Kolkata',
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    };
    // Example output: "14 Aug 2025"
    let formatted = dateObj.toLocaleDateString('en-GB', options);
    // Ensure September is always 'Sep' (not 'Sept')
    formatted = formatted.replace(/\bSept\b/, 'Sep');

    // Convert "14 Aug 2025" to "14-Aug-2025"
    return formatted.replace(/ /g, '-');
  } catch (error) {
    return '';
  }
}

export async function generatePDF(
  invoiceTemplateData: any,
  templateType: any, fileName: string,
): Promise<{ buffer: Buffer }> {
  // Implement PDF generation based on your requirements
  // This is a placeholder implementation


  try {
    // You can use libraries like:
    // - puppeteer for HTML to PDF
    // const puppeteer = require('puppeteer');

    // Launch Puppeteer
    // const browser = await puppeteer.launch()
    const browser = await puppeteer.launch({
      args: [
        ...(Array.isArray(chromium.args) ? chromium.args : []),
        '--no-sandbox',
        '--disable-setuid-sandbox',
        '--disable-features=AudioServiceOutOfProcess',
        '--disable-gpu',
        '--disable-software-rasterize',
        '--disable-features=AudioServiceOutOfProcessKillAtHang',
        '--single-process',
        '--disable-software-rasterizer',
      ],
      defaultViewport: null,
      executablePath: await chromium.executablePath(
        'https://github.com/Sparticuz/chromium/releases/download/v119.0.2/chromium-v119.0.2-pack.tar',
      ),
      headless: true,
    });
    const page = await browser.newPage();
    // Set the HTML content
    await page.setContent(templateType(invoiceTemplateData));

    // For now, return a placeholder buffer
    const pdfBuffer = await page.pdf({
      format: 'A4',
      printBackground: true,
    });
    await browser.close();

    return { buffer: pdfBuffer };
  } catch (error) {
    console.error('Error generating PDF:', error);
    throw error;
  }
}

/**
 * Calculate age in years from a date of birth.
 * @param dob - The date of birth (Date or string)
 * @returns Age in years
 */
export function getAgeFromDOB(dob: Date | string): number {
  const today = new Date();
  const birthDate = new Date(dob);
  let age = today.getFullYear() - birthDate.getFullYear();
  const monthDiff = today.getMonth() - birthDate.getMonth();

  if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
    age--;
  }
  return age;
}

/**
 * Fetch a PDF from a given URL and return its base64-encoded content.
 * @param url - The URL of the PDF file.
 * @returns The base64-encoded content of the PDF.
 */
export async function fetchPdfAsBase64(url: string): Promise<string> {
  try {
    const response = await axios.get(url, {
      responseType: 'arraybuffer',
      headers: {
        referer: process.env.BACKEND_URL,
      },
    });
    return Buffer.from(response.data, 'binary').toString('base64');
  } catch (error) {
    // this.logger.error('Error fetching PDF:', error);
    throw error;
  }
}

/**
 * Determines the parental form status based on person type and current registration.
 * @param personType - The type of person (e.g., CHILD, ADULT, etc.)
 * @param currentRegistration - The current registration object (may contain parentalFormStatus)
 * @returns The appropriate ParentalFormStatusEnum value
 */
export function getParentalFormStatus(
  personType: PersonTypeEnum,
  currentRegistration?: { parentalFormStatus?: ParentalFormStatusEnum },
): ParentalFormStatusEnum {
  if (personType === PersonTypeEnum.CHILD) {
    return currentRegistration?.parentalFormStatus === ParentalFormStatusEnum.COMPLETED
      ? ParentalFormStatusEnum.COMPLETED
      : ParentalFormStatusEnum.PENDING;
  } else {
    return ParentalFormStatusEnum.NOT_APPLICABLE;
  }
}

/**
 * Extracts the academic year string (e.g., "2023-24") from a program's start date.
 * @param startsAt - The start date (Date or undefined/null)
 * @returns The academic year string, or an empty string if startsAt is not provided
 */
export function getAcademicYearFromStartDate(startsAt?: Date | null): string {
  if (!startsAt || isNaN(new Date(startsAt).getTime())) return '';
  const year = startsAt.getFullYear();
  const nextYear = (year + 1).toString().slice(-2);
  return `${year}-${nextYear}`;
}

// Utility function to get the first non-viewer role
export function getPrimaryRole(userRoles: string[]): string {
  if (!userRoles || userRoles.length === 0) return '';
  if (userRoles.length === 1) return userRoles[0];
  return userRoles.find(role => role !== 'viewer') || userRoles[0];
}

/**
 * Helper method to sort requestedPrograms by junction table insertion order
 * @param registration - Registration object with swapsRequests
 */
export function sortSwapRequestedPrograms(registration: any): void {
  registration?.swapsRequests?.forEach(swapRequest => {
    if (swapRequest.swapRequestedProgram && swapRequest.swapRequestedProgram.length > 0) {
      // Sort junction entries by id (insertion order)
      const sortedJunctions = swapRequest.swapRequestedProgram
        .sort((a, b) => a.id - b.id);
      
      // Map back to requestedPrograms in the correct order
      swapRequest.requestedPrograms = sortedJunctions
        .map(junction => junction.program)
        .filter(program => program); // Filter out any null/undefined programs
    }
  });
}

/**
 * Replaces underscores in a string with spaces.
 * @param value - string with underscores
 * @returns The input string with underscores replaced by spaces
 */
export function replaceUnderscoreWithSpace(value: string | undefined): string {
  if (!value) return '';
  return value.replace(/_/g, ' ');
}

/**
 * Converts airline code to full airline name
 * @param airlineCode - The airline code (e.g., 'AI', '6E', etc.)
 * @param otherAirlineName - The custom airline name if code is 'OTHER'
 * @returns Full airline name
 */
export function getAirlineFullName(airlineCode: string | undefined, otherAirlineName?: string): string {
  if (!airlineCode) return '';
  
  // Import at runtime to avoid circular dependencies
 
  
  if (airlineCode === 'OTHER' && otherAirlineName) {
    return otherAirlineName;
  }
  
  return AIRLINE_CODE_TO_NAME[airlineCode] || airlineCode;
}

/**
 * Converts text to title case
 * @param text - The input text
 * @returns Text in title case
 */
export function toTitleCase(text: string | undefined): string {
  if (!text || typeof text !== 'string') return '';
  
  return text
    .toLowerCase()
    .split(' ')
    .map(word => word.charAt(0).toUpperCase() + word.slice(1))
    .join(' ');
}

/**
 * Formats ratria pillar location from value for travel reports
 * @param ratriaPillarLeoniaValue - boolean value for ratria pillar in Leonia
 * @param location - location value 
 * @param otherLocation - other location if location is 'OTHER'
 * @returns Formatted ratria pillar location string
 */
export function formatLocation(location?: string, otherLocation?: string): string {
  // For travel reports, we need the actual location value, not just yes/no for Leonia
  if (!location) {
    return '';
  }

  if (location.toLowerCase() === 'other' && otherLocation) {
    return toTitleCase(otherLocation);
  }

  return toTitleCase(location);
}

/**
 * Formats airport pickup/drop locations based on travel mode
 * @param location - The original location string
 * @param travelType - The travel type (flight, own_transport, etc.)
 * @returns Formatted location string with "Airport" for flight travel
 */
export function formatTravelType(travelType: string | undefined): string {
  if (!travelType) return '';

  const formattedTravelType = replaceUnderscoreWithSpace(travelType);

  if (formattedTravelType?.toLowerCase() === 'flight') {
    return formattedTravelType .replace(/flight/i, 'Airport');
  }

  return toTitleCase(formattedTravelType);
}


      // Helper function to format Ratria Pillar location
export const formatRatriaPillar = (ratriaPillar: boolean, ratriaPillarLocation: string | undefined, ratriaPillarOtherLocation: string | undefined): string => {
  if (ratriaPillar) {
    const location = formatLocation(ratriaPillarLocation, ratriaPillarOtherLocation);
    return location || GOODIES_DISPLAY_VALUES.YES;
  }
  return GOODIES_DISPLAY_VALUES.NO;
};


// Helper function to format T-shirt/Jacket sizes
export const formatSizeSelection = (applicable: boolean, size: string): string => {
  if (!applicable) return GOODIES_DISPLAY_VALUES.NO;
  return size || GOODIES_DISPLAY_VALUES.YES;
};

/**
 * Check if a value is considered empty/null
 * @param value The value to check
 * @returns True if the value is empty/null
 */
const isEmptyValue = (value: string): boolean => {
  const normalized = (value || '').toLowerCase().trim();
  return !normalized || EMPTY_VALUE_IDENTIFIERS.includes(normalized as any);
};

/**
 * Sort sizes in the correct order: null/empty/- >> no >> xs >> s >> m >> l >> xl >> xxl >> xxxl >> yes
 * @param sizeA First size string to compare
 * @param sizeB Second size string to compare
 * @returns Comparison result (-1, 0, 1)
 */
export const sortSizes = (sizeA: string, sizeB: string): number => {
  // Define the correct order for sizes using constants
  const sizeOrder: { [key: string]: number } = {
    [SIZE_VALUES.NO]: SIZE_ORDER.NO,
    [SIZE_VALUES.XS]: SIZE_ORDER.XS,
    [SIZE_VALUES.S]: SIZE_ORDER.S,
    [SIZE_VALUES.M]: SIZE_ORDER.M,
    [SIZE_VALUES.L]: SIZE_ORDER.L,
    [SIZE_VALUES.XL]: SIZE_ORDER.XL,
    [SIZE_VALUES.XXL]: SIZE_ORDER.XXL,
    [SIZE_VALUES.XXXL]: SIZE_ORDER.XXXL,
    [SIZE_VALUES.YES]: SIZE_ORDER.YES,
  };

  const normalizedA = (sizeA || '').toLowerCase().trim();
  const normalizedB = (sizeB || '').toLowerCase().trim();

  // Handle null, undefined, empty string, or '-' as "no data" - should appear FIRST
  const isEmptyA = isEmptyValue(sizeA);
  const isEmptyB = isEmptyValue(sizeB);

  if (isEmptyA && isEmptyB) return 0; // Both empty, consider equal
  if (isEmptyA) return -1; // A is empty, A comes first
  if (isEmptyB) return 1; // B is empty, B comes first

  const orderA = sizeOrder[normalizedA] !== undefined ? sizeOrder[normalizedA] : SIZE_ORDER.UNKNOWN;
  const orderB = sizeOrder[normalizedB] !== undefined ? sizeOrder[normalizedB] : SIZE_ORDER.UNKNOWN;

  // If both have defined order, compare by order
  if (orderA !== orderB) {
    return orderA - orderB;
  }

  // If both are unknown, fall back to alphabetical comparison
  return normalizedA.localeCompare(normalizedB);
};

/**
 * Sort Ratria Pillar Leonia in the correct order: null/empty/- >> no >> Leonia >> other locations >> yes
 * @param valueA First Ratria Pillar value to compare
 * @param valueB Second Ratria Pillar value to compare
 * @returns Comparison result (-1, 0, 1)
 */
export const sortRatriaPillar = (valueA: string, valueB: string): number => {
  // Define the correct order for Ratria Pillar using constants
  const pillarOrder: { [key: string]: number } = {
    [RATRIA_PILLAR_VALUES.NO]: RATRIA_PILLAR_ORDER.NO,
    [RATRIA_PILLAR_VALUES.LEONIA]: RATRIA_PILLAR_ORDER.LEONIA,
    [RATRIA_PILLAR_VALUES.YES]: RATRIA_PILLAR_ORDER.YES,
  };

  const normalizedA = (valueA || '').toLowerCase().trim();
  const normalizedB = (valueB || '').toLowerCase().trim();

  // Handle null, undefined, empty string, or '-' as "no data" - should appear FIRST
  const isEmptyA = isEmptyValue(valueA);
  const isEmptyB = isEmptyValue(valueB);

  if (isEmptyA && isEmptyB) return 0; // Both empty, consider equal
  if (isEmptyA) return -1; // A is empty, A comes first
  if (isEmptyB) return 1; // B is empty, B comes first

  const orderA = pillarOrder[normalizedA] !== undefined ? pillarOrder[normalizedA] : RATRIA_PILLAR_ORDER.OTHER_LOCATIONS;
  const orderB = pillarOrder[normalizedB] !== undefined ? pillarOrder[normalizedB] : RATRIA_PILLAR_ORDER.OTHER_LOCATIONS;

  // If both have defined order, compare by order
  if (orderA !== orderB) {
    return orderA - orderB;
  }

  // If both are "other locations" (order 250), sort alphabetically
  return normalizedA.localeCompare(normalizedB);
};


  /**
   * Get the earliest created date from travel plan, travel info, and goodies
   */
  export function getEarliestAddedDate(travelPlan: any, travelInfo: any, goodies: any): string {
    const dates = [
      travelPlan?.createdAt,
      travelInfo?.createdAt,
      goodies?.createdAt
    ].filter(date => date && date !== null);

    if (dates.length === 0) return '';
    
    const earliestDate = new Date(Math.min(...dates.map(date => new Date(date).getTime())));
    return formatDateTimeIST(earliestDate);
  }

  /**
   * Get the latest updated date from travel plan, travel info, and goodies
   */
  export function getLatestModifiedDate(travelPlan: any, travelInfo: any, goodies: any): string {
    const dates = [
      travelPlan?.updatedAt,
      travelInfo?.updatedAt,
      goodies?.updatedAt
    ].filter(date => date && date !== null);

    if (dates.length === 0) return '';
    
    const latestDate = new Date(Math.max(...dates.map(date => new Date(date).getTime())));
    return formatDateTimeIST(latestDate);
  }


  export const formatGoodiesStatus = (goodiesRecord: any, allocatedProgramId: number | null, approvalStatus: string | null): string | null => {
      // If not approved and allocated, show dash
      if (!allocatedProgramId || approvalStatus !== ApprovalStatusEnum.APPROVED) {
        return null;
      }
  
      // If no goodies record exists, it's pending
      if (!goodiesRecord) {
        return STATUS_MESSAGES.PENDING;
      }

      if (goodiesRecord) {
        return STATUS_MESSAGES.COMPLETED;
      }
  
      return STATUS_MESSAGES.PENDING;
    }

  // Handle travel status sorting - show status based on travel plan existence and approval
  export const getTravelOverAllStatus = (registration: any) => {
    const approvalStatus = registration?.approvals?.[0]?.approvalStatus;
    const travelPlanStatus = registration?.travelPlans?.[0]?.travelPlanStatus;
    const travelInfoStatus = registration?.travelInfo?.[0]?.travelInfoStatus;
    
    // Check if travel plan exists (has any travel plan data)
    const hasTravelPlan = registration?.travelPlans && registration.travelPlans.length > 0;
    
    // If there's a travel plan, show status regardless of approval status
    if (hasTravelPlan) {
      // If both travel plan and travel info are completed, consider as completed
      if (travelPlanStatus === TravelStatusEnum.COMPLETED && travelInfoStatus === TravelStatusEnum.COMPLETED) {
        return toTitleCase(TravelStatusEnum.COMPLETED);
      }
      
      // Otherwise, consider as pending
      return toTitleCase(TravelStatusEnum.PENDING);
    }
    
    // If no travel plan exists, only show pending if approval status is approved
    if (approvalStatus === ApprovalStatusEnum.APPROVED) {
      return toTitleCase(TravelStatusEnum.PENDING);
    }
    
    // No travel plan and not approved - return null
    return null;
  };

  /**
 * Replace placeholders in a string with provided values
 * Replaces {0}, {1}, {2}, etc. with corresponding array values
 * @param template - String template with placeholders like {0}, {1}, etc.
 * @param values - Array of values to replace placeholders with
 * @returns String with placeholders replaced by values
 * @example
 * replaceStringPlaceholders("Hello {0}, you are {1} years old", ["John", "25"])
 * // Returns: "Hello John, you are 25 years old"
 */
export const replaceStringPlaceholders = (template: string, values: (string | number)[]): string => {
  if (!template) return '';
  if (!values || values.length === 0) return template;

  let result = template;
  values.forEach((value, index) => {
    const placeholder = `{${index}}`;
    // Use split and join
    result = result.split(placeholder).join(String(value));
  });
  
  return result;
};

/**
 * Helper function to check if a URL should be excluded from signed URL generation
 * @param url - The URL to check
 * @returns boolean indicating if the URL should be excluded
 */
export function shouldExcludeFromSignedUrl(url: string): boolean {
  if (!url) return true;
  
  const lowerUrl = url.toLowerCase();

  // Extract domain from URL (e.g., 'https://example.com/path' -> 'example.com')
  try {
    const urlObj = new URL(lowerUrl);
    const domain = urlObj.hostname; // Gets 'example.com' from 'https://example.com/path'
    
    // Check if any excluded domain is present in the URL's hostname
    return SIGNED_URL_EXCLUDED_DOMAINS.some(excludedDomain => 
      domain.includes(excludedDomain.toLowerCase())
    );
  } catch (error) {
    // If URL parsing fails, fallback to simple string matching
    return SIGNED_URL_EXCLUDED_DOMAINS.some(domain => lowerUrl.includes(domain.toLowerCase()));
  }
}

/**
 * Formats swap status enum values into human-readable strings
 * @param swapStatus - The swap status enum value
 * @returns Formatted swap status string
 */
export function formatSwapStatus(swapStatus: string | undefined): string {
  if (!swapStatus) return '';
  
  switch (swapStatus.toLowerCase()) {
    case 'active':
      return 'Active';
    case 'accepted':
      return 'Blessed';
    case 'on_hold':
      return 'Swap Demand';
    case 'rejected':
      return 'On Hold';
    case 'closed':
      return 'Cancelled';
    default:
      return toTitleCase(replaceUnderscoreWithSpace(swapStatus));
  }
}

/**
 * Get the appropriate status date/time for a registration based on its current status
 * @param registration - The registration object
 * @param approvalRecord - Pre-computed approval record
 * @param latestSwapRequest - Pre-computed latest swap request
 * @returns Date based on current status
 */
export function getRegistrationStatusDateTime(
  registration: any, 
  approvalRecord?: any, 
  latestSwapRequest?: any,
  kpiFilter?: string,
  kpiCategory?: string,
  isSeatAllocation?: boolean,
): Date | null {

  if ((kpiCategory === KPI_CATEGORIES.ALL && kpiFilter === KPI_FILTERS.ALL) || (isSeatAllocation && kpiCategory === KPI_CATEGORIES.ALL && kpiFilter != KPI_FILTERS.SWAP_REQUEST_HYPEN)) {
    return registration.registrationDate;
  }
  // Check registration status first
  if (registration.registrationStatus === RegistrationStatusEnum.CANCELLED) {
    return registration.cancellationDate;
  }
  
  // Check approval status
  if (approvalRecord?.approvalStatus === ApprovalStatusEnum.APPROVED) {
      // Check for active swap requests
      const shouldUseBlessedDateSort = Boolean(
        (kpiCategory === KPI_CATEGORIES.REGISTRATIONS && kpiFilter === KPI_FILTERS.APPROVED_SEEKERS) ||
        (kpiCategory === KPI_CATEGORIES.PROGRAMS && typeof kpiFilter === 'string' && kpiFilter.startsWith('program_')) ||
        (kpiCategory === KPI_CATEGORIES.BLESSED && (kpiFilter === KPI_FILTERS.APPROVED_SEEKERS || (typeof kpiFilter === 'string' && kpiFilter.startsWith('program_')))) ||
        (typeof kpiCategory === 'string' && kpiCategory.startsWith('program_')) ||
        (kpiCategory === KPI_CATEGORIES.ALLOCATED && kpiFilter === KPI_FILTERS.BLESSED) ||
        (kpiCategory === KPI_CATEGORIES.ALLOCATED && typeof kpiFilter === 'string' && kpiFilter.startsWith('program_'))
      );
      if (shouldUseBlessedDateSort || (isSeatAllocation && kpiFilter === undefined && (kpiCategory === KPI_CATEGORIES.HDB || kpiCategory === KPI_CATEGORIES.BLESSED))) {
        return approvalRecord.approvalDate; // Blessed date
      }

      if (latestSwapRequest && latestSwapRequest?.status === SwapRequestStatus.ACTIVE && latestSwapRequest?.type === SwapType.WantsSwap) {
        return latestSwapRequest.createdAt; // Swap request date
      }
    return approvalRecord.approvalDate; // Blessed date
  } else if (approvalRecord?.approvalStatus === ApprovalStatusEnum.REJECTED || approvalRecord?.approvalStatus === ApprovalStatusEnum.ON_HOLD) {
    return approvalRecord.updatedAt; // Hold date/Swap demand date - use updatedAt
  }
  
  // Default to registration date
  return registration.registrationDate;
}

  /**
   * Get a descriptive label for the status date time
   * @param registration - The registration object
   * @param approvalRecord - Pre-computed approval record
   * @param latestSwapRequest - Pre-computed latest swap request
   * @returns Label describing what the date represents
   */
  export const getStatusDateTimeLabel = (registration: any, approvalRecord?: any, latestSwapRequest?: any, kpiFilter?: string, kpiCategory?: string, isSeatAllocation?: boolean): string => {
    // Check registration status first
    if ((kpiCategory === KPI_CATEGORIES.ALL && kpiFilter === KPI_FILTERS.ALL) || (isSeatAllocation && kpiCategory === KPI_CATEGORIES.ALL && kpiFilter != KPI_FILTERS.SWAP_REQUEST_HYPEN)) {
      return 'Reg. on';
    }
    if (registration.registrationStatus === RegistrationStatusEnum.CANCELLED) {
      return 'Cancelled on';
    }

    // Check approval status
    if (approvalRecord?.approvalStatus === ApprovalStatusEnum.APPROVED) {
        // If KPI category is blessed, always return 'Blessed on'
      const shouldUseBlessedDateSort = Boolean(
        (kpiCategory === KPI_CATEGORIES.REGISTRATIONS && kpiFilter === KPI_FILTERS.APPROVED_SEEKERS) ||
        (kpiCategory === KPI_CATEGORIES.PROGRAMS && typeof kpiFilter === 'string' && kpiFilter.startsWith('program_')) ||
        (kpiCategory === KPI_CATEGORIES.BLESSED && (kpiFilter === KPI_FILTERS.APPROVED_SEEKERS || (typeof kpiFilter === 'string' && kpiFilter.startsWith('program_')))) ||
        (typeof kpiCategory === 'string' && kpiCategory.startsWith('program_')) ||
        (kpiCategory === KPI_CATEGORIES.ALLOCATED && kpiFilter === KPI_FILTERS.BLESSED) ||
        (kpiCategory === KPI_CATEGORIES.ALLOCATED && typeof kpiFilter === 'string' && kpiFilter.startsWith('program_'))
      );
      if (shouldUseBlessedDateSort || (isSeatAllocation && kpiFilter === undefined && (kpiCategory === KPI_CATEGORIES.HDB || kpiCategory === KPI_CATEGORIES.BLESSED))) {
        return 'Blessed on';
      }
        // Check for active swap requests
        if (latestSwapRequest && latestSwapRequest?.status === SwapRequestStatus.ACTIVE && latestSwapRequest?.type === SwapType.WantsSwap) {
          return 'Swap request on';
        }
      return 'Blessed on';
    } else if (approvalRecord?.approvalStatus === ApprovalStatusEnum.REJECTED) {
      return 'Hold on';
    } else if (approvalRecord?.approvalStatus === ApprovalStatusEnum.ON_HOLD) {
      return 'Swap demand on';
    }
    
    // Default label
    return 'Reg. on';
  }