import { Injectable, LoggerService } from '@nestjs/common';
import * as winston from 'winston';
import * as WinstonCloudWatch from 'winston-cloudwatch'; // Import as a namespace
import { logConfig, winstonCloudWatchTransportConfig } from '../config/config';
import { ConfigService } from '@nestjs/config';
import { APP_LOGGER } from '../constants/strings-constants';
import { AsyncLocalStorage } from 'async_hooks';

// Create a context storage to hold user information
export const loggerContextStorage = new AsyncLocalStorage<Map<string, any>>();

// Interface for log context
export interface LogContext {
  userId?: string;
  requestId?: string;
  [key: string]: any;
}

@Injectable()
export class AppLoggerService implements LoggerService {
  private readonly logger: winston.Logger;
  private static instance: AppLoggerService;

  constructor(private readonly configService: ConfigService) {
    if (!AppLoggerService.instance) {
      AppLoggerService.instance = this;
    }
    const cloudWatchLog = logConfig(this.configService);
    const logInCloudWatch = cloudWatchLog?.log || false;

    const transports: winston.transport[] = [
      new winston.transports.Console({
        format: winston.format.combine(
          winston.format.timestamp(),
          winston.format.colorize(),
          winston.format.printf(({ timestamp, level, message }) => {
            return `${timestamp} [${level}]: ${message}`;
          }),
        ),
      }),
    ];

    if (logInCloudWatch) {
      transports.push(
        new (WinstonCloudWatch as any)(winstonCloudWatchTransportConfig(this.configService)), // Use as a constructor
      );
    }

    this.logger = winston.createLogger({
      level: logInCloudWatch ? 'info' : 'debug',
      format: winston.format.json(),
      transports,
    });
  }
  
  static getInstance(): AppLoggerService {
    if (!AppLoggerService.instance) {
      throw new Error(APP_LOGGER.NOT_INITIALIZED);
    }
    return AppLoggerService.instance;
  }

  // Helper to get current context
  private getContext(): LogContext | undefined {
    const store = loggerContextStorage.getStore();
    return store ? Object.fromEntries(store) : undefined;
  }

  // Format the log message with context
  private formatLogEntry(message: string | object): any {
    const formattedMessage = typeof message === 'object' ? JSON.stringify(message) : message;
    const context = this.getContext();
    const requestIdInfo = context?.requestId ? `ReqID: ${context.requestId} | ` : '';
    const userInfo = `Req from userID: ${context?.userId ?? 'anonymous'}`;
    const methodPathInfo =
      context?.method && context?.originalUrl ? `${context.method} ${context.originalUrl} | ` : '';

    const messageWithContext = `${requestIdInfo}${userInfo} | ${methodPathInfo}${formattedMessage}`;
    return messageWithContext;
  }

  log(message: string | object, additionalInfo?: object) {
    const formattedMessage = typeof message === 'object' ? JSON.stringify(message) : message;
    const logEntry = additionalInfo
        ? `${formattedMessage} | Additional Info: ${JSON.stringify(additionalInfo)}`
        : formattedMessage;
    const formattedLogEntry = this.formatLogEntry(logEntry);
    this.logger.info(formattedLogEntry);
}

error(message: string | object, trace?: string, additionalInfo?: object) {
    const formattedMessage = typeof message === 'object' ? JSON.stringify(message) : message;
    const logEntry = additionalInfo
        ? `${formattedMessage} | Trace: ${trace} | Additional Info: ${JSON.stringify(additionalInfo)}`
        : `${formattedMessage} | Trace: ${trace}`;
    const formattedLogEntry = this.formatLogEntry(logEntry);
    this.logger.error(formattedLogEntry);
}

warn(message: string | object, additionalInfo?: object) {
    const formattedMessage = typeof message === 'object' ? JSON.stringify(message) : message;
    const logEntry = additionalInfo
        ? `${formattedMessage} | Additional Info: ${JSON.stringify(additionalInfo)}`
        : formattedMessage;
    const formattedLogEntry = this.formatLogEntry(logEntry);
    this.logger.warn(formattedLogEntry);
}

debug(message: string | object, additionalInfo?: object) {
    const formattedMessage = typeof message === 'object' ? JSON.stringify(message) : message;
    const logEntry = additionalInfo
        ? `${formattedMessage} | Additional Info: ${JSON.stringify(additionalInfo)}`
        : formattedMessage;
    const formattedLogEntry = this.formatLogEntry(logEntry);
    this.logger.debug(formattedLogEntry);
}

verbose(message: string | object, additionalInfo?: object) {
    const formattedMessage = typeof message === 'object' ? JSON.stringify(message) : message;
    const logEntry = additionalInfo
        ? `${formattedMessage} | Additional Info: ${JSON.stringify(additionalInfo)}`
        : formattedMessage;
    const formattedLogEntry = this.formatLogEntry(logEntry);
    this.logger.verbose(formattedLogEntry);
}
// Set context for current async execution
  setContext(context: LogContext) {
    const store = loggerContextStorage.getStore() || new Map();
    Object.entries(context).forEach(([key, value]) => {
      store.set(key, value);
    });
    return store;
  }

  // Run function with logger context
  runWithContext<T>(context: LogContext, fn: () => T): T {
    const contextMap = new Map(Object.entries(context));
    return loggerContextStorage.run(contextMap, fn);
  }
}